// SPDX-FileCopyrightText: 2025 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#ifndef DSEARCH_GLOBAL_H
#define DSEARCH_GLOBAL_H

#include <QObject>

#define DFMSEARCH dfmsearch
#define DFM_SEARCH_BEGIN_NS namespace DFMSEARCH {
#define DFM_SEARCH_END_NS }
#define DFM_SEARCH_USE_NS using namespace DFMSEARCH;

DFM_SEARCH_BEGIN_NS
Q_NAMESPACE

namespace Global {
inline constexpr int kMinContentSearchKeywordLength = 2;

/**
 * @brief Check if the given file extension is supported for full-text search.
 * This function checks if the specified file extension is included in the list of supported extensions
 * for performing full-text searches within the application.
 * @param suffix The file extension to check.
 * @return True if the extension is supported, false otherwise.
 */
bool isSupportedContentSearchExtension(const QString &suffix);

/**
 * @brief Get a list of default file extensions that are supported for full-text search.
 * This function returns a list of file extensions that the application can process for full-text search,
 * ensuring that users can search through various document types.
 * @return A QStringList containing the supported file extensions.
 */
QStringList defaultContentSearchExtensions();

/**
 * @brief Get the default directory path for indexed files.
 * This function retrieves the default directory where indexed files are stored,
 * allowing the application to access and manage the indexed content efficiently.
 * @return A QString representing the path to the default indexed directory.
 */
QStringList defaultIndexedDirectory();

/**
 * @brief Check if the specified path is within the content index directory.
 * This function verifies whether a given file path is located within the designated content index directory,
 * which is important for ensuring that only relevant files are included in search operations.
 * @param path The file path to check.
 * @return True if the path is within the content index directory, false otherwise.
 */
bool isPathInContentIndexDirectory(const QString &path);

/**
 * @brief Check if the content index is available.
 * This function checks the status of the content index to determine if it is accessible and ready for search operations.
 * @return True if the content index is available, false otherwise.
 */
bool isContentIndexAvailable();

/**
 * @brief Get the content index directory path.
 * This function provides the path to the directory where the content index is stored,
 * which is essential for performing searches on indexed content.
 * @return The path to the content index directory.
 */
QString contentIndexDirectory();

/**
 * @brief Check if the specified path is within the filename index directory.
 * This function verifies whether a given file path is located within the designated filename index directory,
 * ensuring that only relevant files are included in filename search operations.
 * @param path The file path to check.
 * @return True if the path is within the filename index directory, false otherwise.
 */
bool isPathInFileNameIndexDirectory(const QString &path);

/**
 * @brief Check if the filename index directory is available.
 * This function checks the status of the filename index directory to determine if it is accessible and ready for search operations.
 * @return True if the filename index directory is available, false otherwise.
 */
bool isFileNameIndexDirectoryAvailable();

/**
 * @brief Get the filename index directory path.
 * This function provides the path to the directory where the filename index is stored,
 * which is essential for performing searches on indexed filenames.
 * @return The path to the filename index directory.
 */
QString fileNameIndexDirectory();

}   // namespace Global

// Enumeration for different types of search methods
enum SearchType {
    FileName,   // Search by file name
    Content,   // Search by content within files
    Custom = 50   // User-defined search type
};
Q_ENUM_NS(SearchType)

// Enumeration for the status of the search operation
enum SearchStatus {
    Ready,   // The search engine is ready to perform a search
    Searching,   // The search operation is currently in progress
    Finished,   // The search operation has completed
    Cancelled,   // The search operation has been cancelled
    Error   // An error occurred during the search operation
    // Pause ?   // Optional: Indicates if the search can be paused
};
Q_ENUM_NS(SearchStatus)

// Enumeration for the method of searching
enum SearchMethod {
    Indexed,   // Search using pre-built indexes for faster results
    Realtime   // Search the file system in real-time for the most current results
};
Q_ENUM_NS(SearchMethod)

DFM_SEARCH_END_NS

#endif   // DSEARCH_GLOBAL_H
