#include <iostream>
#include <fstream>
#include <stdlib.h>
#include <stdio.h>
#include "../RIFF.h"
#include "../SF.h" // just for RIFF_ID() macro
#include "../helper.h"

#if !NO_MAIN || !defined(TEST_ASSERT)
static int iErrors = 0;
static int iChecks = 0;
#endif

#ifndef TEST_ASSERT
# define TEST_ASSERT(...) iChecks++; if (!(__VA_ARGS__)) ++iErrors
#endif

#ifndef TEST_VERIFY
# define TEST_VERIFY(...) do {                                               \
    if (!(__VA_ARGS__)) {                                                    \
        fprintf(stderr, "\n[ERROR] %s() failed:\n\n", __FUNCTION__);         \
        fprintf(stderr, "Violated expectation for this failure was:\n\n  "   \
                #__VA_ARGS__ "\n\n  [%s at line %d]\n\n",                    \
                lastPathComponent(__FILE__).c_str(), __LINE__);              \
    }                                                                        \
    TEST_ASSERT(__VA_ARGS__);                                                \
} while (false);
#endif

static bool fileExists(const char* path) {
    return std::ifstream(path).good();
}

static size_t fileSize(const char* path) {
    std::ifstream file(path, std::ios::binary | std::ios::ate);
    if (file.is_open())
        return file.tellg();
    return 0;
}

static void test_createEmpty() {
    RIFF::File file(RIFF_ID("abcd"));
    file.Save("a.riff");
    TEST_VERIFY(fileExists("a.riff"));
    TEST_VERIFY(fileSize("a.riff") == 3*4);
}

static void test_create24BytesRAM() {
    RIFF::File file(RIFF_ID("abcd"));
    RIFF::Chunk* ck = file.AddSubChunk(RIFF_ID("data"), 24);
    char* buf = (char*) ck->LoadChunkData();
    TEST_VERIFY(buf);
    for (int i = 0; i < 24; ++i)
        buf[i] = i;
    file.Save("b.riff");
    TEST_VERIFY(fileExists("b.riff"));
    TEST_VERIFY(fileSize("b.riff") == 3*4 + 2*4 + 24);
}

static void test_load24BytesRAM() {
    TEST_VERIFY(fileExists("b.riff"));
    RIFF::File file("b.riff");
    RIFF::Chunk* ck = file.GetSubChunk(RIFF_ID("data"));
    TEST_VERIFY(ck);
    char* buf = (char*) ck->LoadChunkData();
    TEST_VERIFY(buf);
    for (int i = 0; i < 24; ++i)
        TEST_VERIFY(buf[i] == i);
}

static void test_create24BytesIO() {
    RIFF::File file(RIFF_ID("abcd"));
    RIFF::Chunk* ck = file.AddSubChunk(RIFF_ID("data"), 24);
    file.Save("c.riff");
    TEST_VERIFY(fileExists("c.riff"));
    char buf[24];
    for (int i = 0; i < 24; ++i)
        buf[i] = i;
    RIFF::file_offset_t written = ck->Write(buf, 24, 1);
    TEST_VERIFY(written == 24);
    TEST_VERIFY(fileExists("c.riff"));
    TEST_VERIFY(fileSize("c.riff") == 3*4 + 2*4 + 24);
}

static void test_load24BytesIO() {
    TEST_VERIFY(fileExists("c.riff"));
    RIFF::File file("c.riff");
    RIFF::Chunk* ck = file.GetSubChunk(RIFF_ID("data"));
    TEST_VERIFY(ck);
    char* buf = (char*) ck->LoadChunkData();
    TEST_VERIFY(buf);
    for (int i = 0; i < 24; ++i)
        TEST_VERIFY(buf[i] == i);
}

static void test_shrink24To12Bytes() {
    RIFF::File file(RIFF_ID("abcd"));
    RIFF::Chunk* ck = file.AddSubChunk(RIFF_ID("data"), 24);
    file.Save("d.riff");
    TEST_VERIFY(fileExists("d.riff"));
    char buf[24];
    for (int i = 0; i < 24; ++i)
        buf[i] = i;
    RIFF::file_offset_t written = ck->Write(buf, 24, 1);
    TEST_VERIFY(written == 24);
    TEST_VERIFY(fileExists("d.riff"));
    TEST_VERIFY(fileSize("d.riff") == 3*4 + 2*4 + 24);

    ck->Resize(12);
    file.Save();
    TEST_VERIFY(fileExists("d.riff"));
    TEST_VERIFY(fileSize("d.riff") == 3*4 + 2*4 + 12);
}

static void test_loadShrunk24To12Bytes() {
    TEST_VERIFY(fileExists("d.riff"));
    RIFF::File file("d.riff");
    RIFF::Chunk* ck = file.GetSubChunk(RIFF_ID("data"));
    TEST_VERIFY(ck);
    char* buf = (char*) ck->LoadChunkData();
    TEST_VERIFY(buf);
    for (int i = 0; i < 12; ++i)
        TEST_VERIFY(buf[i] == i);
}

#if !NO_MAIN

int main() {
    printf("\n");

    int iExceptions = 0;
    try {
        test_createEmpty();
        test_create24BytesRAM();
        test_load24BytesRAM();
        test_create24BytesIO();
        test_load24BytesIO();
        test_shrink24To12Bytes();
        test_loadShrunk24To12Bytes();
    } catch (RIFF::Exception e) {
        std::cerr << "\nRIFF Exception:\n" << std::flush;
        e.PrintMessage();
        iExceptions++;
    } catch (...) {
        std::cerr << "\nUnknown Exception\n" << std::flush;
        iExceptions++;
    }

    if (iErrors || iExceptions)
        printf("\n!!! FAILED !!!  [ There were %d errors, %d exceptions ]\n\n",
               iErrors, iExceptions);
    else
        printf("\nOK (%d checks)\n\n", iChecks);

    return iErrors + iExceptions;
}

#endif // !NO_MAIN
