/*
 * Copyright (C) 2020 Sony Interactive Entertainment Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "MediaStrategy.h"

#include "AudioVideoRenderer.h"
#if USE(AVFOUNDATION)
#include "AudioVideoRendererAVFObjC.h"
#endif
#include "MediaPlayer.h"
#include <wtf/TZoneMallocInlines.h>

#if ENABLE(MEDIA_SOURCE)
#include "DeprecatedGlobalSettings.h"
#include "MockMediaPlayerMediaSource.h"
#endif

namespace WebCore {

WTF_MAKE_TZONE_ALLOCATED_IMPL(MediaStrategy);

MediaStrategy::MediaStrategy() = default;

MediaStrategy::~MediaStrategy() = default;

std::unique_ptr<NowPlayingManager> MediaStrategy::createNowPlayingManager() const
{
    return makeUnique<NowPlayingManager>();
}

void MediaStrategy::resetMediaEngines()
{
#if ENABLE(VIDEO)
    MediaPlayer::resetMediaEngines();
#endif
    m_mockMediaSourceEnabled = false;
}

bool MediaStrategy::hasThreadSafeMediaSourceSupport() const
{
    return false;
}

#if ENABLE(MEDIA_SOURCE)
void MediaStrategy::enableMockMediaSource()
{
#if USE(AVFOUNDATION)
    WebCore::DeprecatedGlobalSettings::setAVFoundationEnabled(false);
#endif
#if USE(GSTREAMER)
    WebCore::DeprecatedGlobalSettings::setGStreamerEnabled(false);
#endif
    addMockMediaSourceEngine();
}

bool MediaStrategy::mockMediaSourceEnabled() const
{
    return m_mockMediaSourceEnabled;
}

void MediaStrategy::addMockMediaSourceEngine()
{
    MediaPlayerFactorySupport::callRegisterMediaEngine(MockMediaPlayerMediaSource::registerMediaEngine);
}
#endif

#if ENABLE(VIDEO)
RefPtr<AudioVideoRenderer> MediaStrategy::createAudioVideoRenderer(WTF::LoggerHelper* loggerHelper, HTMLMediaElementIdentifier, MediaPlayerIdentifier) const
{
#if USE(AVFOUNDATION)
    ASSERT(loggerHelper);
    return AudioVideoRendererAVFObjC::create(Ref { loggerHelper->logger() }, loggerHelper->logIdentifier());
#else
    UNUSED_PARAM(loggerHelper);
    return nullptr;
#endif
}

bool MediaStrategy::hasRemoteRendererFor(MediaPlayerMediaEngineIdentifier type) const
{
    return m_remoteRenderersEnabled.get(static_cast<uint16_t>(type));
}

void MediaStrategy::enableRemoteRenderer(MediaPlayerMediaEngineIdentifier type, bool enabled)
{
    m_remoteRenderersEnabled.set(static_cast<uint16_t>(type), enabled);
}
#endif

#if ENABLE(WIRELESS_PLAYBACK_MEDIA_PLAYER)
void MediaStrategy::setWirelessPlaybackMediaPlayerEnabled(bool enabled)
{
    if (m_wirelessPlaybackMediaPlayerEnabled == enabled)
        return;

    m_wirelessPlaybackMediaPlayerEnabled = enabled;
#if ENABLE(VIDEO)
    MediaPlayer::resetMediaEngines();
#endif
}

bool MediaStrategy::wirelessPlaybackMediaPlayerEnabled() const
{
    return m_wirelessPlaybackMediaPlayerEnabled;
}
#endif

} // namespace WebCore
