# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from threading import Thread
from ConfigParser import ConfigParser, NoSectionError
from time import time
from os.path import join
from thread import start_new_thread
from time import sleep

from common.config import ConfigLoader

from plugins.audioscrobbler.audioscrobbler import AudioScrobbler
from plugins.audioscrobbler.config import Config

class Audioscrobbler:
    def __init__(self, extensionsloader):
        self.extensions = extensionsloader
        self.plugin = {'name': 'Audioscrobbler',
                       'version': 0.1,
                       'logo': 'plugins/audioscrobbler/audioscrobbler.png',
                       'configurable': True,
                       'authors': 'Erwan Briand <erwan@codingteam.net>',
                       'license': 'GNU General Public License 3',
                       'description': _('Send song notifications')}

    def start_plugin(self):
        # Start configuration
        name = self.plugin['name'].lower() + '.cfg'
        self.config = ConfigLoader()
        self.config_file = join(self.config.confdir, 'plugins', name)

        self.config_gui = Config(self.plugin, self.config_file,
                                 self.get_configuration)

        # Start Last.fm scobbler agent
        conf = self.get_configuration()
        self.song_to_scrobble = None
        self.scrob = audioscrobbler.AudioScrobbler()

        self.scrob.audioscrobbler_post_host = conf['server']
        self.scrob_post = self.scrob.post(conf['username'], conf['password'])

        def authenticate():
            print '[Audioscrobbler] Trying to authenticate account…'
            try:
                self.scrob_post.auth()
                print '[Audioscrobbler] Ok, session opened.'
            except Exception, error:
                print '[Audioscrobbler] Error authenticating:\n%s' % str(error)
                self.scrob_post = None

        thread = Thread(group=None, target=authenticate,
                        name='scrobblerauth', args=tuple())
        thread.start()

        def loadconfig(args):
            self.config_gui.configuration(args)

        def saveconfig(args):
            self.config_gui.save_config(args)

        def handler_play_new_song(song=None):
            """Scrobble the previous song."""
            # Get the timestamp
            timestamp = int(str(time()).split('.')[0])

            if self.song_to_scrobble is not None:
                # Don't send if the song length is inferior to 30 secs
                real_duration = int(timestamp - self.song_to_scrobble[1])
                song_length = self.song_to_scrobble[0][7]

                if real_duration > (song_length / 2) and real_duration > 30:
                    # Check if user is authentified
                    if self.scrob_post is not None:
                        title = self.song_to_scrobble[0][0]
                        artist =self.song_to_scrobble[0][1]
                        album = self.song_to_scrobble[0][2]
                        track = str(self.song_to_scrobble[0][7])
                        start = self.song_to_scrobble[1]

                        # Send the song to Last.fm
                        self.scrob_post.addtrack(artist, title, song_length,
                                                 start, track, album)
                        start_new_thread(self.scrob_post.post, tuple())

                        # Wait one second if we have to send a now-playing
                        if song is not None:
                            sleep(1)

            # Add the new song to the queue
            if song is not None:
                # If user authentified, we send the now-playing notification
                if self.scrob_post is not None:
                    title = song[0]
                    artist = song[1]
                    album = song[2]
                    track = str(song[6])
                    length = str(song[7])
                    start = str(time()).split('.')[0]

                    start_new_thread(self.scrob_post.nowplaying, (artist,
                                                                  title,
                                                                  length,
                                                                  track,
                                                                  album,
                                                                  start))

                # Save the song to be scrobbled in the future
                self.song_to_scrobble = [song, timestamp]
            else:
                self.song_to_scrobble = None

        # Connect to Bluemindo's signals
        self.extensions.connect('OnPlayNewSong', handler_play_new_song)
        self.extensions.connect('OnStopPressed', handler_play_new_song)
        self.extensions.connect('OnBluemindoQuitted', handler_play_new_song)
        self.extensions.connect('OnModuleConfiguration', loadconfig)
        self.extensions.connect('OnModuleConfigurationSave', saveconfig)

    def stop_plugin(self):
        self.scrob_post = None

    def get_configuration(self):
        configparser = ConfigParser()
        configparser.read(self.config_file)
        config = {}

        try:
            for item in configparser.items(self.plugin['name']):
                try:
                    value = int(item[1])
                except ValueError:
                    value = str(item[1])

                config[item[0]] = value
        except NoSectionError:
            config['username'] = ''
            config['password'] = ''
            config['server'] = 'post.audioscrobbler.com'

        return config