//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Device/Beam/Beam.cpp
//! @brief     Implements class Beam.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Device/Beam/Beam.h"
#include <numbers>
using std::numbers::pi;
#include "Base/Spin/SpinMatrix.h"
#include "Base/Util/Assert.h"
#include "Base/Vector/GisasDirection.h"
#include "Device/Beam/IFootprint.h"

//... Constructors, destructors:

Beam::Beam(std::vector<double> P)
    : INode(P)
    , m_intensity(m_P[0])
    , m_wavelength(m_P[1])
    , m_alpha(m_P[2])
    , m_phi(m_P[3])
{
    ASSERT(m_intensity > 0);
    ASSERT(m_alpha >= 0);
    precompute();
}

Beam::Beam(double intensity, double wavelength, double alpha, double phi)
    : Beam(std::vector<double>{intensity, wavelength, alpha, phi})
{
}

Beam::~Beam() = default;

Beam* Beam::clone() const
{
    Beam* result = new Beam(m_P);
    result->m_shape_factor = m_shape_factor;
    result->m_polarization = m_polarization;
    return result;
}

//... Getters:

std::vector<const INode*> Beam::nodeChildren() const
{
    if (m_shape_factor)
        return {m_shape_factor.get()};
    return {};
}

R3 Beam::ki() const
{
    return m_k;
}

//! Returns wave vector of beam after specular reflection by the horizontal sample.
R3 Beam::k_reflected() const
{
    return vecOfKAlphaPhi(m_wavenumber, m_alpha, -m_phi);
}

R3 Beam::polVector() const
{
    return m_polarization;
}

SpinMatrix Beam::polMatrix() const
{
    return SpinMatrix::FromBlochVector(m_polarization);
}

const IFootprint* Beam::footprint() const
{
    return m_shape_factor.get();
}

//... Setters:

void Beam::setIntensity(double intensity)
{
    m_intensity = intensity;
}

void Beam::setWavelength(double wavelength)
{
    m_wavelength = wavelength;
    precompute();
}

void Beam::setInclination(const double alpha)
{
    m_alpha = alpha;
    precompute();
}

void Beam::setAzimuthalAngle(double value)
{
    m_phi = value;
    precompute();
}

void Beam::setFootprint(const IFootprint* shape_factor)
{
    m_shape_factor.reset(shape_factor ? shape_factor->clone() : nullptr);
}

//! Sets the polarization density matrix according to the given Bloch vector
void Beam::setPolarization(const R3 polarization)
{
    if (polarization.mag() > 1.0)
        throw std::runtime_error(
            "Beam::setPolarization: "
            "The given Bloch vector cannot represent a real physical ensemble");
    m_polarization = polarization;
}

//... Private function:

//! To be called whenever wavelength or alpha or phi has changed.
void Beam::precompute()
{
    m_wavenumber = (2 * pi) / m_wavelength;
    m_k = vecOfKAlphaPhi(m_wavenumber, -m_alpha, -m_phi);
}
