//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Job/FitSuiteItem.cpp
//! @brief     Implements class FitSuiteItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Job/FitSuiteItem.h"
#include "GUI/Model/Job/FitParameterContainerItem.h"
#include "GUI/Model/Job/MinimizerItem.h"
#include "GUI/Support/XML/UtilXML.h"

namespace {
namespace Tag {

const QString UpdateInterval("UpdateInterval");
const QString IterationsCount("IterationsCount");
const QString Chi2("Chi2");

const QString MinimizerContainer("MinimizerContainer");
const QString FitParameterContainer("FitParameterContainer");

} // namespace Tag
} // namespace

FitSuiteItem::FitSuiteItem()
    : m_updateInterval(10)
    , m_iterCount(0)
    , m_chi2(0.0)
{
    setObjectName("FitSuite");
}

FitParameterContainerItem* FitSuiteItem::createFitParametersContainerItem()
{
    if (m_fitContainer)
        throw std::runtime_error(
            "FitSuiteItem::createFitParametersContainer -> Error. Attempt to create "
            "a second FitParameterContainer.");

    m_fitContainer = std::make_unique<FitParameterContainerItem>();
    return m_fitContainer.get();
}

FitParameterContainerItem* FitSuiteItem::fitParameterContainerItem()
{
    return m_fitContainer.get();
}

MinimizerContainerItem* FitSuiteItem::createMinimizerContainerItem()
{
    if (m_minimizerContainer)
        throw std::runtime_error(
            "FitSuiteItem::createMinimizerContainer -> Error. Attempt to create "
            "a second MinimizerContainer.");

    m_minimizerContainer = std::make_unique<MinimizerContainerItem>();
    return m_minimizerContainer.get();
}

MinimizerContainerItem* FitSuiteItem::minimizerContainerItem()
{
    return m_minimizerContainer.get();
}

int FitSuiteItem::updateInterval() const
{
    return m_updateInterval;
}

void FitSuiteItem::setUpdateInterval(int interval)
{
    m_updateInterval = interval;
    emit updateIntervalChanged(interval);
}

int FitSuiteItem::iterationCount() const
{
    return m_iterCount;
}

void FitSuiteItem::setIterationCount(int count)
{
    m_iterCount = count;
    emit iterationCountChanged(count);
}

double FitSuiteItem::chi2() const
{
    return m_chi2;
}

void FitSuiteItem::setChi2(const double chi2)
{
    m_chi2 = chi2;
}

void FitSuiteItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // update interval
    w->writeStartElement(Tag::UpdateInterval);
    XML::writeAttribute(w, XML::Attrib::value, m_updateInterval);
    w->writeEndElement();

    // iterations count
    w->writeStartElement(Tag::IterationsCount);
    XML::writeAttribute(w, XML::Attrib::value, m_iterCount);
    w->writeEndElement();

    // chi2
    w->writeStartElement(Tag::Chi2);
    XML::writeAttribute(w, XML::Attrib::value, m_chi2);
    w->writeEndElement();

    // minimizer container
    if (m_minimizerContainer) {
        w->writeStartElement(Tag::MinimizerContainer);
        m_minimizerContainer->writeTo(w);
        w->writeEndElement();
    }

    // fit parameter container
    if (m_fitContainer) {
        w->writeStartElement(Tag::FitParameterContainer);
        m_fitContainer->writeTo(w);
        w->writeEndElement();
    }
}

void FitSuiteItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // update interval
        if (tag == Tag::UpdateInterval) {
            XML::readAttribute(r, XML::Attrib::value, &m_updateInterval);
            XML::gotoEndElementOfTag(r, tag);

            // iterations count
        } else if (tag == Tag::IterationsCount) {
            XML::readAttribute(r, XML::Attrib::value, &m_iterCount);
            XML::gotoEndElementOfTag(r, tag);

            // chi2
        } else if (tag == Tag::Chi2) {
            XML::readAttribute(r, XML::Attrib::value, &m_chi2);
            XML::gotoEndElementOfTag(r, tag);

            // minimizer container
        } else if (tag == Tag::MinimizerContainer) {
            createMinimizerContainerItem()->readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // fit parameter container
        } else if (tag == Tag::FitParameterContainer) {
            createFitParametersContainerItem()->readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}
