//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/MaterialEditor/MaterialEditorDialog.h
//! @brief     Defines class MaterialEditorDialog
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_VIEW_MATERIALEDITOR_MATERIALEDITORDIALOG_H
#define BORNAGAIN_GUI_VIEW_MATERIALEDITOR_MATERIALEDITORDIALOG_H

#include "GUI/Model/Sample/MaterialModel.h"
#include <QDialog>
#include <memory>

class MaterialItem;
class MaterialEditorModel;
class SampleItem;

namespace Ui {
class MaterialEditorDialog;
}

//! Dialog to select a material and also to edit the list of existing materials.
//! The dialog operates on a copy of the current materials. The original material store is only
//! updated if changes have been made and the dialog has been closed with OK.
//! Use this dialog only with the two static methods to edit the list, or to choose a material
class MaterialEditorDialog : public QDialog {
    Q_OBJECT
public:
    //! Use this to choose a material. identifierOfPreviousMaterial is the material which should be
    //! selected when opening the dialog. Returns the identifier of the newly selected material.
    //! Returns an empty string, if the dialog is
    //! cancelled.
    static QString chooseMaterial(QWidget* parent, SampleItem* sample,
                                  const QString& identifierOfPreviousMaterial);

    //! Use this to edit the list of existing materials.
    static void editMaterials(QWidget* parent, SampleItem* sample);

private:
    MaterialEditorDialog(SampleItem* sample, QWidget* parent = nullptr);
    ~MaterialEditorDialog() override;

    //! updates original material model with the edited model
    void accept() override;

    void addRefractiveMaterial();
    void addSldMaterial();
    void cloneCurrentMaterial();
    void removeCurrentMaterial();

    void onSelectColor();
    void updateActionEnabling();
    void fill();
    void setCurrentMaterial(const MaterialItem* m);
    void setCurrentMaterial(const QString& identifier);
    QModelIndex currentIndex() const;
    MaterialItem* currentMaterialItem();

    //! Returns the list of material identifiers of the materials currently used in the sample.
    //! E.g. the material selected in a particle.
    QStringList identifiersOfUsedMaterials() const;

private:
    MaterialModel m_tmpMaterialModel; //! Materials for working within this dialog

    QAction* m_cloneMaterialAction;
    QAction* m_removeMaterialAction;

    Ui::MaterialEditorDialog* m_ui;

    MaterialEditorModel* m_model; //! Model for the left list. Works on m_tmpMaterialModel
    SampleItem* m_sample;
};

#endif // BORNAGAIN_GUI_VIEW_MATERIALEDITOR_MATERIALEDITORDIALOG_H
