//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Tests/Suite/Persist/Check.cpp
//! @brief     Implements function checkSimulation for core standard test
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "BABuild.h"
#include "BATesting.h"
#include "Base/Util/Assert.h"
#include "Base/Util/PathUtil.h"
#include "Device/Data/Datafield.h"
#include "Device/Histo/DiffUtil.h"
#include "Device/Histo/SimulationResult.h"
#include "Device/IO/IOFactory.h"
#include "Sim/Simulation/ISimulation.h"
#include <iostream>

//! Run simulation directly (in C++ core), and compare result with reference data.

bool checkSimulation(const std::string& name, ISimulation& direct_simulation, const double limit)
{
    // Run simulation directly.
    const SimulationResult result_data = direct_simulation.simulate();

    std::unique_ptr<Datafield> reference;

    // Load reference if available.
    ASSERT(!name.empty());
    try {
        const std::string refPath =
            Base::Path::jointPath(BATesting::ReferenceDir_Suite(), name + ".int.gz");
        std::cout << "- reference: " << refPath << "\n";
        reference.reset(IO::readData2D(refPath));
    } catch (const std::exception&) {
        std::cerr << "FAILED: cannot read reference\n";
    }

    // Compare with reference if available.
    if (reference) {
        std::cout << "- check diff" << std::endl;
        if (DiffUtil::checkRelativeDifference(result_data.flatVector(), reference->flatVector(),
                                              limit)) {
            std::cout << "- success" << std::endl;
            return true; // regular exit
        }
    } else {
        std::cerr << "FAILED: reference not found\n";
    }

    // Save simulation, as it differs from reference.
    Base::Path::createDirectories(BATesting::TestOutDir_Suite());
    std::string out_fname = Base::Path::jointPath(BATesting::TestOutDir_Suite(), name + ".int.gz");
    IO::writeDatafield(result_data, out_fname);
    std::cout << "Notes:\n- to visualize an intensity map, use " << BABuild::srcDir()
              << "/devtools/view/plot-int.py\n"
              << "- to plot a difference image, use " << BABuild::srcDir()
              << "/devtools/view/plot-diff.py\n"
              << "- if the new simulation result\n"
              << "    " << out_fname << "\n"
              << "  is correct, then move it to the reference directory\n"
              << "    " << BATesting::ReferenceDir_Suite() << "/\n\n";

    return false;
}
