//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Base/Axis/Frame.h
//! @brief     Defines and implements templated class Frame.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_BASE_AXIS_FRAME_H
#define BORNAGAIN_BASE_AXIS_FRAME_H

#include "Base/Types/ICloneable.h"
#include "Base/Types/OwningVector.h"

using std::size_t;

class Scale;

//! Holds one or two axes.

class Frame : public ICloneable {
public:
    //! Constructor that takes ownership of supplied axes.
    Frame(std::vector<const Scale*>&& axes);

    virtual ~Frame();

    Frame* clone() const;

    //! Returns number of dimensions.
    size_t rank() const;

    //! Returns total number of bins.
    size_t size() const;

    //! Returns number of bins along axis.
    size_t projectedSize(size_t k_axis) const;

    //! Returns axis with given serial number
    const Scale& axis(size_t k_axis) const;
    const Scale& xAxis() const;
    const Scale& yAxis() const;

    //! Returns the value of selected axis for given i_flat.
    //! @param i_flat The global index of this data structure.
    //! @param k_axis Serial number of selected axis.
    //! @return corresponding bin center of selected axis
    double projectedCoord(size_t i_flat, size_t k_axis) const;

    //! Returns vector of axes indices for given global index
    //! @param i_flat The global index of this data structure.
    //! @return Vector of bin indices for all axes defined
    std::vector<int> allIndices(size_t i_flat) const;

    //! Returns axis bin index for given global index
    //! @param i_flat The global index of this data structure.
    //! @param k_axis Serial number of selected axis.
    //! @return Corresponding bin index for selected axis
    size_t projectedIndex(size_t i_flat, size_t k_axis) const;


    //! Returns global index for specified indices of axes
    //! @param axes_indices Vector of axes indices for all specified axes in this dataset
    //! @return Corresponding global index
    size_t toGlobalIndex(const std::vector<unsigned>& axes_indices) const;

    //! Returns true if both Frame%s have same rank, and all axes have same sizes.
    bool hasSameSizes(const Frame&) const;

    //! Returns true if both Frame%s have same rank, and all axes are equal.
    bool operator==(const Frame&) const;

protected:
    OwningVector<const Scale> m_axes;

private:
    size_t m_size; // cached product of axis sizes
};

#endif // BORNAGAIN_BASE_AXIS_FRAME_H
