//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Job/JobItem.h
//! @brief     Defines class JobItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_JOB_JOBITEM_H
#define BORNAGAIN_GUI_MODEL_JOB_JOBITEM_H

#include "GUI/Model/Descriptor/SelectionProperty.h"
#include "GUI/Model/Device/InstrumentItemCatalog.h"
#include "GUI/Support/Data/JobStatus.h"
#include "GUI/Support/Data/SimulationOptionsItem.h"
#include <QDateTime>
#include <optional>

class DataItem;
class FitParameterContainerItem;
class FitSuiteItem;
class InstrumentItem;
class IntensityDataItem;
class MessageService;
class ParameterContainerItem;
class ParameterTreeItems;
class RealItem;
class SampleItem;
class SimulationResult;

class JobItem : public QObject {
    Q_OBJECT

public:
    JobItem();
    ~JobItem();

    // job properties

    QString identifier() const { return m_identifier; }
    void setIdentifier(const QString& identifier) { m_identifier = identifier; }

    QString jobName() const;
    void setJobName(const QString& name);

    JobStatus status() const;
    void setStatus(JobStatus status);

    bool isIdle() const;
    bool isRunning() const;
    bool isCompleted() const;
    bool isCanceled() const;
    bool isFailed() const;
    bool isFitting() const;
    bool isValidForFitting();

    QDateTime beginTime() const;
    void setBeginTime(const QDateTime& begin_time);

    QDateTime endTime() const;
    void setEndTime(const QDateTime& end_time);

    //! if begin and end time are both available the duration in ms, otherwise empty
    std::optional<size_t> duration() const;

    QString comments() const;
    void setComments(const QString& comments);

    int progress() const;
    void setProgress(int progress);

    bool runImmediately() const;
    bool runInBackground() const;

    QString presentationType() const { return m_presentationType; }
    void setPresentationType(const QString& type) { m_presentationType = type; }

    QString activity() const { return m_activity; }
    void setActivity(const QString& activity) { m_activity = activity; }

    int sliderRange() const { return m_sliderRange; }
    void setSliderRange(int range) { m_sliderRange = range; }

    bool isSpecularJob() const;
    bool isIntensityJob() const;

    // sample

    SampleItem* sampleItem();
    void copySampleIntoJob(const SampleItem* sample);

    // instrument

    InstrumentItem* instrumentItem() const;
    void copyInstrumentIntoJob(const InstrumentItem* instrument);

    // simulation

    const SimulationOptionsItem& simulationOptionsItem() const;
    void copySimulationOptionsIntoJob(const SimulationOptionsItem& options);
    void setResults(const SimulationResult& result);

    // fitting

    FitSuiteItem* fitSuiteItem();
    FitSuiteItem* createFitSuiteItem();
    void createFitContainers();

    ParameterContainerItem* parameterContainerItem();
    FitParameterContainerItem* fitParameterContainerItem();

    // data

    void createSimulatedDataItem();
    IntensityDataItem* intensityDataItem();
    DataItem* simulatedDataItem();

    DataItem* createDiffDataItem();
    DataItem* diffDataItem();

    RealItem* createRealItem();
    void copyRealItemIntoJob(const RealItem* srcRealItem);
    RealItem* realItem();
    void adjustReaDataToJobInstrument();

    void updateDataFileName();

    // write/read

    void writeTo(QXmlStreamWriter* w) const;
    void readFrom(QXmlStreamReader* r);

    void writeDataFiles(const QString& projectDir) const;
    void readDataFiles(const QString& projectDir, MessageService* messageService);

signals:
    void jobNameChanged(const QString& name);
    void jobStatusChanged(JobStatus status);
    void jobBeginTimeChanged(const QDateTime& begin_time);
    void jobEndTimeChanged(const QDateTime& end_time);
    void jobCommentsChanged(const QString& comments);
    void jobProgressChanged(int progress);

private:
    DataItem* createNewDataItem();
    void cropRealData();
    void importMasksFromRealItem();

private:
    std::unique_ptr<SimulationOptionsItem> m_simulationOptionsItem;
    std::unique_ptr<ParameterContainerItem> m_parameterContainer;
    std::unique_ptr<SampleItem> m_sampleItem;
    SelectionProperty<InstrumentItemCatalog> m_instrument;
    QString m_name;
    QString m_identifier;
    JobStatus m_status = JobStatus::Idle;
    QString m_comments;
    QString m_activity;
    QString m_presentationType;
    int m_sliderRange = 100;
    QDateTime m_beginTime;
    QDateTime m_endTime;
    int m_progress;
    std::unique_ptr<DataItem> m_simulatedDataItem;
    std::unique_ptr<DataItem> m_diffDataItem;
    std::unique_ptr<RealItem> m_realItem;
    std::unique_ptr<FitSuiteItem> m_fitSuiteItem;
};

#endif // BORNAGAIN_GUI_MODEL_JOB_JOBITEM_H
