//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Job/JobListModel.cpp
//! @brief     Implements class JobListModel
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Job/JobListModel.h"
#include "GUI/Model/Project/ProjectDocument.h"

//==================================================================================================
// JobListModel
//==================================================================================================

//--------------------------------------------------------------------------------------------------
// public member functions
//--------------------------------------------------------------------------------------------------

JobListModel::JobListModel(JobModel* jobs, QObject* parent)
    : QAbstractListModel(parent)
    , m_jobs(jobs)
{
    for (JobItem* job : m_jobs->jobItems())
        enableJobNotification(job);
    connect(jobs, &JobModel::jobAdded, this, &JobListModel::onJobAdded, Qt::UniqueConnection);
}

JobListModel::~JobListModel()
{
    for (JobItem* job : m_jobs->jobItems())
        disableJobNotification(job);
}

int JobListModel::rowCount(const QModelIndex&) const
{
    return m_jobs->jobItems().size();
}

QVariant JobListModel::data(const QModelIndex& index, int role) const
{
    QVector<JobItem*> jobs = m_jobs->jobItems();
    if (!index.isValid() || index.row() >= jobs.size() || index.row() < 0)
        return {};

    JobItem* item = jobs[index.row()];
    if (role == Qt::DisplayRole)
        return item->jobName();

    return {};
}

JobItem* JobListModel::jobItemForIndex(const QModelIndex& index) const
{
    QVector<JobItem*> jobs = m_jobs->jobItems();
    if (index.row() >= 0 && index.row() < jobs.size())
        return jobs[index.row()];
    return nullptr;
}

QModelIndex JobListModel::indexForJob(JobItem* job)
{
    QVector<JobItem*> jobs = m_jobs->jobItems();
    int idx = jobs.indexOf(job);
    if (idx != -1)
        return index(idx, 0);
    return {};
}

void JobListModel::runJob(const QModelIndex& index)
{
    m_jobs->runJob(jobItemForIndex(index));
}

void JobListModel::removeJob(const QModelIndex& index)
{
    beginRemoveRows(QModelIndex(), index.row(), index.row());
    JobItem* job = jobItemForIndex(index);
    disableJobNotification(job);
    m_jobs->removeJob(job);
    endRemoveRows();
}

void JobListModel::cancelJob(const QModelIndex& index)
{
    m_jobs->cancelJob(jobItemForIndex(index));
}

void JobListModel::emitJobListModelChanged(JobItem* job)
{
    QVector<JobItem*> jobs = m_jobs->jobItems();
    int i = jobs.indexOf(job);
    if (i != -1) {
        QModelIndex idx = index(i, 0);
        emit dataChanged(idx, idx);
    }
    gProjectDocument.value()->setModified();
}

//--------------------------------------------------------------------------------------------------
// private slots
//--------------------------------------------------------------------------------------------------

void JobListModel::onJobAdded()
{
    for (JobItem* job : m_jobs->jobItems())
        enableJobNotification(job);
}

//--------------------------------------------------------------------------------------------------
// private member functions
//--------------------------------------------------------------------------------------------------

void JobListModel::enableJobNotification(JobItem* job)
{
    // name
    connect(job, &JobItem::jobNameChanged, this, [this, job]() { emitJobListModelChanged(job); });

    // status
    connect(job, &JobItem::jobStatusChanged, this, [this, job]() { emitJobListModelChanged(job); });

    // progress
    connect(job, &JobItem::jobProgressChanged, this,
            [this, job]() { emitJobListModelChanged(job); });
}

void JobListModel::disableJobNotification(JobItem* job)
{
    disconnect(job, nullptr, this, nullptr);
}
