//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Projection/ProjectionsEditorActions.cpp
//! @brief     Implements class ProjectionsEditorActions
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Projection/ProjectionsEditorActions.h"
#include "GUI/Model/Data/IntensityDataItem.h"
#include "GUI/Model/Data/ProjectionItems.h"
#include "GUI/Model/Project/ProjectDocument.h"
#include "GUI/View/Projection/SaveProjectionsAssistant.h"
#include <QAction>
#include <QItemSelectionModel>
#include <QModelIndexList>
#include <QWidget>

class MaskItem;

ProjectionsEditorActions::ProjectionsEditorActions(QWidget* parent)
    : QObject(parent)
    , m_resetViewAction(new QAction(this))
    , m_togglePanelAction(new QAction(this))
    , m_deleteAction(new QAction("Remove selected", this))
    , m_parent(parent)
{
    // Actions for top toolbar
    m_resetViewAction->setText("Center view");
    m_resetViewAction->setIcon(QIcon(":/images/camera-metering-center.svg"));
    m_resetViewAction->setToolTip("Reset view\n"
                                  "x,y,z axes range will be set to default");
    connect(m_resetViewAction, &QAction::triggered, this,
            &ProjectionsEditorActions::resetViewRequest, Qt::UniqueConnection);

    m_togglePanelAction->setText("Properties");
    m_togglePanelAction->setIcon(QIcon(":/images/dock-right.svg"));
    m_togglePanelAction->setToolTip("Toggle property panel");
    connect(m_togglePanelAction, &QAction::triggered, this,
            &ProjectionsEditorActions::propertyPanelRequest, Qt::UniqueConnection);

    m_deleteAction->setToolTip("Remove selected (Del)");
    m_deleteAction->setShortcuts(QKeySequence::Delete);
    connect(m_deleteAction, &QAction::triggered, this, &ProjectionsEditorActions::onDeleteAction);
}

void ProjectionsEditorActions::setContext(IntensityDataItem* intensityItem)
{
    m_intensityDataItem = intensityItem;

    ProjectionContainerItem* containerItem = intensityItem->getOrCreateProjectionContainerItem();
    ASSERT(containerItem);

    m_maskContainerModel = containerItem->model();
    m_selectionModel = containerItem->selectionModel();
}

QList<QAction*> ProjectionsEditorActions::topToolbarActions()
{
    return QList<QAction*>() << m_resetViewAction << m_togglePanelAction;
}

void ProjectionsEditorActions::onDeleteAction()
{
    ASSERT(m_maskContainerModel);
    ASSERT(m_selectionModel);

    QModelIndexList indexes = m_selectionModel->selectedIndexes();
    while (!indexes.empty()) {
        m_maskContainerModel->removeMaskAt(indexes.back().row());
        indexes = m_selectionModel->selectedIndexes();
    }
    gProjectDocument.value()->setModified();
}

//! Performs saving of projections in ascii file
void ProjectionsEditorActions::onSaveAction()
{
    if (!m_intensityDataItem)
        return;

    SaveProjectionsAssistant assistant;
    assistant.saveProjections(m_parent, m_intensityDataItem);
}
