/*
 * Data, and supporting functions, for PostScript-based output
 * formats.
 */

#include <assert.h>
#include "halibut.h"
#include "paper.h"

/*
 * Within the paper backends, PostScript glyph names are represented
 * by small integers.  For standard glyphs, these are indicies into
 * a table generated from the Adobe Glyph List from
 * <http://partners.adobe.com/public/developer/en/opentype/glyphlist.txt>.
 * There are a few glyphs that are needed statically but aren't in the
 * Glyph List.  We manually add them with a mapping of U+FFFF.
 * Since all the scripts that generate fragments of code for this file
 * need that list, it's worth generating a file containing just the names.

echo ".notdef;FFFF" >>glyphlist.txt
echo ".null;FFFF" >>glyphlist.txt
echo "nonmarkingreturn;FFFF" >>glyphlist.txt
grep '^[^#;][^;]*;[^ ][^ ][^ ][^ ]$' glyphlist.txt | sort -t\; -k1,1 | \
    cut -f1 -d\; > glyphnames.txt

< glyphnames.txt xargs printf '"%s", ' | fold -s -w68 | sed 's/^/    /'; echo

 */
static const char *const ps_glyphs_alphabetic[] = {
    ".notdef", ".null", "A", "AE", "AEacute", "AEmacron", "AEsmall", 
    "Aacute", "Aacutesmall", "Abreve", "Abreveacute", "Abrevecyrillic", 
    "Abrevedotbelow", "Abrevegrave", "Abrevehookabove", "Abrevetilde", 
    "Acaron", "Acircle", "Acircumflex", "Acircumflexacute", 
    "Acircumflexdotbelow", "Acircumflexgrave", "Acircumflexhookabove", 
    "Acircumflexsmall", "Acircumflextilde", "Acute", "Acutesmall", 
    "Acyrillic", "Adblgrave", "Adieresis", "Adieresiscyrillic", 
    "Adieresismacron", "Adieresissmall", "Adotbelow", "Adotmacron", 
    "Agrave", "Agravesmall", "Ahookabove", "Aiecyrillic", 
    "Ainvertedbreve", "Alpha", "Alphatonos", "Amacron", "Amonospace", 
    "Aogonek", "Aring", "Aringacute", "Aringbelow", "Aringsmall", 
    "Asmall", "Atilde", "Atildesmall", "Aybarmenian", "B", "Bcircle", 
    "Bdotaccent", "Bdotbelow", "Becyrillic", "Benarmenian", "Beta", 
    "Bhook", "Blinebelow", "Bmonospace", "Brevesmall", "Bsmall", 
    "Btopbar", "C", "Caarmenian", "Cacute", "Caron", "Caronsmall", 
    "Ccaron", "Ccedilla", "Ccedillaacute", "Ccedillasmall", "Ccircle", 
    "Ccircumflex", "Cdot", "Cdotaccent", "Cedillasmall", "Chaarmenian", 
    "Cheabkhasiancyrillic", "Checyrillic", 
    "Chedescenderabkhasiancyrillic", "Chedescendercyrillic", 
    "Chedieresiscyrillic", "Cheharmenian", "Chekhakassiancyrillic", 
    "Cheverticalstrokecyrillic", "Chi", "Chook", "Circumflexsmall", 
    "Cmonospace", "Coarmenian", "Csmall", "D", "DZ", "DZcaron", 
    "Daarmenian", "Dafrican", "Dcaron", "Dcedilla", "Dcircle", 
    "Dcircumflexbelow", "Dcroat", "Ddotaccent", "Ddotbelow", 
    "Decyrillic", "Deicoptic", "Delta", "Deltagreek", "Dhook", 
    "Dieresis", "DieresisAcute", "DieresisGrave", "Dieresissmall", 
    "Digammagreek", "Djecyrillic", "Dlinebelow", "Dmonospace", 
    "Dotaccentsmall", "Dslash", "Dsmall", "Dtopbar", "Dz", "Dzcaron", 
    "Dzeabkhasiancyrillic", "Dzecyrillic", "Dzhecyrillic", "E", 
    "Eacute", "Eacutesmall", "Ebreve", "Ecaron", "Ecedillabreve", 
    "Echarmenian", "Ecircle", "Ecircumflex", "Ecircumflexacute", 
    "Ecircumflexbelow", "Ecircumflexdotbelow", "Ecircumflexgrave", 
    "Ecircumflexhookabove", "Ecircumflexsmall", "Ecircumflextilde", 
    "Ecyrillic", "Edblgrave", "Edieresis", "Edieresissmall", "Edot", 
    "Edotaccent", "Edotbelow", "Efcyrillic", "Egrave", "Egravesmall", 
    "Eharmenian", "Ehookabove", "Eightroman", "Einvertedbreve", 
    "Eiotifiedcyrillic", "Elcyrillic", "Elevenroman", "Emacron", 
    "Emacronacute", "Emacrongrave", "Emcyrillic", "Emonospace", 
    "Encyrillic", "Endescendercyrillic", "Eng", "Enghecyrillic", 
    "Enhookcyrillic", "Eogonek", "Eopen", "Epsilon", "Epsilontonos", 
    "Ercyrillic", "Ereversed", "Ereversedcyrillic", "Escyrillic", 
    "Esdescendercyrillic", "Esh", "Esmall", "Eta", "Etarmenian", 
    "Etatonos", "Eth", "Ethsmall", "Etilde", "Etildebelow", "Euro", 
    "Ezh", "Ezhcaron", "Ezhreversed", "F", "Fcircle", "Fdotaccent", 
    "Feharmenian", "Feicoptic", "Fhook", "Fitacyrillic", "Fiveroman", 
    "Fmonospace", "Fourroman", "Fsmall", "G", "GBsquare", "Gacute", 
    "Gamma", "Gammaafrican", "Gangiacoptic", "Gbreve", "Gcaron", 
    "Gcedilla", "Gcircle", "Gcircumflex", "Gcommaaccent", "Gdot", 
    "Gdotaccent", "Gecyrillic", "Ghadarmenian", 
    "Ghemiddlehookcyrillic", "Ghestrokecyrillic", "Gheupturncyrillic", 
    "Ghook", "Gimarmenian", "Gjecyrillic", "Gmacron", "Gmonospace", 
    "Grave", "Gravesmall", "Gsmall", "Gsmallhook", "Gstroke", "H", 
    "H18533", "H18543", "H18551", "H22073", "HPsquare", 
    "Haabkhasiancyrillic", "Hadescendercyrillic", "Hardsigncyrillic", 
    "Hbar", "Hbrevebelow", "Hcedilla", "Hcircle", "Hcircumflex", 
    "Hdieresis", "Hdotaccent", "Hdotbelow", "Hmonospace", "Hoarmenian", 
    "Horicoptic", "Hsmall", "Hungarumlaut", "Hungarumlautsmall", 
    "Hzsquare", "I", "IAcyrillic", "IJ", "IUcyrillic", "Iacute", 
    "Iacutesmall", "Ibreve", "Icaron", "Icircle", "Icircumflex", 
    "Icircumflexsmall", "Icyrillic", "Idblgrave", "Idieresis", 
    "Idieresisacute", "Idieresiscyrillic", "Idieresissmall", "Idot", 
    "Idotaccent", "Idotbelow", "Iebrevecyrillic", "Iecyrillic", 
    "Ifraktur", "Igrave", "Igravesmall", "Ihookabove", "Iicyrillic", 
    "Iinvertedbreve", "Iishortcyrillic", "Imacron", "Imacroncyrillic", 
    "Imonospace", "Iniarmenian", "Iocyrillic", "Iogonek", "Iota", 
    "Iotaafrican", "Iotadieresis", "Iotatonos", "Ismall", "Istroke", 
    "Itilde", "Itildebelow", "Izhitsacyrillic", 
    "Izhitsadblgravecyrillic", "J", "Jaarmenian", "Jcircle", 
    "Jcircumflex", "Jecyrillic", "Jheharmenian", "Jmonospace", 
    "Jsmall", "K", "KBsquare", "KKsquare", "Kabashkircyrillic", 
    "Kacute", "Kacyrillic", "Kadescendercyrillic", "Kahookcyrillic", 
    "Kappa", "Kastrokecyrillic", "Kaverticalstrokecyrillic", "Kcaron", 
    "Kcedilla", "Kcircle", "Kcommaaccent", "Kdotbelow", "Keharmenian", 
    "Kenarmenian", "Khacyrillic", "Kheicoptic", "Khook", "Kjecyrillic", 
    "Klinebelow", "Kmonospace", "Koppacyrillic", "Koppagreek", 
    "Ksicyrillic", "Ksmall", "L", "LJ", "LL", "Lacute", "Lambda", 
    "Lcaron", "Lcedilla", "Lcircle", "Lcircumflexbelow", 
    "Lcommaaccent", "Ldot", "Ldotaccent", "Ldotbelow", 
    "Ldotbelowmacron", "Liwnarmenian", "Lj", "Ljecyrillic", 
    "Llinebelow", "Lmonospace", "Lslash", "Lslashsmall", "Lsmall", "M", 
    "MBsquare", "Macron", "Macronsmall", "Macute", "Mcircle", 
    "Mdotaccent", "Mdotbelow", "Menarmenian", "Mmonospace", "Msmall", 
    "Mturned", "Mu", "N", "NJ", "Nacute", "Ncaron", "Ncedilla", 
    "Ncircle", "Ncircumflexbelow", "Ncommaaccent", "Ndotaccent", 
    "Ndotbelow", "Nhookleft", "Nineroman", "Nj", "Njecyrillic", 
    "Nlinebelow", "Nmonospace", "Nowarmenian", "Nsmall", "Ntilde", 
    "Ntildesmall", "Nu", "O", "OE", "OEsmall", "Oacute", "Oacutesmall", 
    "Obarredcyrillic", "Obarreddieresiscyrillic", "Obreve", "Ocaron", 
    "Ocenteredtilde", "Ocircle", "Ocircumflex", "Ocircumflexacute", 
    "Ocircumflexdotbelow", "Ocircumflexgrave", "Ocircumflexhookabove", 
    "Ocircumflexsmall", "Ocircumflextilde", "Ocyrillic", "Odblacute", 
    "Odblgrave", "Odieresis", "Odieresiscyrillic", "Odieresissmall", 
    "Odotbelow", "Ogoneksmall", "Ograve", "Ogravesmall", "Oharmenian", 
    "Ohm", "Ohookabove", "Ohorn", "Ohornacute", "Ohorndotbelow", 
    "Ohorngrave", "Ohornhookabove", "Ohorntilde", "Ohungarumlaut", 
    "Oi", "Oinvertedbreve", "Omacron", "Omacronacute", "Omacrongrave", 
    "Omega", "Omegacyrillic", "Omegagreek", "Omegaroundcyrillic", 
    "Omegatitlocyrillic", "Omegatonos", "Omicron", "Omicrontonos", 
    "Omonospace", "Oneroman", "Oogonek", "Oogonekmacron", "Oopen", 
    "Oslash", "Oslashacute", "Oslashsmall", "Osmall", "Ostrokeacute", 
    "Otcyrillic", "Otilde", "Otildeacute", "Otildedieresis", 
    "Otildesmall", "P", "Pacute", "Pcircle", "Pdotaccent", 
    "Pecyrillic", "Peharmenian", "Pemiddlehookcyrillic", "Phi", 
    "Phook", "Pi", "Piwrarmenian", "Pmonospace", "Psi", "Psicyrillic", 
    "Psmall", "Q", "Qcircle", "Qmonospace", "Qsmall", "R", 
    "Raarmenian", "Racute", "Rcaron", "Rcedilla", "Rcircle", 
    "Rcommaaccent", "Rdblgrave", "Rdotaccent", "Rdotbelow", 
    "Rdotbelowmacron", "Reharmenian", "Rfraktur", "Rho", "Ringsmall", 
    "Rinvertedbreve", "Rlinebelow", "Rmonospace", "Rsmall", 
    "Rsmallinverted", "Rsmallinvertedsuperior", "S", "SF010000", 
    "SF020000", "SF030000", "SF040000", "SF050000", "SF060000", 
    "SF070000", "SF080000", "SF090000", "SF100000", "SF110000", 
    "SF190000", "SF200000", "SF210000", "SF220000", "SF230000", 
    "SF240000", "SF250000", "SF260000", "SF270000", "SF280000", 
    "SF360000", "SF370000", "SF380000", "SF390000", "SF400000", 
    "SF410000", "SF420000", "SF430000", "SF440000", "SF450000", 
    "SF460000", "SF470000", "SF480000", "SF490000", "SF500000", 
    "SF510000", "SF520000", "SF530000", "SF540000", "Sacute", 
    "Sacutedotaccent", "Sampigreek", "Scaron", "Scarondotaccent", 
    "Scaronsmall", "Scedilla", "Schwa", "Schwacyrillic", 
    "Schwadieresiscyrillic", "Scircle", "Scircumflex", "Scommaaccent", 
    "Sdotaccent", "Sdotbelow", "Sdotbelowdotaccent", "Seharmenian", 
    "Sevenroman", "Shaarmenian", "Shacyrillic", "Shchacyrillic", 
    "Sheicoptic", "Shhacyrillic", "Shimacoptic", "Sigma", "Sixroman", 
    "Smonospace", "Softsigncyrillic", "Ssmall", "Stigmagreek", "T", 
    "Tau", "Tbar", "Tcaron", "Tcedilla", "Tcircle", "Tcircumflexbelow", 
    "Tcommaaccent", "Tdotaccent", "Tdotbelow", "Tecyrillic", 
    "Tedescendercyrillic", "Tenroman", "Tetsecyrillic", "Theta", 
    "Thook", "Thorn", "Thornsmall", "Threeroman", "Tildesmall", 
    "Tiwnarmenian", "Tlinebelow", "Tmonospace", "Toarmenian", 
    "Tonefive", "Tonesix", "Tonetwo", "Tretroflexhook", "Tsecyrillic", 
    "Tshecyrillic", "Tsmall", "Twelveroman", "Tworoman", "U", "Uacute", 
    "Uacutesmall", "Ubreve", "Ucaron", "Ucircle", "Ucircumflex", 
    "Ucircumflexbelow", "Ucircumflexsmall", "Ucyrillic", "Udblacute", 
    "Udblgrave", "Udieresis", "Udieresisacute", "Udieresisbelow", 
    "Udieresiscaron", "Udieresiscyrillic", "Udieresisgrave", 
    "Udieresismacron", "Udieresissmall", "Udotbelow", "Ugrave", 
    "Ugravesmall", "Uhookabove", "Uhorn", "Uhornacute", 
    "Uhorndotbelow", "Uhorngrave", "Uhornhookabove", "Uhorntilde", 
    "Uhungarumlaut", "Uhungarumlautcyrillic", "Uinvertedbreve", 
    "Ukcyrillic", "Umacron", "Umacroncyrillic", "Umacrondieresis", 
    "Umonospace", "Uogonek", "Upsilon", "Upsilon1", 
    "Upsilonacutehooksymbolgreek", "Upsilonafrican", "Upsilondieresis", 
    "Upsilondieresishooksymbolgreek", "Upsilonhooksymbol", 
    "Upsilontonos", "Uring", "Ushortcyrillic", "Usmall", 
    "Ustraightcyrillic", "Ustraightstrokecyrillic", "Utilde", 
    "Utildeacute", "Utildebelow", "V", "Vcircle", "Vdotbelow", 
    "Vecyrillic", "Vewarmenian", "Vhook", "Vmonospace", "Voarmenian", 
    "Vsmall", "Vtilde", "W", "Wacute", "Wcircle", "Wcircumflex", 
    "Wdieresis", "Wdotaccent", "Wdotbelow", "Wgrave", "Wmonospace", 
    "Wsmall", "X", "Xcircle", "Xdieresis", "Xdotaccent", "Xeharmenian", 
    "Xi", "Xmonospace", "Xsmall", "Y", "Yacute", "Yacutesmall", 
    "Yatcyrillic", "Ycircle", "Ycircumflex", "Ydieresis", 
    "Ydieresissmall", "Ydotaccent", "Ydotbelow", "Yericyrillic", 
    "Yerudieresiscyrillic", "Ygrave", "Yhook", "Yhookabove", 
    "Yiarmenian", "Yicyrillic", "Yiwnarmenian", "Ymonospace", "Ysmall", 
    "Ytilde", "Yusbigcyrillic", "Yusbigiotifiedcyrillic", 
    "Yuslittlecyrillic", "Yuslittleiotifiedcyrillic", "Z", 
    "Zaarmenian", "Zacute", "Zcaron", "Zcaronsmall", "Zcircle", 
    "Zcircumflex", "Zdot", "Zdotaccent", "Zdotbelow", "Zecyrillic", 
    "Zedescendercyrillic", "Zedieresiscyrillic", "Zeta", "Zhearmenian", 
    "Zhebrevecyrillic", "Zhecyrillic", "Zhedescendercyrillic", 
    "Zhedieresiscyrillic", "Zlinebelow", "Zmonospace", "Zsmall", 
    "Zstroke", "a", "aabengali", "aacute", "aadeva", "aagujarati", 
    "aagurmukhi", "aamatragurmukhi", "aarusquare", 
    "aavowelsignbengali", "aavowelsigndeva", "aavowelsigngujarati", 
    "abbreviationmarkarmenian", "abbreviationsigndeva", "abengali", 
    "abopomofo", "abreve", "abreveacute", "abrevecyrillic", 
    "abrevedotbelow", "abrevegrave", "abrevehookabove", "abrevetilde", 
    "acaron", "acircle", "acircumflex", "acircumflexacute", 
    "acircumflexdotbelow", "acircumflexgrave", "acircumflexhookabove", 
    "acircumflextilde", "acute", "acutebelowcmb", "acutecmb", 
    "acutecomb", "acutedeva", "acutelowmod", "acutetonecmb", 
    "acyrillic", "adblgrave", "addakgurmukhi", "adeva", "adieresis", 
    "adieresiscyrillic", "adieresismacron", "adotbelow", "adotmacron", 
    "ae", "aeacute", "aekorean", "aemacron", "afii00208", "afii08941", 
    "afii10017", "afii10018", "afii10019", "afii10020", "afii10021", 
    "afii10022", "afii10023", "afii10024", "afii10025", "afii10026", 
    "afii10027", "afii10028", "afii10029", "afii10030", "afii10031", 
    "afii10032", "afii10033", "afii10034", "afii10035", "afii10036", 
    "afii10037", "afii10038", "afii10039", "afii10040", "afii10041", 
    "afii10042", "afii10043", "afii10044", "afii10045", "afii10046", 
    "afii10047", "afii10048", "afii10049", "afii10050", "afii10051", 
    "afii10052", "afii10053", "afii10054", "afii10055", "afii10056", 
    "afii10057", "afii10058", "afii10059", "afii10060", "afii10061", 
    "afii10062", "afii10063", "afii10064", "afii10065", "afii10066", 
    "afii10067", "afii10068", "afii10069", "afii10070", "afii10071", 
    "afii10072", "afii10073", "afii10074", "afii10075", "afii10076", 
    "afii10077", "afii10078", "afii10079", "afii10080", "afii10081", 
    "afii10082", "afii10083", "afii10084", "afii10085", "afii10086", 
    "afii10087", "afii10088", "afii10089", "afii10090", "afii10091", 
    "afii10092", "afii10093", "afii10094", "afii10095", "afii10096", 
    "afii10097", "afii10098", "afii10099", "afii10100", "afii10101", 
    "afii10102", "afii10103", "afii10104", "afii10105", "afii10106", 
    "afii10107", "afii10108", "afii10109", "afii10110", "afii10145", 
    "afii10146", "afii10147", "afii10148", "afii10192", "afii10193", 
    "afii10194", "afii10195", "afii10196", "afii10831", "afii10832", 
    "afii10846", "afii299", "afii300", "afii301", "afii57381", 
    "afii57388", "afii57392", "afii57393", "afii57394", "afii57395", 
    "afii57396", "afii57397", "afii57398", "afii57399", "afii57400", 
    "afii57401", "afii57403", "afii57407", "afii57409", "afii57410", 
    "afii57411", "afii57412", "afii57413", "afii57414", "afii57415", 
    "afii57416", "afii57417", "afii57418", "afii57419", "afii57420", 
    "afii57421", "afii57422", "afii57423", "afii57424", "afii57425", 
    "afii57426", "afii57427", "afii57428", "afii57429", "afii57430", 
    "afii57431", "afii57432", "afii57433", "afii57434", "afii57440", 
    "afii57441", "afii57442", "afii57443", "afii57444", "afii57445", 
    "afii57446", "afii57448", "afii57449", "afii57450", "afii57451", 
    "afii57452", "afii57453", "afii57454", "afii57455", "afii57456", 
    "afii57457", "afii57458", "afii57470", "afii57505", "afii57506", 
    "afii57507", "afii57508", "afii57509", "afii57511", "afii57512", 
    "afii57513", "afii57514", "afii57519", "afii57534", "afii57636", 
    "afii57645", "afii57658", "afii57664", "afii57665", "afii57666", 
    "afii57667", "afii57668", "afii57669", "afii57670", "afii57671", 
    "afii57672", "afii57673", "afii57674", "afii57675", "afii57676", 
    "afii57677", "afii57678", "afii57679", "afii57680", "afii57681", 
    "afii57682", "afii57683", "afii57684", "afii57685", "afii57686", 
    "afii57687", "afii57688", "afii57689", "afii57690", "afii57694", 
    "afii57695", "afii57700", "afii57705", "afii57716", "afii57717", 
    "afii57718", "afii57723", "afii57793", "afii57794", "afii57795", 
    "afii57796", "afii57797", "afii57798", "afii57799", "afii57800", 
    "afii57801", "afii57802", "afii57803", "afii57804", "afii57806", 
    "afii57807", "afii57839", "afii57841", "afii57842", "afii57929", 
    "afii61248", "afii61289", "afii61352", "afii61573", "afii61574", 
    "afii61575", "afii61664", "afii63167", "afii64937", "agrave", 
    "agujarati", "agurmukhi", "ahiragana", "ahookabove", "aibengali", 
    "aibopomofo", "aideva", "aiecyrillic", "aigujarati", "aigurmukhi", 
    "aimatragurmukhi", "ainarabic", "ainfinalarabic", 
    "aininitialarabic", "ainmedialarabic", "ainvertedbreve", 
    "aivowelsignbengali", "aivowelsigndeva", "aivowelsigngujarati", 
    "akatakana", "akatakanahalfwidth", "akorean", "alef", "alefarabic", 
    "alefdageshhebrew", "aleffinalarabic", "alefhamzaabovearabic", 
    "alefhamzaabovefinalarabic", "alefhamzabelowarabic", 
    "alefhamzabelowfinalarabic", "alefhebrew", "aleflamedhebrew", 
    "alefmaddaabovearabic", "alefmaddaabovefinalarabic", 
    "alefmaksuraarabic", "alefmaksurafinalarabic", 
    "alefmaksurainitialarabic", "alefmaksuramedialarabic", 
    "alefpatahhebrew", "alefqamatshebrew", "aleph", "allequal", 
    "alpha", "alphatonos", "amacron", "amonospace", "ampersand", 
    "ampersandmonospace", "ampersandsmall", "amsquare", "anbopomofo", 
    "angbopomofo", "angkhankhuthai", "angle", "anglebracketleft", 
    "anglebracketleftvertical", "anglebracketright", 
    "anglebracketrightvertical", "angleleft", "angleright", "angstrom", 
    "anoteleia", "anudattadeva", "anusvarabengali", "anusvaradeva", 
    "anusvaragujarati", "aogonek", "apaatosquare", "aparen", 
    "apostrophearmenian", "apostrophemod", "apple", "approaches", 
    "approxequal", "approxequalorimage", "approximatelyequal", 
    "araeaekorean", "araeakorean", "arc", "arighthalfring", "aring", 
    "aringacute", "aringbelow", "arrowboth", "arrowdashdown", 
    "arrowdashleft", "arrowdashright", "arrowdashup", "arrowdblboth", 
    "arrowdbldown", "arrowdblleft", "arrowdblright", "arrowdblup", 
    "arrowdown", "arrowdownleft", "arrowdownright", "arrowdownwhite", 
    "arrowheaddownmod", "arrowheadleftmod", "arrowheadrightmod", 
    "arrowheadupmod", "arrowhorizex", "arrowleft", "arrowleftdbl", 
    "arrowleftdblstroke", "arrowleftoverright", "arrowleftwhite", 
    "arrowright", "arrowrightdblstroke", "arrowrightheavy", 
    "arrowrightoverleft", "arrowrightwhite", "arrowtableft", 
    "arrowtabright", "arrowup", "arrowupdn", "arrowupdnbse", 
    "arrowupdownbase", "arrowupleft", "arrowupleftofdown", 
    "arrowupright", "arrowupwhite", "arrowvertex", "asciicircum", 
    "asciicircummonospace", "asciitilde", "asciitildemonospace", 
    "ascript", "ascriptturned", "asmallhiragana", "asmallkatakana", 
    "asmallkatakanahalfwidth", "asterisk", "asteriskaltonearabic", 
    "asteriskarabic", "asteriskmath", "asteriskmonospace", 
    "asterisksmall", "asterism", "asuperior", "asymptoticallyequal", 
    "at", "atilde", "atmonospace", "atsmall", "aturned", "aubengali", 
    "aubopomofo", "audeva", "augujarati", "augurmukhi", 
    "aulengthmarkbengali", "aumatragurmukhi", "auvowelsignbengali", 
    "auvowelsigndeva", "auvowelsigngujarati", "avagrahadeva", 
    "aybarmenian", "ayin", "ayinaltonehebrew", "ayinhebrew", "b", 
    "babengali", "backslash", "backslashmonospace", "badeva", 
    "bagujarati", "bagurmukhi", "bahiragana", "bahtthai", "bakatakana", 
    "bar", "barmonospace", "bbopomofo", "bcircle", "bdotaccent", 
    "bdotbelow", "beamedsixteenthnotes", "because", "becyrillic", 
    "beharabic", "behfinalarabic", "behinitialarabic", "behiragana", 
    "behmedialarabic", "behmeeminitialarabic", "behmeemisolatedarabic", 
    "behnoonfinalarabic", "bekatakana", "benarmenian", "bet", "beta", 
    "betasymbolgreek", "betdagesh", "betdageshhebrew", "bethebrew", 
    "betrafehebrew", "bhabengali", "bhadeva", "bhagujarati", 
    "bhagurmukhi", "bhook", "bihiragana", "bikatakana", 
    "bilabialclick", "bindigurmukhi", "birusquare", "blackcircle", 
    "blackdiamond", "blackdownpointingtriangle", 
    "blackleftpointingpointer", "blackleftpointingtriangle", 
    "blacklenticularbracketleft", "blacklenticularbracketleftvertical", 
    "blacklenticularbracketright", 
    "blacklenticularbracketrightvertical", "blacklowerlefttriangle", 
    "blacklowerrighttriangle", "blackrectangle", 
    "blackrightpointingpointer", "blackrightpointingtriangle", 
    "blacksmallsquare", "blacksmilingface", "blacksquare", "blackstar", 
    "blackupperlefttriangle", "blackupperrighttriangle", 
    "blackuppointingsmalltriangle", "blackuppointingtriangle", "blank", 
    "blinebelow", "block", "bmonospace", "bobaimaithai", "bohiragana", 
    "bokatakana", "bparen", "bqsquare", "braceex", "braceleft", 
    "braceleftbt", "braceleftmid", "braceleftmonospace", 
    "braceleftsmall", "bracelefttp", "braceleftvertical", "braceright", 
    "bracerightbt", "bracerightmid", "bracerightmonospace", 
    "bracerightsmall", "bracerighttp", "bracerightvertical", 
    "bracketleft", "bracketleftbt", "bracketleftex", 
    "bracketleftmonospace", "bracketlefttp", "bracketright", 
    "bracketrightbt", "bracketrightex", "bracketrightmonospace", 
    "bracketrighttp", "breve", "brevebelowcmb", "brevecmb", 
    "breveinvertedbelowcmb", "breveinvertedcmb", 
    "breveinverteddoublecmb", "bridgebelowcmb", 
    "bridgeinvertedbelowcmb", "brokenbar", "bstroke", "bsuperior", 
    "btopbar", "buhiragana", "bukatakana", "bullet", "bulletinverse", 
    "bulletoperator", "bullseye", "c", "caarmenian", "cabengali", 
    "cacute", "cadeva", "cagujarati", "cagurmukhi", "calsquare", 
    "candrabindubengali", "candrabinducmb", "candrabindudeva", 
    "candrabindugujarati", "capslock", "careof", "caron", 
    "caronbelowcmb", "caroncmb", "carriagereturn", "cbopomofo", 
    "ccaron", "ccedilla", "ccedillaacute", "ccircle", "ccircumflex", 
    "ccurl", "cdot", "cdotaccent", "cdsquare", "cedilla", "cedillacmb", 
    "cent", "centigrade", "centinferior", "centmonospace", 
    "centoldstyle", "centsuperior", "chaarmenian", "chabengali", 
    "chadeva", "chagujarati", "chagurmukhi", "chbopomofo", 
    "cheabkhasiancyrillic", "checkmark", "checyrillic", 
    "chedescenderabkhasiancyrillic", "chedescendercyrillic", 
    "chedieresiscyrillic", "cheharmenian", "chekhakassiancyrillic", 
    "cheverticalstrokecyrillic", "chi", "chieuchacirclekorean", 
    "chieuchaparenkorean", "chieuchcirclekorean", "chieuchkorean", 
    "chieuchparenkorean", "chochangthai", "chochanthai", 
    "chochingthai", "chochoethai", "chook", "cieucacirclekorean", 
    "cieucaparenkorean", "cieuccirclekorean", "cieuckorean", 
    "cieucparenkorean", "cieucuparenkorean", "circle", 
    "circlemultiply", "circleot", "circleplus", "circlepostalmark", 
    "circlewithlefthalfblack", "circlewithrighthalfblack", 
    "circumflex", "circumflexbelowcmb", "circumflexcmb", "clear", 
    "clickalveolar", "clickdental", "clicklateral", "clickretroflex", 
    "club", "clubsuitblack", "clubsuitwhite", "cmcubedsquare", 
    "cmonospace", "cmsquaredsquare", "coarmenian", "colon", 
    "colonmonetary", "colonmonospace", "colonsign", "colonsmall", 
    "colontriangularhalfmod", "colontriangularmod", "comma", 
    "commaabovecmb", "commaaboverightcmb", "commaaccent", 
    "commaarabic", "commaarmenian", "commainferior", "commamonospace", 
    "commareversedabovecmb", "commareversedmod", "commasmall", 
    "commasuperior", "commaturnedabovecmb", "commaturnedmod", 
    "compass", "congruent", "contourintegral", "control", "controlACK", 
    "controlBEL", "controlBS", "controlCAN", "controlCR", "controlDC1", 
    "controlDC2", "controlDC3", "controlDC4", "controlDEL", 
    "controlDLE", "controlEM", "controlENQ", "controlEOT", 
    "controlESC", "controlETB", "controlETX", "controlFF", "controlFS", 
    "controlGS", "controlHT", "controlLF", "controlNAK", "controlRS", 
    "controlSI", "controlSO", "controlSOT", "controlSTX", "controlSUB", 
    "controlSYN", "controlUS", "controlVT", "copyright", 
    "copyrightsans", "copyrightserif", "cornerbracketleft", 
    "cornerbracketlefthalfwidth", "cornerbracketleftvertical", 
    "cornerbracketright", "cornerbracketrighthalfwidth", 
    "cornerbracketrightvertical", "corporationsquare", "cosquare", 
    "coverkgsquare", "cparen", "cruzeiro", "cstretched", "curlyand", 
    "curlyor", "currency", "cyrBreve", "cyrFlex", "cyrbreve", 
    "cyrflex", "d", "daarmenian", "dabengali", "dadarabic", "dadeva", 
    "dadfinalarabic", "dadinitialarabic", "dadmedialarabic", "dagesh", 
    "dageshhebrew", "dagger", "daggerdbl", "dagujarati", "dagurmukhi", 
    "dahiragana", "dakatakana", "dalarabic", "dalet", "daletdagesh", 
    "daletdageshhebrew", "dalethebrew", "dalfinalarabic", 
    "dammaarabic", "dammalowarabic", "dammatanaltonearabic", 
    "dammatanarabic", "danda", "dargahebrew", "dargalefthebrew", 
    "dasiapneumatacyrilliccmb", "dblGrave", "dblanglebracketleft", 
    "dblanglebracketleftvertical", "dblanglebracketright", 
    "dblanglebracketrightvertical", "dblarchinvertedbelowcmb", 
    "dblarrowleft", "dblarrowright", "dbldanda", "dblgrave", 
    "dblgravecmb", "dblintegral", "dbllowline", "dbllowlinecmb", 
    "dbloverlinecmb", "dblprimemod", "dblverticalbar", 
    "dblverticallineabovecmb", "dbopomofo", "dbsquare", "dcaron", 
    "dcedilla", "dcircle", "dcircumflexbelow", "dcroat", "ddabengali", 
    "ddadeva", "ddagujarati", "ddagurmukhi", "ddalarabic", 
    "ddalfinalarabic", "dddhadeva", "ddhabengali", "ddhadeva", 
    "ddhagujarati", "ddhagurmukhi", "ddotaccent", "ddotbelow", 
    "decimalseparatorarabic", "decimalseparatorpersian", "decyrillic", 
    "degree", "dehihebrew", "dehiragana", "deicoptic", "dekatakana", 
    "deleteleft", "deleteright", "delta", "deltaturned", 
    "denominatorminusonenumeratorbengali", "dezh", "dhabengali", 
    "dhadeva", "dhagujarati", "dhagurmukhi", "dhook", "dialytikatonos", 
    "dialytikatonoscmb", "diamond", "diamondsuitwhite", "dieresis", 
    "dieresisacute", "dieresisbelowcmb", "dieresiscmb", 
    "dieresisgrave", "dieresistonos", "dihiragana", "dikatakana", 
    "dittomark", "divide", "divides", "divisionslash", "djecyrillic", 
    "dkshade", "dlinebelow", "dlsquare", "dmacron", "dmonospace", 
    "dnblock", "dochadathai", "dodekthai", "dohiragana", "dokatakana", 
    "dollar", "dollarinferior", "dollarmonospace", "dollaroldstyle", 
    "dollarsmall", "dollarsuperior", "dong", "dorusquare", "dotaccent", 
    "dotaccentcmb", "dotbelowcmb", "dotbelowcomb", "dotkatakana", 
    "dotlessi", "dotlessj", "dotlessjstrokehook", "dotmath", 
    "dottedcircle", "doubleyodpatah", "doubleyodpatahhebrew", 
    "downtackbelowcmb", "downtackmod", "dparen", "dsuperior", "dtail", 
    "dtopbar", "duhiragana", "dukatakana", "dz", "dzaltone", "dzcaron", 
    "dzcurl", "dzeabkhasiancyrillic", "dzecyrillic", "dzhecyrillic", 
    "e", "eacute", "earth", "ebengali", "ebopomofo", "ebreve", 
    "ecandradeva", "ecandragujarati", "ecandravowelsigndeva", 
    "ecandravowelsigngujarati", "ecaron", "ecedillabreve", 
    "echarmenian", "echyiwnarmenian", "ecircle", "ecircumflex", 
    "ecircumflexacute", "ecircumflexbelow", "ecircumflexdotbelow", 
    "ecircumflexgrave", "ecircumflexhookabove", "ecircumflextilde", 
    "ecyrillic", "edblgrave", "edeva", "edieresis", "edot", 
    "edotaccent", "edotbelow", "eegurmukhi", "eematragurmukhi", 
    "efcyrillic", "egrave", "egujarati", "eharmenian", "ehbopomofo", 
    "ehiragana", "ehookabove", "eibopomofo", "eight", "eightarabic", 
    "eightbengali", "eightcircle", "eightcircleinversesansserif", 
    "eightdeva", "eighteencircle", "eighteenparen", "eighteenperiod", 
    "eightgujarati", "eightgurmukhi", "eighthackarabic", 
    "eighthangzhou", "eighthnotebeamed", "eightideographicparen", 
    "eightinferior", "eightmonospace", "eightoldstyle", "eightparen", 
    "eightperiod", "eightpersian", "eightroman", "eightsuperior", 
    "eightthai", "einvertedbreve", "eiotifiedcyrillic", "ekatakana", 
    "ekatakanahalfwidth", "ekonkargurmukhi", "ekorean", "elcyrillic", 
    "element", "elevencircle", "elevenparen", "elevenperiod", 
    "elevenroman", "ellipsis", "ellipsisvertical", "emacron", 
    "emacronacute", "emacrongrave", "emcyrillic", "emdash", 
    "emdashvertical", "emonospace", "emphasismarkarmenian", "emptyset", 
    "enbopomofo", "encyrillic", "endash", "endashvertical", 
    "endescendercyrillic", "eng", "engbopomofo", "enghecyrillic", 
    "enhookcyrillic", "enspace", "eogonek", "eokorean", "eopen", 
    "eopenclosed", "eopenreversed", "eopenreversedclosed", 
    "eopenreversedhook", "eparen", "epsilon", "epsilontonos", "equal", 
    "equalmonospace", "equalsmall", "equalsuperior", "equivalence", 
    "erbopomofo", "ercyrillic", "ereversed", "ereversedcyrillic", 
    "escyrillic", "esdescendercyrillic", "esh", "eshcurl", 
    "eshortdeva", "eshortvowelsigndeva", "eshreversedloop", 
    "eshsquatreversed", "esmallhiragana", "esmallkatakana", 
    "esmallkatakanahalfwidth", "estimated", "esuperior", "eta", 
    "etarmenian", "etatonos", "eth", "etilde", "etildebelow", 
    "etnahtafoukhhebrew", "etnahtafoukhlefthebrew", "etnahtahebrew", 
    "etnahtalefthebrew", "eturned", "eukorean", "euro", 
    "evowelsignbengali", "evowelsigndeva", "evowelsigngujarati", 
    "exclam", "exclamarmenian", "exclamdbl", "exclamdown", 
    "exclamdownsmall", "exclammonospace", "exclamsmall", "existential", 
    "ezh", "ezhcaron", "ezhcurl", "ezhreversed", "ezhtail", "f", 
    "fadeva", "fagurmukhi", "fahrenheit", "fathaarabic", 
    "fathalowarabic", "fathatanarabic", "fbopomofo", "fcircle", 
    "fdotaccent", "feharabic", "feharmenian", "fehfinalarabic", 
    "fehinitialarabic", "fehmedialarabic", "feicoptic", "female", "ff", 
    "ffi", "ffl", "fi", "fifteencircle", "fifteenparen", 
    "fifteenperiod", "figuredash", "filledbox", "filledrect", 
    "finalkaf", "finalkafdagesh", "finalkafdageshhebrew", 
    "finalkafhebrew", "finalmem", "finalmemhebrew", "finalnun", 
    "finalnunhebrew", "finalpe", "finalpehebrew", "finaltsadi", 
    "finaltsadihebrew", "firsttonechinese", "fisheye", "fitacyrillic", 
    "five", "fivearabic", "fivebengali", "fivecircle", 
    "fivecircleinversesansserif", "fivedeva", "fiveeighths", 
    "fivegujarati", "fivegurmukhi", "fivehackarabic", "fivehangzhou", 
    "fiveideographicparen", "fiveinferior", "fivemonospace", 
    "fiveoldstyle", "fiveparen", "fiveperiod", "fivepersian", 
    "fiveroman", "fivesuperior", "fivethai", "fl", "florin", 
    "fmonospace", "fmsquare", "fofanthai", "fofathai", "fongmanthai", 
    "forall", "four", "fourarabic", "fourbengali", "fourcircle", 
    "fourcircleinversesansserif", "fourdeva", "fourgujarati", 
    "fourgurmukhi", "fourhackarabic", "fourhangzhou", 
    "fourideographicparen", "fourinferior", "fourmonospace", 
    "fournumeratorbengali", "fouroldstyle", "fourparen", "fourperiod", 
    "fourpersian", "fourroman", "foursuperior", "fourteencircle", 
    "fourteenparen", "fourteenperiod", "fourthai", "fourthtonechinese", 
    "fparen", "fraction", "franc", "g", "gabengali", "gacute", 
    "gadeva", "gafarabic", "gaffinalarabic", "gafinitialarabic", 
    "gafmedialarabic", "gagujarati", "gagurmukhi", "gahiragana", 
    "gakatakana", "gamma", "gammalatinsmall", "gammasuperior", 
    "gangiacoptic", "gbopomofo", "gbreve", "gcaron", "gcedilla", 
    "gcircle", "gcircumflex", "gcommaaccent", "gdot", "gdotaccent", 
    "gecyrillic", "gehiragana", "gekatakana", "geometricallyequal", 
    "gereshaccenthebrew", "gereshhebrew", "gereshmuqdamhebrew", 
    "germandbls", "gershayimaccenthebrew", "gershayimhebrew", 
    "getamark", "ghabengali", "ghadarmenian", "ghadeva", "ghagujarati", 
    "ghagurmukhi", "ghainarabic", "ghainfinalarabic", 
    "ghaininitialarabic", "ghainmedialarabic", "ghemiddlehookcyrillic", 
    "ghestrokecyrillic", "gheupturncyrillic", "ghhadeva", 
    "ghhagurmukhi", "ghook", "ghzsquare", "gihiragana", "gikatakana", 
    "gimarmenian", "gimel", "gimeldagesh", "gimeldageshhebrew", 
    "gimelhebrew", "gjecyrillic", "glottalinvertedstroke", 
    "glottalstop", "glottalstopinverted", "glottalstopmod", 
    "glottalstopreversed", "glottalstopreversedmod", 
    "glottalstopreversedsuperior", "glottalstopstroke", 
    "glottalstopstrokereversed", "gmacron", "gmonospace", "gohiragana", 
    "gokatakana", "gparen", "gpasquare", "gradient", "grave", 
    "gravebelowcmb", "gravecmb", "gravecomb", "gravedeva", 
    "gravelowmod", "gravemonospace", "gravetonecmb", "greater", 
    "greaterequal", "greaterequalorless", "greatermonospace", 
    "greaterorequivalent", "greaterorless", "greateroverequal", 
    "greatersmall", "gscript", "gstroke", "guhiragana", 
    "guillemotleft", "guillemotright", "guilsinglleft", 
    "guilsinglright", "gukatakana", "guramusquare", "gysquare", "h", 
    "haabkhasiancyrillic", "haaltonearabic", "habengali", 
    "hadescendercyrillic", "hadeva", "hagujarati", "hagurmukhi", 
    "haharabic", "hahfinalarabic", "hahinitialarabic", "hahiragana", 
    "hahmedialarabic", "haitusquare", "hakatakana", 
    "hakatakanahalfwidth", "halantgurmukhi", "hamzaarabic", 
    "hamzalowarabic", "hangulfiller", "hardsigncyrillic", 
    "harpoonleftbarbup", "harpoonrightbarbup", "hasquare", 
    "hatafpatah", "hatafpatah16", "hatafpatah23", "hatafpatah2f", 
    "hatafpatahhebrew", "hatafpatahnarrowhebrew", 
    "hatafpatahquarterhebrew", "hatafpatahwidehebrew", "hatafqamats", 
    "hatafqamats1b", "hatafqamats28", "hatafqamats34", 
    "hatafqamatshebrew", "hatafqamatsnarrowhebrew", 
    "hatafqamatsquarterhebrew", "hatafqamatswidehebrew", "hatafsegol", 
    "hatafsegol17", "hatafsegol24", "hatafsegol30", "hatafsegolhebrew", 
    "hatafsegolnarrowhebrew", "hatafsegolquarterhebrew", 
    "hatafsegolwidehebrew", "hbar", "hbopomofo", "hbrevebelow", 
    "hcedilla", "hcircle", "hcircumflex", "hdieresis", "hdotaccent", 
    "hdotbelow", "he", "heart", "heartsuitblack", "heartsuitwhite", 
    "hedagesh", "hedageshhebrew", "hehaltonearabic", "heharabic", 
    "hehebrew", "hehfinalaltonearabic", "hehfinalalttwoarabic", 
    "hehfinalarabic", "hehhamzaabovefinalarabic", 
    "hehhamzaaboveisolatedarabic", "hehinitialaltonearabic", 
    "hehinitialarabic", "hehiragana", "hehmedialaltonearabic", 
    "hehmedialarabic", "heiseierasquare", "hekatakana", 
    "hekatakanahalfwidth", "hekutaarusquare", "henghook", 
    "herutusquare", "het", "hethebrew", "hhook", "hhooksuperior", 
    "hieuhacirclekorean", "hieuhaparenkorean", "hieuhcirclekorean", 
    "hieuhkorean", "hieuhparenkorean", "hihiragana", "hikatakana", 
    "hikatakanahalfwidth", "hiriq", "hiriq14", "hiriq21", "hiriq2d", 
    "hiriqhebrew", "hiriqnarrowhebrew", "hiriqquarterhebrew", 
    "hiriqwidehebrew", "hlinebelow", "hmonospace", "hoarmenian", 
    "hohipthai", "hohiragana", "hokatakana", "hokatakanahalfwidth", 
    "holam", "holam19", "holam26", "holam32", "holamhebrew", 
    "holamnarrowhebrew", "holamquarterhebrew", "holamwidehebrew", 
    "honokhukthai", "hookabovecomb", "hookcmb", 
    "hookpalatalizedbelowcmb", "hookretroflexbelowcmb", "hoonsquare", 
    "horicoptic", "horizontalbar", "horncmb", "hotsprings", "house", 
    "hparen", "hsuperior", "hturned", "huhiragana", "huiitosquare", 
    "hukatakana", "hukatakanahalfwidth", "hungarumlaut", 
    "hungarumlautcmb", "hv", "hyphen", "hypheninferior", 
    "hyphenmonospace", "hyphensmall", "hyphensuperior", "hyphentwo", 
    "i", "iacute", "iacyrillic", "ibengali", "ibopomofo", "ibreve", 
    "icaron", "icircle", "icircumflex", "icyrillic", "idblgrave", 
    "ideographearthcircle", "ideographfirecircle", 
    "ideographicallianceparen", "ideographiccallparen", 
    "ideographiccentrecircle", "ideographicclose", "ideographiccomma", 
    "ideographiccommaleft", "ideographiccongratulationparen", 
    "ideographiccorrectcircle", "ideographicearthparen", 
    "ideographicenterpriseparen", "ideographicexcellentcircle", 
    "ideographicfestivalparen", "ideographicfinancialcircle", 
    "ideographicfinancialparen", "ideographicfireparen", 
    "ideographichaveparen", "ideographichighcircle", 
    "ideographiciterationmark", "ideographiclaborcircle", 
    "ideographiclaborparen", "ideographicleftcircle", 
    "ideographiclowcircle", "ideographicmedicinecircle", 
    "ideographicmetalparen", "ideographicmoonparen", 
    "ideographicnameparen", "ideographicperiod", 
    "ideographicprintcircle", "ideographicreachparen", 
    "ideographicrepresentparen", "ideographicresourceparen", 
    "ideographicrightcircle", "ideographicsecretcircle", 
    "ideographicselfparen", "ideographicsocietyparen", 
    "ideographicspace", "ideographicspecialparen", 
    "ideographicstockparen", "ideographicstudyparen", 
    "ideographicsunparen", "ideographicsuperviseparen", 
    "ideographicwaterparen", "ideographicwoodparen", "ideographiczero", 
    "ideographmetalcircle", "ideographmooncircle", 
    "ideographnamecircle", "ideographsuncircle", 
    "ideographwatercircle", "ideographwoodcircle", "ideva", 
    "idieresis", "idieresisacute", "idieresiscyrillic", "idotbelow", 
    "iebrevecyrillic", "iecyrillic", "ieungacirclekorean", 
    "ieungaparenkorean", "ieungcirclekorean", "ieungkorean", 
    "ieungparenkorean", "igrave", "igujarati", "igurmukhi", 
    "ihiragana", "ihookabove", "iibengali", "iicyrillic", "iideva", 
    "iigujarati", "iigurmukhi", "iimatragurmukhi", "iinvertedbreve", 
    "iishortcyrillic", "iivowelsignbengali", "iivowelsigndeva", 
    "iivowelsigngujarati", "ij", "ikatakana", "ikatakanahalfwidth", 
    "ikorean", "ilde", "iluyhebrew", "imacron", "imacroncyrillic", 
    "imageorapproximatelyequal", "imatragurmukhi", "imonospace", 
    "increment", "infinity", "iniarmenian", "integral", 
    "integralbottom", "integralbt", "integralex", "integraltop", 
    "integraltp", "intersection", "intisquare", "invbullet", 
    "invcircle", "invsmileface", "iocyrillic", "iogonek", "iota", 
    "iotadieresis", "iotadieresistonos", "iotalatin", "iotatonos", 
    "iparen", "irigurmukhi", "ismallhiragana", "ismallkatakana", 
    "ismallkatakanahalfwidth", "issharbengali", "istroke", "isuperior", 
    "iterationhiragana", "iterationkatakana", "itilde", "itildebelow", 
    "iubopomofo", "iucyrillic", "ivowelsignbengali", "ivowelsigndeva", 
    "ivowelsigngujarati", "izhitsacyrillic", "izhitsadblgravecyrillic", 
    "j", "jaarmenian", "jabengali", "jadeva", "jagujarati", 
    "jagurmukhi", "jbopomofo", "jcaron", "jcircle", "jcircumflex", 
    "jcrossedtail", "jdotlessstroke", "jecyrillic", "jeemarabic", 
    "jeemfinalarabic", "jeeminitialarabic", "jeemmedialarabic", 
    "jeharabic", "jehfinalarabic", "jhabengali", "jhadeva", 
    "jhagujarati", "jhagurmukhi", "jheharmenian", "jis", "jmonospace", 
    "jparen", "jsuperior", "k", "kabashkircyrillic", "kabengali", 
    "kacute", "kacyrillic", "kadescendercyrillic", "kadeva", "kaf", 
    "kafarabic", "kafdagesh", "kafdageshhebrew", "kaffinalarabic", 
    "kafhebrew", "kafinitialarabic", "kafmedialarabic", 
    "kafrafehebrew", "kagujarati", "kagurmukhi", "kahiragana", 
    "kahookcyrillic", "kakatakana", "kakatakanahalfwidth", "kappa", 
    "kappasymbolgreek", "kapyeounmieumkorean", "kapyeounphieuphkorean", 
    "kapyeounpieupkorean", "kapyeounssangpieupkorean", "karoriisquare", 
    "kashidaautoarabic", "kashidaautonosidebearingarabic", 
    "kasmallkatakana", "kasquare", "kasraarabic", "kasratanarabic", 
    "kastrokecyrillic", "katahiraprolongmarkhalfwidth", 
    "kaverticalstrokecyrillic", "kbopomofo", "kcalsquare", "kcaron", 
    "kcedilla", "kcircle", "kcommaaccent", "kdotbelow", "keharmenian", 
    "kehiragana", "kekatakana", "kekatakanahalfwidth", "kenarmenian", 
    "kesmallkatakana", "kgreenlandic", "khabengali", "khacyrillic", 
    "khadeva", "khagujarati", "khagurmukhi", "khaharabic", 
    "khahfinalarabic", "khahinitialarabic", "khahmedialarabic", 
    "kheicoptic", "khhadeva", "khhagurmukhi", "khieukhacirclekorean", 
    "khieukhaparenkorean", "khieukhcirclekorean", "khieukhkorean", 
    "khieukhparenkorean", "khokhaithai", "khokhonthai", "khokhuatthai", 
    "khokhwaithai", "khomutthai", "khook", "khorakhangthai", 
    "khzsquare", "kihiragana", "kikatakana", "kikatakanahalfwidth", 
    "kiroguramusquare", "kiromeetorusquare", "kirosquare", 
    "kiyeokacirclekorean", "kiyeokaparenkorean", "kiyeokcirclekorean", 
    "kiyeokkorean", "kiyeokparenkorean", "kiyeoksioskorean", 
    "kjecyrillic", "klinebelow", "klsquare", "kmcubedsquare", 
    "kmonospace", "kmsquaredsquare", "kohiragana", "kohmsquare", 
    "kokaithai", "kokatakana", "kokatakanahalfwidth", "kooposquare", 
    "koppacyrillic", "koreanstandardsymbol", "koroniscmb", "kparen", 
    "kpasquare", "ksicyrillic", "ktsquare", "kturned", "kuhiragana", 
    "kukatakana", "kukatakanahalfwidth", "kvsquare", "kwsquare", "l", 
    "labengali", "lacute", "ladeva", "lagujarati", "lagurmukhi", 
    "lakkhangyaothai", "lamaleffinalarabic", 
    "lamalefhamzaabovefinalarabic", "lamalefhamzaaboveisolatedarabic", 
    "lamalefhamzabelowfinalarabic", "lamalefhamzabelowisolatedarabic", 
    "lamalefisolatedarabic", "lamalefmaddaabovefinalarabic", 
    "lamalefmaddaaboveisolatedarabic", "lamarabic", "lambda", 
    "lambdastroke", "lamed", "lameddagesh", "lameddageshhebrew", 
    "lamedhebrew", "lamfinalarabic", "lamhahinitialarabic", 
    "laminitialarabic", "lamjeeminitialarabic", "lamkhahinitialarabic", 
    "lamlamhehisolatedarabic", "lammedialarabic", 
    "lammeemhahinitialarabic", "lammeeminitialarabic", "largecircle", 
    "lbar", "lbelt", "lbopomofo", "lcaron", "lcedilla", "lcircle", 
    "lcircumflexbelow", "lcommaaccent", "ldot", "ldotaccent", 
    "ldotbelow", "ldotbelowmacron", "leftangleabovecmb", 
    "lefttackbelowcmb", "less", "lessequal", "lessequalorgreater", 
    "lessmonospace", "lessorequivalent", "lessorgreater", 
    "lessoverequal", "lesssmall", "lezh", "lfblock", "lhookretroflex", 
    "lira", "liwnarmenian", "lj", "ljecyrillic", "ll", "lladeva", 
    "llagujarati", "llinebelow", "llladeva", "llvocalicbengali", 
    "llvocalicdeva", "llvocalicvowelsignbengali", 
    "llvocalicvowelsigndeva", "lmiddletilde", "lmonospace", "lmsquare", 
    "lochulathai", "logicaland", "logicalnot", "logicalnotreversed", 
    "logicalor", "lolingthai", "longs", "lowlinecenterline", 
    "lowlinecmb", "lowlinedashed", "lozenge", "lparen", "lslash", 
    "lsquare", "lsuperior", "ltshade", "luthai", "lvocalicbengali", 
    "lvocalicdeva", "lvocalicvowelsignbengali", 
    "lvocalicvowelsigndeva", "lxsquare", "m", "mabengali", "macron", 
    "macronbelowcmb", "macroncmb", "macronlowmod", "macronmonospace", 
    "macute", "madeva", "magujarati", "magurmukhi", "mahapakhhebrew", 
    "mahapakhlefthebrew", "mahiragana", "maichattawalowleftthai", 
    "maichattawalowrightthai", "maichattawathai", 
    "maichattawaupperleftthai", "maieklowleftthai", 
    "maieklowrightthai", "maiekthai", "maiekupperleftthai", 
    "maihanakatleftthai", "maihanakatthai", "maitaikhuleftthai", 
    "maitaikhuthai", "maitholowleftthai", "maitholowrightthai", 
    "maithothai", "maithoupperleftthai", "maitrilowleftthai", 
    "maitrilowrightthai", "maitrithai", "maitriupperleftthai", 
    "maiyamokthai", "makatakana", "makatakanahalfwidth", "male", 
    "mansyonsquare", "maqafhebrew", "mars", "masoracirclehebrew", 
    "masquare", "mbopomofo", "mbsquare", "mcircle", "mcubedsquare", 
    "mdotaccent", "mdotbelow", "meemarabic", "meemfinalarabic", 
    "meeminitialarabic", "meemmedialarabic", "meemmeeminitialarabic", 
    "meemmeemisolatedarabic", "meetorusquare", "mehiragana", 
    "meizierasquare", "mekatakana", "mekatakanahalfwidth", "mem", 
    "memdagesh", "memdageshhebrew", "memhebrew", "menarmenian", 
    "merkhahebrew", "merkhakefulahebrew", "merkhakefulalefthebrew", 
    "merkhalefthebrew", "mhook", "mhzsquare", 
    "middledotkatakanahalfwidth", "middot", "mieumacirclekorean", 
    "mieumaparenkorean", "mieumcirclekorean", "mieumkorean", 
    "mieumpansioskorean", "mieumparenkorean", "mieumpieupkorean", 
    "mieumsioskorean", "mihiragana", "mikatakana", 
    "mikatakanahalfwidth", "minus", "minusbelowcmb", "minuscircle", 
    "minusmod", "minusplus", "minute", "miribaarusquare", "mirisquare", 
    "mlonglegturned", "mlsquare", "mmcubedsquare", "mmonospace", 
    "mmsquaredsquare", "mohiragana", "mohmsquare", "mokatakana", 
    "mokatakanahalfwidth", "molsquare", "momathai", "moverssquare", 
    "moverssquaredsquare", "mparen", "mpasquare", "mssquare", 
    "msuperior", "mturned", "mu", "mu1", "muasquare", "muchgreater", 
    "muchless", "mufsquare", "mugreek", "mugsquare", "muhiragana", 
    "mukatakana", "mukatakanahalfwidth", "mulsquare", "multiply", 
    "mumsquare", "munahhebrew", "munahlefthebrew", "musicalnote", 
    "musicalnotedbl", "musicflatsign", "musicsharpsign", "mussquare", 
    "muvsquare", "muwsquare", "mvmegasquare", "mvsquare", 
    "mwmegasquare", "mwsquare", "n", "nabengali", "nabla", "nacute", 
    "nadeva", "nagujarati", "nagurmukhi", "nahiragana", "nakatakana", 
    "nakatakanahalfwidth", "napostrophe", "nasquare", "nbopomofo", 
    "nbspace", "ncaron", "ncedilla", "ncircle", "ncircumflexbelow", 
    "ncommaaccent", "ndotaccent", "ndotbelow", "nehiragana", 
    "nekatakana", "nekatakanahalfwidth", "newsheqelsign", "nfsquare", 
    "ngabengali", "ngadeva", "ngagujarati", "ngagurmukhi", 
    "ngonguthai", "nhiragana", "nhookleft", "nhookretroflex", 
    "nieunacirclekorean", "nieunaparenkorean", "nieuncieuckorean", 
    "nieuncirclekorean", "nieunhieuhkorean", "nieunkorean", 
    "nieunpansioskorean", "nieunparenkorean", "nieunsioskorean", 
    "nieuntikeutkorean", "nihiragana", "nikatakana", 
    "nikatakanahalfwidth", "nikhahitleftthai", "nikhahitthai", "nine", 
    "ninearabic", "ninebengali", "ninecircle", 
    "ninecircleinversesansserif", "ninedeva", "ninegujarati", 
    "ninegurmukhi", "ninehackarabic", "ninehangzhou", 
    "nineideographicparen", "nineinferior", "ninemonospace", 
    "nineoldstyle", "nineparen", "nineperiod", "ninepersian", 
    "nineroman", "ninesuperior", "nineteencircle", "nineteenparen", 
    "nineteenperiod", "ninethai", "nj", "njecyrillic", "nkatakana", 
    "nkatakanahalfwidth", "nlegrightlong", "nlinebelow", "nmonospace", 
    "nmsquare", "nnabengali", "nnadeva", "nnagujarati", "nnagurmukhi", 
    "nnnadeva", "nohiragana", "nokatakana", "nokatakanahalfwidth", 
    "nonbreakingspace", "nonenthai", "nonmarkingreturn", "nonuthai", 
    "noonarabic", "noonfinalarabic", "noonghunnaarabic", 
    "noonghunnafinalarabic", "nooninitialarabic", 
    "noonjeeminitialarabic", "noonjeemisolatedarabic", 
    "noonmedialarabic", "noonmeeminitialarabic", 
    "noonmeemisolatedarabic", "noonnoonfinalarabic", "notcontains", 
    "notelement", "notelementof", "notequal", "notgreater", 
    "notgreaternorequal", "notgreaternorless", "notidentical", 
    "notless", "notlessnorequal", "notparallel", "notprecedes", 
    "notsubset", "notsucceeds", "notsuperset", "nowarmenian", "nparen", 
    "nssquare", "nsuperior", "ntilde", "nu", "nuhiragana", 
    "nukatakana", "nukatakanahalfwidth", "nuktabengali", "nuktadeva", 
    "nuktagujarati", "nuktagurmukhi", "numbersign", 
    "numbersignmonospace", "numbersignsmall", "numeralsigngreek", 
    "numeralsignlowergreek", "numero", "nun", "nundagesh", 
    "nundageshhebrew", "nunhebrew", "nvsquare", "nwsquare", 
    "nyabengali", "nyadeva", "nyagujarati", "nyagurmukhi", "o", 
    "oacute", "oangthai", "obarred", "obarredcyrillic", 
    "obarreddieresiscyrillic", "obengali", "obopomofo", "obreve", 
    "ocandradeva", "ocandragujarati", "ocandravowelsigndeva", 
    "ocandravowelsigngujarati", "ocaron", "ocircle", "ocircumflex", 
    "ocircumflexacute", "ocircumflexdotbelow", "ocircumflexgrave", 
    "ocircumflexhookabove", "ocircumflextilde", "ocyrillic", 
    "odblacute", "odblgrave", "odeva", "odieresis", 
    "odieresiscyrillic", "odotbelow", "oe", "oekorean", "ogonek", 
    "ogonekcmb", "ograve", "ogujarati", "oharmenian", "ohiragana", 
    "ohookabove", "ohorn", "ohornacute", "ohorndotbelow", "ohorngrave", 
    "ohornhookabove", "ohorntilde", "ohungarumlaut", "oi", 
    "oinvertedbreve", "okatakana", "okatakanahalfwidth", "okorean", 
    "olehebrew", "omacron", "omacronacute", "omacrongrave", "omdeva", 
    "omega", "omega1", "omegacyrillic", "omegalatinclosed", 
    "omegaroundcyrillic", "omegatitlocyrillic", "omegatonos", 
    "omgujarati", "omicron", "omicrontonos", "omonospace", "one", 
    "onearabic", "onebengali", "onecircle", 
    "onecircleinversesansserif", "onedeva", "onedotenleader", 
    "oneeighth", "onefitted", "onegujarati", "onegurmukhi", 
    "onehackarabic", "onehalf", "onehangzhou", "oneideographicparen", 
    "oneinferior", "onemonospace", "onenumeratorbengali", 
    "oneoldstyle", "oneparen", "oneperiod", "onepersian", "onequarter", 
    "oneroman", "onesuperior", "onethai", "onethird", "oogonek", 
    "oogonekmacron", "oogurmukhi", "oomatragurmukhi", "oopen", 
    "oparen", "openbullet", "option", "ordfeminine", "ordmasculine", 
    "orthogonal", "oshortdeva", "oshortvowelsigndeva", "oslash", 
    "oslashacute", "osmallhiragana", "osmallkatakana", 
    "osmallkatakanahalfwidth", "ostrokeacute", "osuperior", 
    "otcyrillic", "otilde", "otildeacute", "otildedieresis", 
    "oubopomofo", "overline", "overlinecenterline", "overlinecmb", 
    "overlinedashed", "overlinedblwavy", "overlinewavy", "overscore", 
    "ovowelsignbengali", "ovowelsigndeva", "ovowelsigngujarati", "p", 
    "paampssquare", "paasentosquare", "pabengali", "pacute", "padeva", 
    "pagedown", "pageup", "pagujarati", "pagurmukhi", "pahiragana", 
    "paiyannoithai", "pakatakana", "palatalizationcyrilliccmb", 
    "palochkacyrillic", "pansioskorean", "paragraph", "parallel", 
    "parenleft", "parenleftaltonearabic", "parenleftbt", "parenleftex", 
    "parenleftinferior", "parenleftmonospace", "parenleftsmall", 
    "parenleftsuperior", "parenlefttp", "parenleftvertical", 
    "parenright", "parenrightaltonearabic", "parenrightbt", 
    "parenrightex", "parenrightinferior", "parenrightmonospace", 
    "parenrightsmall", "parenrightsuperior", "parenrighttp", 
    "parenrightvertical", "partialdiff", "paseqhebrew", "pashtahebrew", 
    "pasquare", "patah", "patah11", "patah1d", "patah2a", 
    "patahhebrew", "patahnarrowhebrew", "patahquarterhebrew", 
    "patahwidehebrew", "pazerhebrew", "pbopomofo", "pcircle", 
    "pdotaccent", "pe", "pecyrillic", "pedagesh", "pedageshhebrew", 
    "peezisquare", "pefinaldageshhebrew", "peharabic", "peharmenian", 
    "pehebrew", "pehfinalarabic", "pehinitialarabic", "pehiragana", 
    "pehmedialarabic", "pekatakana", "pemiddlehookcyrillic", 
    "perafehebrew", "percent", "percentarabic", "percentmonospace", 
    "percentsmall", "period", "periodarmenian", "periodcentered", 
    "periodhalfwidth", "periodinferior", "periodmonospace", 
    "periodsmall", "periodsuperior", "perispomenigreekcmb", 
    "perpendicular", "perthousand", "peseta", "pfsquare", "phabengali", 
    "phadeva", "phagujarati", "phagurmukhi", "phi", "phi1", 
    "phieuphacirclekorean", "phieuphaparenkorean", 
    "phieuphcirclekorean", "phieuphkorean", "phieuphparenkorean", 
    "philatin", "phinthuthai", "phisymbolgreek", "phook", 
    "phophanthai", "phophungthai", "phosamphaothai", "pi", 
    "pieupacirclekorean", "pieupaparenkorean", "pieupcieuckorean", 
    "pieupcirclekorean", "pieupkiyeokkorean", "pieupkorean", 
    "pieupparenkorean", "pieupsioskiyeokkorean", "pieupsioskorean", 
    "pieupsiostikeutkorean", "pieupthieuthkorean", "pieuptikeutkorean", 
    "pihiragana", "pikatakana", "pisymbolgreek", "piwrarmenian", 
    "plus", "plusbelowcmb", "pluscircle", "plusminus", "plusmod", 
    "plusmonospace", "plussmall", "plussuperior", "pmonospace", 
    "pmsquare", "pohiragana", "pointingindexdownwhite", 
    "pointingindexleftwhite", "pointingindexrightwhite", 
    "pointingindexupwhite", "pokatakana", "poplathai", "postalmark", 
    "postalmarkface", "pparen", "precedes", "prescription", "primemod", 
    "primereversed", "product", "projective", "prolongedkana", 
    "propellor", "propersubset", "propersuperset", "proportion", 
    "proportional", "psi", "psicyrillic", "psilipneumatacyrilliccmb", 
    "pssquare", "puhiragana", "pukatakana", "pvsquare", "pwsquare", 
    "q", "qadeva", "qadmahebrew", "qafarabic", "qaffinalarabic", 
    "qafinitialarabic", "qafmedialarabic", "qamats", "qamats10", 
    "qamats1a", "qamats1c", "qamats27", "qamats29", "qamats33", 
    "qamatsde", "qamatshebrew", "qamatsnarrowhebrew", 
    "qamatsqatanhebrew", "qamatsqatannarrowhebrew", 
    "qamatsqatanquarterhebrew", "qamatsqatanwidehebrew", 
    "qamatsquarterhebrew", "qamatswidehebrew", "qarneyparahebrew", 
    "qbopomofo", "qcircle", "qhook", "qmonospace", "qof", "qofdagesh", 
    "qofdageshhebrew", "qofhebrew", "qparen", "quarternote", "qubuts", 
    "qubuts18", "qubuts25", "qubuts31", "qubutshebrew", 
    "qubutsnarrowhebrew", "qubutsquarterhebrew", "qubutswidehebrew", 
    "question", "questionarabic", "questionarmenian", "questiondown", 
    "questiondownsmall", "questiongreek", "questionmonospace", 
    "questionsmall", "quotedbl", "quotedblbase", "quotedblleft", 
    "quotedblmonospace", "quotedblprime", "quotedblprimereversed", 
    "quotedblright", "quoteleft", "quoteleftreversed", "quotereversed", 
    "quoteright", "quoterightn", "quotesinglbase", "quotesingle", 
    "quotesinglemonospace", "r", "raarmenian", "rabengali", "racute", 
    "radeva", "radical", "radicalex", "radoverssquare", 
    "radoverssquaredsquare", "radsquare", "rafe", "rafehebrew", 
    "ragujarati", "ragurmukhi", "rahiragana", "rakatakana", 
    "rakatakanahalfwidth", "ralowerdiagonalbengali", 
    "ramiddlediagonalbengali", "ramshorn", "ratio", "rbopomofo", 
    "rcaron", "rcedilla", "rcircle", "rcommaaccent", "rdblgrave", 
    "rdotaccent", "rdotbelow", "rdotbelowmacron", "referencemark", 
    "reflexsubset", "reflexsuperset", "registered", "registersans", 
    "registerserif", "reharabic", "reharmenian", "rehfinalarabic", 
    "rehiragana", "rekatakana", "rekatakanahalfwidth", "resh", 
    "reshdageshhebrew", "reshhebrew", "reversedtilde", "reviahebrew", 
    "reviamugrashhebrew", "revlogicalnot", "rfishhook", 
    "rfishhookreversed", "rhabengali", "rhadeva", "rho", "rhook", 
    "rhookturned", "rhookturnedsuperior", "rhosymbolgreek", 
    "rhotichookmod", "rieulacirclekorean", "rieulaparenkorean", 
    "rieulcirclekorean", "rieulhieuhkorean", "rieulkiyeokkorean", 
    "rieulkiyeoksioskorean", "rieulkorean", "rieulmieumkorean", 
    "rieulpansioskorean", "rieulparenkorean", "rieulphieuphkorean", 
    "rieulpieupkorean", "rieulpieupsioskorean", "rieulsioskorean", 
    "rieulthieuthkorean", "rieultikeutkorean", 
    "rieulyeorinhieuhkorean", "rightangle", "righttackbelowcmb", 
    "righttriangle", "rihiragana", "rikatakana", "rikatakanahalfwidth", 
    "ring", "ringbelowcmb", "ringcmb", "ringhalfleft", 
    "ringhalfleftarmenian", "ringhalfleftbelowcmb", 
    "ringhalfleftcentered", "ringhalfright", "ringhalfrightbelowcmb", 
    "ringhalfrightcentered", "rinvertedbreve", "rittorusquare", 
    "rlinebelow", "rlongleg", "rlonglegturned", "rmonospace", 
    "rohiragana", "rokatakana", "rokatakanahalfwidth", "roruathai", 
    "rparen", "rrabengali", "rradeva", "rragurmukhi", "rreharabic", 
    "rrehfinalarabic", "rrvocalicbengali", "rrvocalicdeva", 
    "rrvocalicgujarati", "rrvocalicvowelsignbengali", 
    "rrvocalicvowelsigndeva", "rrvocalicvowelsigngujarati", 
    "rsuperior", "rtblock", "rturned", "rturnedsuperior", "ruhiragana", 
    "rukatakana", "rukatakanahalfwidth", "rupeemarkbengali", 
    "rupeesignbengali", "rupiah", "ruthai", "rvocalicbengali", 
    "rvocalicdeva", "rvocalicgujarati", "rvocalicvowelsignbengali", 
    "rvocalicvowelsigndeva", "rvocalicvowelsigngujarati", "s", 
    "sabengali", "sacute", "sacutedotaccent", "sadarabic", "sadeva", 
    "sadfinalarabic", "sadinitialarabic", "sadmedialarabic", 
    "sagujarati", "sagurmukhi", "sahiragana", "sakatakana", 
    "sakatakanahalfwidth", "sallallahoualayhewasallamarabic", "samekh", 
    "samekhdagesh", "samekhdageshhebrew", "samekhhebrew", "saraaathai", 
    "saraaethai", "saraaimaimalaithai", "saraaimaimuanthai", 
    "saraamthai", "saraathai", "saraethai", "saraiileftthai", 
    "saraiithai", "saraileftthai", "saraithai", "saraothai", 
    "saraueeleftthai", "saraueethai", "saraueleftthai", "sarauethai", 
    "sarauthai", "sarauuthai", "sbopomofo", "scaron", 
    "scarondotaccent", "scedilla", "schwa", "schwacyrillic", 
    "schwadieresiscyrillic", "schwahook", "scircle", "scircumflex", 
    "scommaaccent", "sdotaccent", "sdotbelow", "sdotbelowdotaccent", 
    "seagullbelowcmb", "second", "secondtonechinese", "section", 
    "seenarabic", "seenfinalarabic", "seeninitialarabic", 
    "seenmedialarabic", "segol", "segol13", "segol1f", "segol2c", 
    "segolhebrew", "segolnarrowhebrew", "segolquarterhebrew", 
    "segoltahebrew", "segolwidehebrew", "seharmenian", "sehiragana", 
    "sekatakana", "sekatakanahalfwidth", "semicolon", 
    "semicolonarabic", "semicolonmonospace", "semicolonsmall", 
    "semivoicedmarkkana", "semivoicedmarkkanahalfwidth", "sentisquare", 
    "sentosquare", "seven", "sevenarabic", "sevenbengali", 
    "sevencircle", "sevencircleinversesansserif", "sevendeva", 
    "seveneighths", "sevengujarati", "sevengurmukhi", 
    "sevenhackarabic", "sevenhangzhou", "sevenideographicparen", 
    "seveninferior", "sevenmonospace", "sevenoldstyle", "sevenparen", 
    "sevenperiod", "sevenpersian", "sevenroman", "sevensuperior", 
    "seventeencircle", "seventeenparen", "seventeenperiod", 
    "seventhai", "sfthyphen", "shaarmenian", "shabengali", 
    "shacyrillic", "shaddaarabic", "shaddadammaarabic", 
    "shaddadammatanarabic", "shaddafathaarabic", "shaddakasraarabic", 
    "shaddakasratanarabic", "shade", "shadedark", "shadelight", 
    "shademedium", "shadeva", "shagujarati", "shagurmukhi", 
    "shalshelethebrew", "shbopomofo", "shchacyrillic", "sheenarabic", 
    "sheenfinalarabic", "sheeninitialarabic", "sheenmedialarabic", 
    "sheicoptic", "sheqel", "sheqelhebrew", "sheva", "sheva115", 
    "sheva15", "sheva22", "sheva2e", "shevahebrew", 
    "shevanarrowhebrew", "shevaquarterhebrew", "shevawidehebrew", 
    "shhacyrillic", "shimacoptic", "shin", "shindagesh", 
    "shindageshhebrew", "shindageshshindot", "shindageshshindothebrew", 
    "shindageshsindot", "shindageshsindothebrew", "shindothebrew", 
    "shinhebrew", "shinshindot", "shinshindothebrew", "shinsindot", 
    "shinsindothebrew", "shook", "sigma", "sigma1", "sigmafinal", 
    "sigmalunatesymbolgreek", "sihiragana", "sikatakana", 
    "sikatakanahalfwidth", "siluqhebrew", "siluqlefthebrew", "similar", 
    "sindothebrew", "siosacirclekorean", "siosaparenkorean", 
    "sioscieuckorean", "sioscirclekorean", "sioskiyeokkorean", 
    "sioskorean", "siosnieunkorean", "siosparenkorean", 
    "siospieupkorean", "siostikeutkorean", "six", "sixarabic", 
    "sixbengali", "sixcircle", "sixcircleinversesansserif", "sixdeva", 
    "sixgujarati", "sixgurmukhi", "sixhackarabic", "sixhangzhou", 
    "sixideographicparen", "sixinferior", "sixmonospace", 
    "sixoldstyle", "sixparen", "sixperiod", "sixpersian", "sixroman", 
    "sixsuperior", "sixteencircle", 
    "sixteencurrencydenominatorbengali", "sixteenparen", 
    "sixteenperiod", "sixthai", "slash", "slashmonospace", "slong", 
    "slongdotaccent", "smileface", "smonospace", "sofpasuqhebrew", 
    "softhyphen", "softsigncyrillic", "sohiragana", "sokatakana", 
    "sokatakanahalfwidth", "soliduslongoverlaycmb", 
    "solidusshortoverlaycmb", "sorusithai", "sosalathai", "sosothai", 
    "sosuathai", "space", "spacehackarabic", "spade", "spadesuitblack", 
    "spadesuitwhite", "sparen", "squarebelowcmb", "squarecc", 
    "squarecm", "squarediagonalcrosshatchfill", "squarehorizontalfill", 
    "squarekg", "squarekm", "squarekmcapital", "squareln", "squarelog", 
    "squaremg", "squaremil", "squaremm", "squaremsquared", 
    "squareorthogonalcrosshatchfill", 
    "squareupperlefttolowerrightfill", 
    "squareupperrighttolowerleftfill", "squareverticalfill", 
    "squarewhitewithsmallblack", "srsquare", "ssabengali", "ssadeva", 
    "ssagujarati", "ssangcieuckorean", "ssanghieuhkorean", 
    "ssangieungkorean", "ssangkiyeokkorean", "ssangnieunkorean", 
    "ssangpieupkorean", "ssangsioskorean", "ssangtikeutkorean", 
    "ssuperior", "sterling", "sterlingmonospace", 
    "strokelongoverlaycmb", "strokeshortoverlaycmb", "subset", 
    "subsetnotequal", "subsetorequal", "succeeds", "suchthat", 
    "suhiragana", "sukatakana", "sukatakanahalfwidth", "sukunarabic", 
    "summation", "sun", "superset", "supersetnotequal", 
    "supersetorequal", "svsquare", "syouwaerasquare", "t", "tabengali", 
    "tackdown", "tackleft", "tadeva", "tagujarati", "tagurmukhi", 
    "taharabic", "tahfinalarabic", "tahinitialarabic", "tahiragana", 
    "tahmedialarabic", "taisyouerasquare", "takatakana", 
    "takatakanahalfwidth", "tatweelarabic", "tau", "tav", "tavdages", 
    "tavdagesh", "tavdageshhebrew", "tavhebrew", "tbar", "tbopomofo", 
    "tcaron", "tccurl", "tcedilla", "tcheharabic", "tchehfinalarabic", 
    "tchehinitialarabic", "tchehmedialarabic", "tcircle", 
    "tcircumflexbelow", "tcommaaccent", "tdieresis", "tdotaccent", 
    "tdotbelow", "tecyrillic", "tedescendercyrillic", "teharabic", 
    "tehfinalarabic", "tehhahinitialarabic", "tehhahisolatedarabic", 
    "tehinitialarabic", "tehiragana", "tehjeeminitialarabic", 
    "tehjeemisolatedarabic", "tehmarbutaarabic", 
    "tehmarbutafinalarabic", "tehmedialarabic", "tehmeeminitialarabic", 
    "tehmeemisolatedarabic", "tehnoonfinalarabic", "tekatakana", 
    "tekatakanahalfwidth", "telephone", "telephoneblack", 
    "telishagedolahebrew", "telishaqetanahebrew", "tencircle", 
    "tenideographicparen", "tenparen", "tenperiod", "tenroman", "tesh", 
    "tet", "tetdagesh", "tetdageshhebrew", "tethebrew", 
    "tetsecyrillic", "tevirhebrew", "tevirlefthebrew", "thabengali", 
    "thadeva", "thagujarati", "thagurmukhi", "thalarabic", 
    "thalfinalarabic", "thanthakhatlowleftthai", 
    "thanthakhatlowrightthai", "thanthakhatthai", 
    "thanthakhatupperleftthai", "theharabic", "thehfinalarabic", 
    "thehinitialarabic", "thehmedialarabic", "thereexists", 
    "therefore", "theta", "theta1", "thetasymbolgreek", 
    "thieuthacirclekorean", "thieuthaparenkorean", 
    "thieuthcirclekorean", "thieuthkorean", "thieuthparenkorean", 
    "thirteencircle", "thirteenparen", "thirteenperiod", 
    "thonangmonthothai", "thook", "thophuthaothai", "thorn", 
    "thothahanthai", "thothanthai", "thothongthai", "thothungthai", 
    "thousandcyrillic", "thousandsseparatorarabic", 
    "thousandsseparatorpersian", "three", "threearabic", 
    "threebengali", "threecircle", "threecircleinversesansserif", 
    "threedeva", "threeeighths", "threegujarati", "threegurmukhi", 
    "threehackarabic", "threehangzhou", "threeideographicparen", 
    "threeinferior", "threemonospace", "threenumeratorbengali", 
    "threeoldstyle", "threeparen", "threeperiod", "threepersian", 
    "threequarters", "threequartersemdash", "threeroman", 
    "threesuperior", "threethai", "thzsquare", "tihiragana", 
    "tikatakana", "tikatakanahalfwidth", "tikeutacirclekorean", 
    "tikeutaparenkorean", "tikeutcirclekorean", "tikeutkorean", 
    "tikeutparenkorean", "tilde", "tildebelowcmb", "tildecmb", 
    "tildecomb", "tildedoublecmb", "tildeoperator", "tildeoverlaycmb", 
    "tildeverticalcmb", "timescircle", "tipehahebrew", 
    "tipehalefthebrew", "tippigurmukhi", "titlocyrilliccmb", 
    "tiwnarmenian", "tlinebelow", "tmonospace", "toarmenian", 
    "tohiragana", "tokatakana", "tokatakanahalfwidth", 
    "tonebarextrahighmod", "tonebarextralowmod", "tonebarhighmod", 
    "tonebarlowmod", "tonebarmidmod", "tonefive", "tonesix", "tonetwo", 
    "tonos", "tonsquare", "topatakthai", "tortoiseshellbracketleft", 
    "tortoiseshellbracketleftsmall", 
    "tortoiseshellbracketleftvertical", "tortoiseshellbracketright", 
    "tortoiseshellbracketrightsmall", 
    "tortoiseshellbracketrightvertical", "totaothai", "tpalatalhook", 
    "tparen", "trademark", "trademarksans", "trademarkserif", 
    "tretroflexhook", "triagdn", "triaglf", "triagrt", "triagup", "ts", 
    "tsadi", "tsadidagesh", "tsadidageshhebrew", "tsadihebrew", 
    "tsecyrillic", "tsere", "tsere12", "tsere1e", "tsere2b", 
    "tserehebrew", "tserenarrowhebrew", "tserequarterhebrew", 
    "tserewidehebrew", "tshecyrillic", "tsuperior", "ttabengali", 
    "ttadeva", "ttagujarati", "ttagurmukhi", "tteharabic", 
    "ttehfinalarabic", "ttehinitialarabic", "ttehmedialarabic", 
    "tthabengali", "tthadeva", "tthagujarati", "tthagurmukhi", 
    "tturned", "tuhiragana", "tukatakana", "tukatakanahalfwidth", 
    "tusmallhiragana", "tusmallkatakana", "tusmallkatakanahalfwidth", 
    "twelvecircle", "twelveparen", "twelveperiod", "twelveroman", 
    "twentycircle", "twentyhangzhou", "twentyparen", "twentyperiod", 
    "two", "twoarabic", "twobengali", "twocircle", 
    "twocircleinversesansserif", "twodeva", "twodotenleader", 
    "twodotleader", "twodotleadervertical", "twogujarati", 
    "twogurmukhi", "twohackarabic", "twohangzhou", 
    "twoideographicparen", "twoinferior", "twomonospace", 
    "twonumeratorbengali", "twooldstyle", "twoparen", "twoperiod", 
    "twopersian", "tworoman", "twostroke", "twosuperior", "twothai", 
    "twothirds", "u", "uacute", "ubar", "ubengali", "ubopomofo", 
    "ubreve", "ucaron", "ucircle", "ucircumflex", "ucircumflexbelow", 
    "ucyrillic", "udattadeva", "udblacute", "udblgrave", "udeva", 
    "udieresis", "udieresisacute", "udieresisbelow", "udieresiscaron", 
    "udieresiscyrillic", "udieresisgrave", "udieresismacron", 
    "udotbelow", "ugrave", "ugujarati", "ugurmukhi", "uhiragana", 
    "uhookabove", "uhorn", "uhornacute", "uhorndotbelow", "uhorngrave", 
    "uhornhookabove", "uhorntilde", "uhungarumlaut", 
    "uhungarumlautcyrillic", "uinvertedbreve", "ukatakana", 
    "ukatakanahalfwidth", "ukcyrillic", "ukorean", "umacron", 
    "umacroncyrillic", "umacrondieresis", "umatragurmukhi", 
    "umonospace", "underscore", "underscoredbl", "underscoremonospace", 
    "underscorevertical", "underscorewavy", "union", "universal", 
    "uogonek", "uparen", "upblock", "upperdothebrew", "upsilon", 
    "upsilondieresis", "upsilondieresistonos", "upsilonlatin", 
    "upsilontonos", "uptackbelowcmb", "uptackmod", "uragurmukhi", 
    "uring", "ushortcyrillic", "usmallhiragana", "usmallkatakana", 
    "usmallkatakanahalfwidth", "ustraightcyrillic", 
    "ustraightstrokecyrillic", "utilde", "utildeacute", "utildebelow", 
    "uubengali", "uudeva", "uugujarati", "uugurmukhi", 
    "uumatragurmukhi", "uuvowelsignbengali", "uuvowelsigndeva", 
    "uuvowelsigngujarati", "uvowelsignbengali", "uvowelsigndeva", 
    "uvowelsigngujarati", "v", "vadeva", "vagujarati", "vagurmukhi", 
    "vakatakana", "vav", "vavdagesh", "vavdagesh65", "vavdageshhebrew", 
    "vavhebrew", "vavholam", "vavholamhebrew", "vavvavhebrew", 
    "vavyodhebrew", "vcircle", "vdotbelow", "vecyrillic", "veharabic", 
    "vehfinalarabic", "vehinitialarabic", "vehmedialarabic", 
    "vekatakana", "venus", "verticalbar", "verticallineabovecmb", 
    "verticallinebelowcmb", "verticallinelowmod", "verticallinemod", 
    "vewarmenian", "vhook", "vikatakana", "viramabengali", 
    "viramadeva", "viramagujarati", "visargabengali", "visargadeva", 
    "visargagujarati", "vmonospace", "voarmenian", 
    "voicediterationhiragana", "voicediterationkatakana", 
    "voicedmarkkana", "voicedmarkkanahalfwidth", "vokatakana", 
    "vparen", "vtilde", "vturned", "vuhiragana", "vukatakana", "w", 
    "wacute", "waekorean", "wahiragana", "wakatakana", 
    "wakatakanahalfwidth", "wakorean", "wasmallhiragana", 
    "wasmallkatakana", "wattosquare", "wavedash", 
    "wavyunderscorevertical", "wawarabic", "wawfinalarabic", 
    "wawhamzaabovearabic", "wawhamzaabovefinalarabic", "wbsquare", 
    "wcircle", "wcircumflex", "wdieresis", "wdotaccent", "wdotbelow", 
    "wehiragana", "weierstrass", "wekatakana", "wekorean", "weokorean", 
    "wgrave", "whitebullet", "whitecircle", "whitecircleinverse", 
    "whitecornerbracketleft", "whitecornerbracketleftvertical", 
    "whitecornerbracketright", "whitecornerbracketrightvertical", 
    "whitediamond", "whitediamondcontainingblacksmalldiamond", 
    "whitedownpointingsmalltriangle", "whitedownpointingtriangle", 
    "whiteleftpointingsmalltriangle", "whiteleftpointingtriangle", 
    "whitelenticularbracketleft", "whitelenticularbracketright", 
    "whiterightpointingsmalltriangle", "whiterightpointingtriangle", 
    "whitesmallsquare", "whitesmilingface", "whitesquare", "whitestar", 
    "whitetelephone", "whitetortoiseshellbracketleft", 
    "whitetortoiseshellbracketright", "whiteuppointingsmalltriangle", 
    "whiteuppointingtriangle", "wihiragana", "wikatakana", "wikorean", 
    "wmonospace", "wohiragana", "wokatakana", "wokatakanahalfwidth", 
    "won", "wonmonospace", "wowaenthai", "wparen", "wring", 
    "wsuperior", "wturned", "wynn", "x", "xabovecmb", "xbopomofo", 
    "xcircle", "xdieresis", "xdotaccent", "xeharmenian", "xi", 
    "xmonospace", "xparen", "xsuperior", "y", "yaadosquare", 
    "yabengali", "yacute", "yadeva", "yaekorean", "yagujarati", 
    "yagurmukhi", "yahiragana", "yakatakana", "yakatakanahalfwidth", 
    "yakorean", "yamakkanthai", "yasmallhiragana", "yasmallkatakana", 
    "yasmallkatakanahalfwidth", "yatcyrillic", "ycircle", 
    "ycircumflex", "ydieresis", "ydotaccent", "ydotbelow", "yeharabic", 
    "yehbarreearabic", "yehbarreefinalarabic", "yehfinalarabic", 
    "yehhamzaabovearabic", "yehhamzaabovefinalarabic", 
    "yehhamzaaboveinitialarabic", "yehhamzaabovemedialarabic", 
    "yehinitialarabic", "yehmedialarabic", "yehmeeminitialarabic", 
    "yehmeemisolatedarabic", "yehnoonfinalarabic", 
    "yehthreedotsbelowarabic", "yekorean", "yen", "yenmonospace", 
    "yeokorean", "yeorinhieuhkorean", "yerahbenyomohebrew", 
    "yerahbenyomolefthebrew", "yericyrillic", "yerudieresiscyrillic", 
    "yesieungkorean", "yesieungpansioskorean", "yesieungsioskorean", 
    "yetivhebrew", "ygrave", "yhook", "yhookabove", "yiarmenian", 
    "yicyrillic", "yikorean", "yinyang", "yiwnarmenian", "ymonospace", 
    "yod", "yoddagesh", "yoddageshhebrew", "yodhebrew", "yodyodhebrew", 
    "yodyodpatahhebrew", "yohiragana", "yoikorean", "yokatakana", 
    "yokatakanahalfwidth", "yokorean", "yosmallhiragana", 
    "yosmallkatakana", "yosmallkatakanahalfwidth", "yotgreek", 
    "yoyaekorean", "yoyakorean", "yoyakthai", "yoyingthai", "yparen", 
    "ypogegrammeni", "ypogegrammenigreekcmb", "yr", "yring", 
    "ysuperior", "ytilde", "yturned", "yuhiragana", "yuikorean", 
    "yukatakana", "yukatakanahalfwidth", "yukorean", "yusbigcyrillic", 
    "yusbigiotifiedcyrillic", "yuslittlecyrillic", 
    "yuslittleiotifiedcyrillic", "yusmallhiragana", "yusmallkatakana", 
    "yusmallkatakanahalfwidth", "yuyekorean", "yuyeokorean", 
    "yyabengali", "yyadeva", "z", "zaarmenian", "zacute", "zadeva", 
    "zagurmukhi", "zaharabic", "zahfinalarabic", "zahinitialarabic", 
    "zahiragana", "zahmedialarabic", "zainarabic", "zainfinalarabic", 
    "zakatakana", "zaqefgadolhebrew", "zaqefqatanhebrew", 
    "zarqahebrew", "zayin", "zayindagesh", "zayindageshhebrew", 
    "zayinhebrew", "zbopomofo", "zcaron", "zcircle", "zcircumflex", 
    "zcurl", "zdot", "zdotaccent", "zdotbelow", "zecyrillic", 
    "zedescendercyrillic", "zedieresiscyrillic", "zehiragana", 
    "zekatakana", "zero", "zeroarabic", "zerobengali", "zerodeva", 
    "zerogujarati", "zerogurmukhi", "zerohackarabic", "zeroinferior", 
    "zeromonospace", "zerooldstyle", "zeropersian", "zerosuperior", 
    "zerothai", "zerowidthjoiner", "zerowidthnonjoiner", 
    "zerowidthspace", "zeta", "zhbopomofo", "zhearmenian", 
    "zhebrevecyrillic", "zhecyrillic", "zhedescendercyrillic", 
    "zhedieresiscyrillic", "zihiragana", "zikatakana", "zinorhebrew", 
    "zlinebelow", "zmonospace", "zohiragana", "zokatakana", "zparen", 
    "zretroflexhook", "zstroke", "zuhiragana", "zukatakana", 
};

char const **extraglyphs = NULL;
glyph nextglyph = lenof(ps_glyphs_alphabetic);
tree234 *extrabyname = NULL;

char const *glyph_extern(glyph glyph) {
    if (glyph == NOGLYPH) return ".notdef";
    if (glyph < lenof(ps_glyphs_alphabetic))
	return ps_glyphs_alphabetic[glyph];
    else
	return extraglyphs[glyph - lenof(ps_glyphs_alphabetic)];
}

static int glyphcmp(void *a, void *b) {
    glyph ga = *(glyph *)a, gb = *(glyph *)b;
    return strcmp(glyph_extern(ga), glyph_extern(gb));
}

static int glyphcmp_search(void *a, void *b) {
    glyph gb = *(glyph *)b;
    return strcmp(a, glyph_extern(gb));
}

glyph glyph_intern(char const *glyphname) {
    int i, j, k, c;
    glyph *gp;

    i = -1;
    j = lenof(ps_glyphs_alphabetic);
    while (j-i > 1) {
	k = (i + j) / 2;
	c = strcmp(glyphname, ps_glyphs_alphabetic[k]);

	if (c == 0)
	    return k;
	else if (c < 0)
	    j = k;
	else
	    i = k;
    }
    /* Non-standard glyph.  We may need to add it to our tree. */
    if (extrabyname == NULL)
	extrabyname = newtree234(glyphcmp);
    gp = find234(extrabyname, (void *)glyphname, glyphcmp_search);
    if (gp) {
	k = *gp;
    } else {
	extraglyphs = sresize(extraglyphs, nextglyph, char const *);
	k = nextglyph++;
	extraglyphs[k - lenof(ps_glyphs_alphabetic)] = dupstr(glyphname);
	gp = snew(glyph);
	*gp = k;
	add234(extrabyname, gp);
    }
    return k;
}

/* ----------------------------------------------------------------------
 * Mapping between PS character names (/aacute, /zcaron etc) and
 * Unicode code points.
 * 
 * Generated from the Adobe Glyph List at
 * 
 *   http://partners.adobe.com/public/developer/en/opentype/glyphlist.txt
 * 
 * by another Perl/sh fragment.

grep '^[^#;][^;]*;[^ ][^ ][^ ][^ ]$' glyphlist.txt | sort -t\; -k1,1 | \
    cut -f2 -d\; | perl -ne 'chomp; print "0x$_, "' | \
    fold -s -w68 | sed 's/^/    /'; echo

 */
static const unsigned short ps_codes_alphabetic[] = {
    0xFFFF, 0xFFFF, 0x0041, 0x00C6, 0x01FC, 0x01E2, 0xF7E6, 0x00C1, 
    0xF7E1, 0x0102, 0x1EAE, 0x04D0, 0x1EB6, 0x1EB0, 0x1EB2, 0x1EB4, 
    0x01CD, 0x24B6, 0x00C2, 0x1EA4, 0x1EAC, 0x1EA6, 0x1EA8, 0xF7E2, 
    0x1EAA, 0xF6C9, 0xF7B4, 0x0410, 0x0200, 0x00C4, 0x04D2, 0x01DE, 
    0xF7E4, 0x1EA0, 0x01E0, 0x00C0, 0xF7E0, 0x1EA2, 0x04D4, 0x0202, 
    0x0391, 0x0386, 0x0100, 0xFF21, 0x0104, 0x00C5, 0x01FA, 0x1E00, 
    0xF7E5, 0xF761, 0x00C3, 0xF7E3, 0x0531, 0x0042, 0x24B7, 0x1E02, 
    0x1E04, 0x0411, 0x0532, 0x0392, 0x0181, 0x1E06, 0xFF22, 0xF6F4, 
    0xF762, 0x0182, 0x0043, 0x053E, 0x0106, 0xF6CA, 0xF6F5, 0x010C, 
    0x00C7, 0x1E08, 0xF7E7, 0x24B8, 0x0108, 0x010A, 0x010A, 0xF7B8, 
    0x0549, 0x04BC, 0x0427, 0x04BE, 0x04B6, 0x04F4, 0x0543, 0x04CB, 
    0x04B8, 0x03A7, 0x0187, 0xF6F6, 0xFF23, 0x0551, 0xF763, 0x0044, 
    0x01F1, 0x01C4, 0x0534, 0x0189, 0x010E, 0x1E10, 0x24B9, 0x1E12, 
    0x0110, 0x1E0A, 0x1E0C, 0x0414, 0x03EE, 0x2206, 0x0394, 0x018A, 
    0xF6CB, 0xF6CC, 0xF6CD, 0xF7A8, 0x03DC, 0x0402, 0x1E0E, 0xFF24, 
    0xF6F7, 0x0110, 0xF764, 0x018B, 0x01F2, 0x01C5, 0x04E0, 0x0405, 
    0x040F, 0x0045, 0x00C9, 0xF7E9, 0x0114, 0x011A, 0x1E1C, 0x0535, 
    0x24BA, 0x00CA, 0x1EBE, 0x1E18, 0x1EC6, 0x1EC0, 0x1EC2, 0xF7EA, 
    0x1EC4, 0x0404, 0x0204, 0x00CB, 0xF7EB, 0x0116, 0x0116, 0x1EB8, 
    0x0424, 0x00C8, 0xF7E8, 0x0537, 0x1EBA, 0x2167, 0x0206, 0x0464, 
    0x041B, 0x216A, 0x0112, 0x1E16, 0x1E14, 0x041C, 0xFF25, 0x041D, 
    0x04A2, 0x014A, 0x04A4, 0x04C7, 0x0118, 0x0190, 0x0395, 0x0388, 
    0x0420, 0x018E, 0x042D, 0x0421, 0x04AA, 0x01A9, 0xF765, 0x0397, 
    0x0538, 0x0389, 0x00D0, 0xF7F0, 0x1EBC, 0x1E1A, 0x20AC, 0x01B7, 
    0x01EE, 0x01B8, 0x0046, 0x24BB, 0x1E1E, 0x0556, 0x03E4, 0x0191, 
    0x0472, 0x2164, 0xFF26, 0x2163, 0xF766, 0x0047, 0x3387, 0x01F4, 
    0x0393, 0x0194, 0x03EA, 0x011E, 0x01E6, 0x0122, 0x24BC, 0x011C, 
    0x0122, 0x0120, 0x0120, 0x0413, 0x0542, 0x0494, 0x0492, 0x0490, 
    0x0193, 0x0533, 0x0403, 0x1E20, 0xFF27, 0xF6CE, 0xF760, 0xF767, 
    0x029B, 0x01E4, 0x0048, 0x25CF, 0x25AA, 0x25AB, 0x25A1, 0x33CB, 
    0x04A8, 0x04B2, 0x042A, 0x0126, 0x1E2A, 0x1E28, 0x24BD, 0x0124, 
    0x1E26, 0x1E22, 0x1E24, 0xFF28, 0x0540, 0x03E8, 0xF768, 0xF6CF, 
    0xF6F8, 0x3390, 0x0049, 0x042F, 0x0132, 0x042E, 0x00CD, 0xF7ED, 
    0x012C, 0x01CF, 0x24BE, 0x00CE, 0xF7EE, 0x0406, 0x0208, 0x00CF, 
    0x1E2E, 0x04E4, 0xF7EF, 0x0130, 0x0130, 0x1ECA, 0x04D6, 0x0415, 
    0x2111, 0x00CC, 0xF7EC, 0x1EC8, 0x0418, 0x020A, 0x0419, 0x012A, 
    0x04E2, 0xFF29, 0x053B, 0x0401, 0x012E, 0x0399, 0x0196, 0x03AA, 
    0x038A, 0xF769, 0x0197, 0x0128, 0x1E2C, 0x0474, 0x0476, 0x004A, 
    0x0541, 0x24BF, 0x0134, 0x0408, 0x054B, 0xFF2A, 0xF76A, 0x004B, 
    0x3385, 0x33CD, 0x04A0, 0x1E30, 0x041A, 0x049A, 0x04C3, 0x039A, 
    0x049E, 0x049C, 0x01E8, 0x0136, 0x24C0, 0x0136, 0x1E32, 0x0554, 
    0x053F, 0x0425, 0x03E6, 0x0198, 0x040C, 0x1E34, 0xFF2B, 0x0480, 
    0x03DE, 0x046E, 0xF76B, 0x004C, 0x01C7, 0xF6BF, 0x0139, 0x039B, 
    0x013D, 0x013B, 0x24C1, 0x1E3C, 0x013B, 0x013F, 0x013F, 0x1E36, 
    0x1E38, 0x053C, 0x01C8, 0x0409, 0x1E3A, 0xFF2C, 0x0141, 0xF6F9, 
    0xF76C, 0x004D, 0x3386, 0xF6D0, 0xF7AF, 0x1E3E, 0x24C2, 0x1E40, 
    0x1E42, 0x0544, 0xFF2D, 0xF76D, 0x019C, 0x039C, 0x004E, 0x01CA, 
    0x0143, 0x0147, 0x0145, 0x24C3, 0x1E4A, 0x0145, 0x1E44, 0x1E46, 
    0x019D, 0x2168, 0x01CB, 0x040A, 0x1E48, 0xFF2E, 0x0546, 0xF76E, 
    0x00D1, 0xF7F1, 0x039D, 0x004F, 0x0152, 0xF6FA, 0x00D3, 0xF7F3, 
    0x04E8, 0x04EA, 0x014E, 0x01D1, 0x019F, 0x24C4, 0x00D4, 0x1ED0, 
    0x1ED8, 0x1ED2, 0x1ED4, 0xF7F4, 0x1ED6, 0x041E, 0x0150, 0x020C, 
    0x00D6, 0x04E6, 0xF7F6, 0x1ECC, 0xF6FB, 0x00D2, 0xF7F2, 0x0555, 
    0x2126, 0x1ECE, 0x01A0, 0x1EDA, 0x1EE2, 0x1EDC, 0x1EDE, 0x1EE0, 
    0x0150, 0x01A2, 0x020E, 0x014C, 0x1E52, 0x1E50, 0x2126, 0x0460, 
    0x03A9, 0x047A, 0x047C, 0x038F, 0x039F, 0x038C, 0xFF2F, 0x2160, 
    0x01EA, 0x01EC, 0x0186, 0x00D8, 0x01FE, 0xF7F8, 0xF76F, 0x01FE, 
    0x047E, 0x00D5, 0x1E4C, 0x1E4E, 0xF7F5, 0x0050, 0x1E54, 0x24C5, 
    0x1E56, 0x041F, 0x054A, 0x04A6, 0x03A6, 0x01A4, 0x03A0, 0x0553, 
    0xFF30, 0x03A8, 0x0470, 0xF770, 0x0051, 0x24C6, 0xFF31, 0xF771, 
    0x0052, 0x054C, 0x0154, 0x0158, 0x0156, 0x24C7, 0x0156, 0x0210, 
    0x1E58, 0x1E5A, 0x1E5C, 0x0550, 0x211C, 0x03A1, 0xF6FC, 0x0212, 
    0x1E5E, 0xFF32, 0xF772, 0x0281, 0x02B6, 0x0053, 0x250C, 0x2514, 
    0x2510, 0x2518, 0x253C, 0x252C, 0x2534, 0x251C, 0x2524, 0x2500, 
    0x2502, 0x2561, 0x2562, 0x2556, 0x2555, 0x2563, 0x2551, 0x2557, 
    0x255D, 0x255C, 0x255B, 0x255E, 0x255F, 0x255A, 0x2554, 0x2569, 
    0x2566, 0x2560, 0x2550, 0x256C, 0x2567, 0x2568, 0x2564, 0x2565, 
    0x2559, 0x2558, 0x2552, 0x2553, 0x256B, 0x256A, 0x015A, 0x1E64, 
    0x03E0, 0x0160, 0x1E66, 0xF6FD, 0x015E, 0x018F, 0x04D8, 0x04DA, 
    0x24C8, 0x015C, 0x0218, 0x1E60, 0x1E62, 0x1E68, 0x054D, 0x2166, 
    0x0547, 0x0428, 0x0429, 0x03E2, 0x04BA, 0x03EC, 0x03A3, 0x2165, 
    0xFF33, 0x042C, 0xF773, 0x03DA, 0x0054, 0x03A4, 0x0166, 0x0164, 
    0x0162, 0x24C9, 0x1E70, 0x0162, 0x1E6A, 0x1E6C, 0x0422, 0x04AC, 
    0x2169, 0x04B4, 0x0398, 0x01AC, 0x00DE, 0xF7FE, 0x2162, 0xF6FE, 
    0x054F, 0x1E6E, 0xFF34, 0x0539, 0x01BC, 0x0184, 0x01A7, 0x01AE, 
    0x0426, 0x040B, 0xF774, 0x216B, 0x2161, 0x0055, 0x00DA, 0xF7FA, 
    0x016C, 0x01D3, 0x24CA, 0x00DB, 0x1E76, 0xF7FB, 0x0423, 0x0170, 
    0x0214, 0x00DC, 0x01D7, 0x1E72, 0x01D9, 0x04F0, 0x01DB, 0x01D5, 
    0xF7FC, 0x1EE4, 0x00D9, 0xF7F9, 0x1EE6, 0x01AF, 0x1EE8, 0x1EF0, 
    0x1EEA, 0x1EEC, 0x1EEE, 0x0170, 0x04F2, 0x0216, 0x0478, 0x016A, 
    0x04EE, 0x1E7A, 0xFF35, 0x0172, 0x03A5, 0x03D2, 0x03D3, 0x01B1, 
    0x03AB, 0x03D4, 0x03D2, 0x038E, 0x016E, 0x040E, 0xF775, 0x04AE, 
    0x04B0, 0x0168, 0x1E78, 0x1E74, 0x0056, 0x24CB, 0x1E7E, 0x0412, 
    0x054E, 0x01B2, 0xFF36, 0x0548, 0xF776, 0x1E7C, 0x0057, 0x1E82, 
    0x24CC, 0x0174, 0x1E84, 0x1E86, 0x1E88, 0x1E80, 0xFF37, 0xF777, 
    0x0058, 0x24CD, 0x1E8C, 0x1E8A, 0x053D, 0x039E, 0xFF38, 0xF778, 
    0x0059, 0x00DD, 0xF7FD, 0x0462, 0x24CE, 0x0176, 0x0178, 0xF7FF, 
    0x1E8E, 0x1EF4, 0x042B, 0x04F8, 0x1EF2, 0x01B3, 0x1EF6, 0x0545, 
    0x0407, 0x0552, 0xFF39, 0xF779, 0x1EF8, 0x046A, 0x046C, 0x0466, 
    0x0468, 0x005A, 0x0536, 0x0179, 0x017D, 0xF6FF, 0x24CF, 0x1E90, 
    0x017B, 0x017B, 0x1E92, 0x0417, 0x0498, 0x04DE, 0x0396, 0x053A, 
    0x04C1, 0x0416, 0x0496, 0x04DC, 0x1E94, 0xFF3A, 0xF77A, 0x01B5, 
    0x0061, 0x0986, 0x00E1, 0x0906, 0x0A86, 0x0A06, 0x0A3E, 0x3303, 
    0x09BE, 0x093E, 0x0ABE, 0x055F, 0x0970, 0x0985, 0x311A, 0x0103, 
    0x1EAF, 0x04D1, 0x1EB7, 0x1EB1, 0x1EB3, 0x1EB5, 0x01CE, 0x24D0, 
    0x00E2, 0x1EA5, 0x1EAD, 0x1EA7, 0x1EA9, 0x1EAB, 0x00B4, 0x0317, 
    0x0301, 0x0301, 0x0954, 0x02CF, 0x0341, 0x0430, 0x0201, 0x0A71, 
    0x0905, 0x00E4, 0x04D3, 0x01DF, 0x1EA1, 0x01E1, 0x00E6, 0x01FD, 
    0x3150, 0x01E3, 0x2015, 0x20A4, 0x0410, 0x0411, 0x0412, 0x0413, 
    0x0414, 0x0415, 0x0401, 0x0416, 0x0417, 0x0418, 0x0419, 0x041A, 
    0x041B, 0x041C, 0x041D, 0x041E, 0x041F, 0x0420, 0x0421, 0x0422, 
    0x0423, 0x0424, 0x0425, 0x0426, 0x0427, 0x0428, 0x0429, 0x042A, 
    0x042B, 0x042C, 0x042D, 0x042E, 0x042F, 0x0490, 0x0402, 0x0403, 
    0x0404, 0x0405, 0x0406, 0x0407, 0x0408, 0x0409, 0x040A, 0x040B, 
    0x040C, 0x040E, 0xF6C4, 0xF6C5, 0x0430, 0x0431, 0x0432, 0x0433, 
    0x0434, 0x0435, 0x0451, 0x0436, 0x0437, 0x0438, 0x0439, 0x043A, 
    0x043B, 0x043C, 0x043D, 0x043E, 0x043F, 0x0440, 0x0441, 0x0442, 
    0x0443, 0x0444, 0x0445, 0x0446, 0x0447, 0x0448, 0x0449, 0x044A, 
    0x044B, 0x044C, 0x044D, 0x044E, 0x044F, 0x0491, 0x0452, 0x0453, 
    0x0454, 0x0455, 0x0456, 0x0457, 0x0458, 0x0459, 0x045A, 0x045B, 
    0x045C, 0x045E, 0x040F, 0x0462, 0x0472, 0x0474, 0xF6C6, 0x045F, 
    0x0463, 0x0473, 0x0475, 0xF6C7, 0xF6C8, 0x04D9, 0x200E, 0x200F, 
    0x200D, 0x066A, 0x060C, 0x0660, 0x0661, 0x0662, 0x0663, 0x0664, 
    0x0665, 0x0666, 0x0667, 0x0668, 0x0669, 0x061B, 0x061F, 0x0621, 
    0x0622, 0x0623, 0x0624, 0x0625, 0x0626, 0x0627, 0x0628, 0x0629, 
    0x062A, 0x062B, 0x062C, 0x062D, 0x062E, 0x062F, 0x0630, 0x0631, 
    0x0632, 0x0633, 0x0634, 0x0635, 0x0636, 0x0637, 0x0638, 0x0639, 
    0x063A, 0x0640, 0x0641, 0x0642, 0x0643, 0x0644, 0x0645, 0x0646, 
    0x0648, 0x0649, 0x064A, 0x064B, 0x064C, 0x064D, 0x064E, 0x064F, 
    0x0650, 0x0651, 0x0652, 0x0647, 0x06A4, 0x067E, 0x0686, 0x0698, 
    0x06AF, 0x0679, 0x0688, 0x0691, 0x06BA, 0x06D2, 0x06D5, 0x20AA, 
    0x05BE, 0x05C3, 0x05D0, 0x05D1, 0x05D2, 0x05D3, 0x05D4, 0x05D5, 
    0x05D6, 0x05D7, 0x05D8, 0x05D9, 0x05DA, 0x05DB, 0x05DC, 0x05DD, 
    0x05DE, 0x05DF, 0x05E0, 0x05E1, 0x05E2, 0x05E3, 0x05E4, 0x05E5, 
    0x05E6, 0x05E7, 0x05E8, 0x05E9, 0x05EA, 0xFB2A, 0xFB2B, 0xFB4B, 
    0xFB1F, 0x05F0, 0x05F1, 0x05F2, 0xFB35, 0x05B4, 0x05B5, 0x05B6, 
    0x05BB, 0x05B8, 0x05B7, 0x05B0, 0x05B2, 0x05B1, 0x05B3, 0x05C2, 
    0x05C1, 0x05B9, 0x05BC, 0x05BD, 0x05BF, 0x05C0, 0x02BC, 0x2105, 
    0x2113, 0x2116, 0x202C, 0x202D, 0x202E, 0x200C, 0x066D, 0x02BD, 
    0x00E0, 0x0A85, 0x0A05, 0x3042, 0x1EA3, 0x0990, 0x311E, 0x0910, 
    0x04D5, 0x0A90, 0x0A10, 0x0A48, 0x0639, 0xFECA, 0xFECB, 0xFECC, 
    0x0203, 0x09C8, 0x0948, 0x0AC8, 0x30A2, 0xFF71, 0x314F, 0x05D0, 
    0x0627, 0xFB30, 0xFE8E, 0x0623, 0xFE84, 0x0625, 0xFE88, 0x05D0, 
    0xFB4F, 0x0622, 0xFE82, 0x0649, 0xFEF0, 0xFEF3, 0xFEF4, 0xFB2E, 
    0xFB2F, 0x2135, 0x224C, 0x03B1, 0x03AC, 0x0101, 0xFF41, 0x0026, 
    0xFF06, 0xF726, 0x33C2, 0x3122, 0x3124, 0x0E5A, 0x2220, 0x3008, 
    0xFE3F, 0x3009, 0xFE40, 0x2329, 0x232A, 0x212B, 0x0387, 0x0952, 
    0x0982, 0x0902, 0x0A82, 0x0105, 0x3300, 0x249C, 0x055A, 0x02BC, 
    0xF8FF, 0x2250, 0x2248, 0x2252, 0x2245, 0x318E, 0x318D, 0x2312, 
    0x1E9A, 0x00E5, 0x01FB, 0x1E01, 0x2194, 0x21E3, 0x21E0, 0x21E2, 
    0x21E1, 0x21D4, 0x21D3, 0x21D0, 0x21D2, 0x21D1, 0x2193, 0x2199, 
    0x2198, 0x21E9, 0x02C5, 0x02C2, 0x02C3, 0x02C4, 0xF8E7, 0x2190, 
    0x21D0, 0x21CD, 0x21C6, 0x21E6, 0x2192, 0x21CF, 0x279E, 0x21C4, 
    0x21E8, 0x21E4, 0x21E5, 0x2191, 0x2195, 0x21A8, 0x21A8, 0x2196, 
    0x21C5, 0x2197, 0x21E7, 0xF8E6, 0x005E, 0xFF3E, 0x007E, 0xFF5E, 
    0x0251, 0x0252, 0x3041, 0x30A1, 0xFF67, 0x002A, 0x066D, 0x066D, 
    0x2217, 0xFF0A, 0xFE61, 0x2042, 0xF6E9, 0x2243, 0x0040, 0x00E3, 
    0xFF20, 0xFE6B, 0x0250, 0x0994, 0x3120, 0x0914, 0x0A94, 0x0A14, 
    0x09D7, 0x0A4C, 0x09CC, 0x094C, 0x0ACC, 0x093D, 0x0561, 0x05E2, 
    0xFB20, 0x05E2, 0x0062, 0x09AC, 0x005C, 0xFF3C, 0x092C, 0x0AAC, 
    0x0A2C, 0x3070, 0x0E3F, 0x30D0, 0x007C, 0xFF5C, 0x3105, 0x24D1, 
    0x1E03, 0x1E05, 0x266C, 0x2235, 0x0431, 0x0628, 0xFE90, 0xFE91, 
    0x3079, 0xFE92, 0xFC9F, 0xFC08, 0xFC6D, 0x30D9, 0x0562, 0x05D1, 
    0x03B2, 0x03D0, 0xFB31, 0xFB31, 0x05D1, 0xFB4C, 0x09AD, 0x092D, 
    0x0AAD, 0x0A2D, 0x0253, 0x3073, 0x30D3, 0x0298, 0x0A02, 0x3331, 
    0x25CF, 0x25C6, 0x25BC, 0x25C4, 0x25C0, 0x3010, 0xFE3B, 0x3011, 
    0xFE3C, 0x25E3, 0x25E2, 0x25AC, 0x25BA, 0x25B6, 0x25AA, 0x263B, 
    0x25A0, 0x2605, 0x25E4, 0x25E5, 0x25B4, 0x25B2, 0x2423, 0x1E07, 
    0x2588, 0xFF42, 0x0E1A, 0x307C, 0x30DC, 0x249D, 0x33C3, 0xF8F4, 
    0x007B, 0xF8F3, 0xF8F2, 0xFF5B, 0xFE5B, 0xF8F1, 0xFE37, 0x007D, 
    0xF8FE, 0xF8FD, 0xFF5D, 0xFE5C, 0xF8FC, 0xFE38, 0x005B, 0xF8F0, 
    0xF8EF, 0xFF3B, 0xF8EE, 0x005D, 0xF8FB, 0xF8FA, 0xFF3D, 0xF8F9, 
    0x02D8, 0x032E, 0x0306, 0x032F, 0x0311, 0x0361, 0x032A, 0x033A, 
    0x00A6, 0x0180, 0xF6EA, 0x0183, 0x3076, 0x30D6, 0x2022, 0x25D8, 
    0x2219, 0x25CE, 0x0063, 0x056E, 0x099A, 0x0107, 0x091A, 0x0A9A, 
    0x0A1A, 0x3388, 0x0981, 0x0310, 0x0901, 0x0A81, 0x21EA, 0x2105, 
    0x02C7, 0x032C, 0x030C, 0x21B5, 0x3118, 0x010D, 0x00E7, 0x1E09, 
    0x24D2, 0x0109, 0x0255, 0x010B, 0x010B, 0x33C5, 0x00B8, 0x0327, 
    0x00A2, 0x2103, 0xF6DF, 0xFFE0, 0xF7A2, 0xF6E0, 0x0579, 0x099B, 
    0x091B, 0x0A9B, 0x0A1B, 0x3114, 0x04BD, 0x2713, 0x0447, 0x04BF, 
    0x04B7, 0x04F5, 0x0573, 0x04CC, 0x04B9, 0x03C7, 0x3277, 0x3217, 
    0x3269, 0x314A, 0x3209, 0x0E0A, 0x0E08, 0x0E09, 0x0E0C, 0x0188, 
    0x3276, 0x3216, 0x3268, 0x3148, 0x3208, 0x321C, 0x25CB, 0x2297, 
    0x2299, 0x2295, 0x3036, 0x25D0, 0x25D1, 0x02C6, 0x032D, 0x0302, 
    0x2327, 0x01C2, 0x01C0, 0x01C1, 0x01C3, 0x2663, 0x2663, 0x2667, 
    0x33A4, 0xFF43, 0x33A0, 0x0581, 0x003A, 0x20A1, 0xFF1A, 0x20A1, 
    0xFE55, 0x02D1, 0x02D0, 0x002C, 0x0313, 0x0315, 0xF6C3, 0x060C, 
    0x055D, 0xF6E1, 0xFF0C, 0x0314, 0x02BD, 0xFE50, 0xF6E2, 0x0312, 
    0x02BB, 0x263C, 0x2245, 0x222E, 0x2303, 0x0006, 0x0007, 0x0008, 
    0x0018, 0x000D, 0x0011, 0x0012, 0x0013, 0x0014, 0x007F, 0x0010, 
    0x0019, 0x0005, 0x0004, 0x001B, 0x0017, 0x0003, 0x000C, 0x001C, 
    0x001D, 0x0009, 0x000A, 0x0015, 0x001E, 0x000F, 0x000E, 0x0002, 
    0x0001, 0x001A, 0x0016, 0x001F, 0x000B, 0x00A9, 0xF8E9, 0xF6D9, 
    0x300C, 0xFF62, 0xFE41, 0x300D, 0xFF63, 0xFE42, 0x337F, 0x33C7, 
    0x33C6, 0x249E, 0x20A2, 0x0297, 0x22CF, 0x22CE, 0x00A4, 0xF6D1, 
    0xF6D2, 0xF6D4, 0xF6D5, 0x0064, 0x0564, 0x09A6, 0x0636, 0x0926, 
    0xFEBE, 0xFEBF, 0xFEC0, 0x05BC, 0x05BC, 0x2020, 0x2021, 0x0AA6, 
    0x0A26, 0x3060, 0x30C0, 0x062F, 0x05D3, 0xFB33, 0xFB33, 0x05D3, 
    0xFEAA, 0x064F, 0x064F, 0x064C, 0x064C, 0x0964, 0x05A7, 0x05A7, 
    0x0485, 0xF6D3, 0x300A, 0xFE3D, 0x300B, 0xFE3E, 0x032B, 0x21D4, 
    0x21D2, 0x0965, 0xF6D6, 0x030F, 0x222C, 0x2017, 0x0333, 0x033F, 
    0x02BA, 0x2016, 0x030E, 0x3109, 0x33C8, 0x010F, 0x1E11, 0x24D3, 
    0x1E13, 0x0111, 0x09A1, 0x0921, 0x0AA1, 0x0A21, 0x0688, 0xFB89, 
    0x095C, 0x09A2, 0x0922, 0x0AA2, 0x0A22, 0x1E0B, 0x1E0D, 0x066B, 
    0x066B, 0x0434, 0x00B0, 0x05AD, 0x3067, 0x03EF, 0x30C7, 0x232B, 
    0x2326, 0x03B4, 0x018D, 0x09F8, 0x02A4, 0x09A7, 0x0927, 0x0AA7, 
    0x0A27, 0x0257, 0x0385, 0x0344, 0x2666, 0x2662, 0x00A8, 0xF6D7, 
    0x0324, 0x0308, 0xF6D8, 0x0385, 0x3062, 0x30C2, 0x3003, 0x00F7, 
    0x2223, 0x2215, 0x0452, 0x2593, 0x1E0F, 0x3397, 0x0111, 0xFF44, 
    0x2584, 0x0E0E, 0x0E14, 0x3069, 0x30C9, 0x0024, 0xF6E3, 0xFF04, 
    0xF724, 0xFE69, 0xF6E4, 0x20AB, 0x3326, 0x02D9, 0x0307, 0x0323, 
    0x0323, 0x30FB, 0x0131, 0xF6BE, 0x0284, 0x22C5, 0x25CC, 0xFB1F, 
    0xFB1F, 0x031E, 0x02D5, 0x249F, 0xF6EB, 0x0256, 0x018C, 0x3065, 
    0x30C5, 0x01F3, 0x02A3, 0x01C6, 0x02A5, 0x04E1, 0x0455, 0x045F, 
    0x0065, 0x00E9, 0x2641, 0x098F, 0x311C, 0x0115, 0x090D, 0x0A8D, 
    0x0945, 0x0AC5, 0x011B, 0x1E1D, 0x0565, 0x0587, 0x24D4, 0x00EA, 
    0x1EBF, 0x1E19, 0x1EC7, 0x1EC1, 0x1EC3, 0x1EC5, 0x0454, 0x0205, 
    0x090F, 0x00EB, 0x0117, 0x0117, 0x1EB9, 0x0A0F, 0x0A47, 0x0444, 
    0x00E8, 0x0A8F, 0x0567, 0x311D, 0x3048, 0x1EBB, 0x311F, 0x0038, 
    0x0668, 0x09EE, 0x2467, 0x2791, 0x096E, 0x2471, 0x2485, 0x2499, 
    0x0AEE, 0x0A6E, 0x0668, 0x3028, 0x266B, 0x3227, 0x2088, 0xFF18, 
    0xF738, 0x247B, 0x248F, 0x06F8, 0x2177, 0x2078, 0x0E58, 0x0207, 
    0x0465, 0x30A8, 0xFF74, 0x0A74, 0x3154, 0x043B, 0x2208, 0x246A, 
    0x247E, 0x2492, 0x217A, 0x2026, 0x22EE, 0x0113, 0x1E17, 0x1E15, 
    0x043C, 0x2014, 0xFE31, 0xFF45, 0x055B, 0x2205, 0x3123, 0x043D, 
    0x2013, 0xFE32, 0x04A3, 0x014B, 0x3125, 0x04A5, 0x04C8, 0x2002, 
    0x0119, 0x3153, 0x025B, 0x029A, 0x025C, 0x025E, 0x025D, 0x24A0, 
    0x03B5, 0x03AD, 0x003D, 0xFF1D, 0xFE66, 0x207C, 0x2261, 0x3126, 
    0x0440, 0x0258, 0x044D, 0x0441, 0x04AB, 0x0283, 0x0286, 0x090E, 
    0x0946, 0x01AA, 0x0285, 0x3047, 0x30A7, 0xFF6A, 0x212E, 0xF6EC, 
    0x03B7, 0x0568, 0x03AE, 0x00F0, 0x1EBD, 0x1E1B, 0x0591, 0x0591, 
    0x0591, 0x0591, 0x01DD, 0x3161, 0x20AC, 0x09C7, 0x0947, 0x0AC7, 
    0x0021, 0x055C, 0x203C, 0x00A1, 0xF7A1, 0xFF01, 0xF721, 0x2203, 
    0x0292, 0x01EF, 0x0293, 0x01B9, 0x01BA, 0x0066, 0x095E, 0x0A5E, 
    0x2109, 0x064E, 0x064E, 0x064B, 0x3108, 0x24D5, 0x1E1F, 0x0641, 
    0x0586, 0xFED2, 0xFED3, 0xFED4, 0x03E5, 0x2640, 0xFB00, 0xFB03, 
    0xFB04, 0xFB01, 0x246E, 0x2482, 0x2496, 0x2012, 0x25A0, 0x25AC, 
    0x05DA, 0xFB3A, 0xFB3A, 0x05DA, 0x05DD, 0x05DD, 0x05DF, 0x05DF, 
    0x05E3, 0x05E3, 0x05E5, 0x05E5, 0x02C9, 0x25C9, 0x0473, 0x0035, 
    0x0665, 0x09EB, 0x2464, 0x278E, 0x096B, 0x215D, 0x0AEB, 0x0A6B, 
    0x0665, 0x3025, 0x3224, 0x2085, 0xFF15, 0xF735, 0x2478, 0x248C, 
    0x06F5, 0x2174, 0x2075, 0x0E55, 0xFB02, 0x0192, 0xFF46, 0x3399, 
    0x0E1F, 0x0E1D, 0x0E4F, 0x2200, 0x0034, 0x0664, 0x09EA, 0x2463, 
    0x278D, 0x096A, 0x0AEA, 0x0A6A, 0x0664, 0x3024, 0x3223, 0x2084, 
    0xFF14, 0x09F7, 0xF734, 0x2477, 0x248B, 0x06F4, 0x2173, 0x2074, 
    0x246D, 0x2481, 0x2495, 0x0E54, 0x02CB, 0x24A1, 0x2044, 0x20A3, 
    0x0067, 0x0997, 0x01F5, 0x0917, 0x06AF, 0xFB93, 0xFB94, 0xFB95, 
    0x0A97, 0x0A17, 0x304C, 0x30AC, 0x03B3, 0x0263, 0x02E0, 0x03EB, 
    0x310D, 0x011F, 0x01E7, 0x0123, 0x24D6, 0x011D, 0x0123, 0x0121, 
    0x0121, 0x0433, 0x3052, 0x30B2, 0x2251, 0x059C, 0x05F3, 0x059D, 
    0x00DF, 0x059E, 0x05F4, 0x3013, 0x0998, 0x0572, 0x0918, 0x0A98, 
    0x0A18, 0x063A, 0xFECE, 0xFECF, 0xFED0, 0x0495, 0x0493, 0x0491, 
    0x095A, 0x0A5A, 0x0260, 0x3393, 0x304E, 0x30AE, 0x0563, 0x05D2, 
    0xFB32, 0xFB32, 0x05D2, 0x0453, 0x01BE, 0x0294, 0x0296, 0x02C0, 
    0x0295, 0x02C1, 0x02E4, 0x02A1, 0x02A2, 0x1E21, 0xFF47, 0x3054, 
    0x30B4, 0x24A2, 0x33AC, 0x2207, 0x0060, 0x0316, 0x0300, 0x0300, 
    0x0953, 0x02CE, 0xFF40, 0x0340, 0x003E, 0x2265, 0x22DB, 0xFF1E, 
    0x2273, 0x2277, 0x2267, 0xFE65, 0x0261, 0x01E5, 0x3050, 0x00AB, 
    0x00BB, 0x2039, 0x203A, 0x30B0, 0x3318, 0x33C9, 0x0068, 0x04A9, 
    0x06C1, 0x09B9, 0x04B3, 0x0939, 0x0AB9, 0x0A39, 0x062D, 0xFEA2, 
    0xFEA3, 0x306F, 0xFEA4, 0x332A, 0x30CF, 0xFF8A, 0x0A4D, 0x0621, 
    0x0621, 0x3164, 0x044A, 0x21BC, 0x21C0, 0x33CA, 0x05B2, 0x05B2, 
    0x05B2, 0x05B2, 0x05B2, 0x05B2, 0x05B2, 0x05B2, 0x05B3, 0x05B3, 
    0x05B3, 0x05B3, 0x05B3, 0x05B3, 0x05B3, 0x05B3, 0x05B1, 0x05B1, 
    0x05B1, 0x05B1, 0x05B1, 0x05B1, 0x05B1, 0x05B1, 0x0127, 0x310F, 
    0x1E2B, 0x1E29, 0x24D7, 0x0125, 0x1E27, 0x1E23, 0x1E25, 0x05D4, 
    0x2665, 0x2665, 0x2661, 0xFB34, 0xFB34, 0x06C1, 0x0647, 0x05D4, 
    0xFBA7, 0xFEEA, 0xFEEA, 0xFBA5, 0xFBA4, 0xFBA8, 0xFEEB, 0x3078, 
    0xFBA9, 0xFEEC, 0x337B, 0x30D8, 0xFF8D, 0x3336, 0x0267, 0x3339, 
    0x05D7, 0x05D7, 0x0266, 0x02B1, 0x327B, 0x321B, 0x326D, 0x314E, 
    0x320D, 0x3072, 0x30D2, 0xFF8B, 0x05B4, 0x05B4, 0x05B4, 0x05B4, 
    0x05B4, 0x05B4, 0x05B4, 0x05B4, 0x1E96, 0xFF48, 0x0570, 0x0E2B, 
    0x307B, 0x30DB, 0xFF8E, 0x05B9, 0x05B9, 0x05B9, 0x05B9, 0x05B9, 
    0x05B9, 0x05B9, 0x05B9, 0x0E2E, 0x0309, 0x0309, 0x0321, 0x0322, 
    0x3342, 0x03E9, 0x2015, 0x031B, 0x2668, 0x2302, 0x24A3, 0x02B0, 
    0x0265, 0x3075, 0x3333, 0x30D5, 0xFF8C, 0x02DD, 0x030B, 0x0195, 
    0x002D, 0xF6E5, 0xFF0D, 0xFE63, 0xF6E6, 0x2010, 0x0069, 0x00ED, 
    0x044F, 0x0987, 0x3127, 0x012D, 0x01D0, 0x24D8, 0x00EE, 0x0456, 
    0x0209, 0x328F, 0x328B, 0x323F, 0x323A, 0x32A5, 0x3006, 0x3001, 
    0xFF64, 0x3237, 0x32A3, 0x322F, 0x323D, 0x329D, 0x3240, 0x3296, 
    0x3236, 0x322B, 0x3232, 0x32A4, 0x3005, 0x3298, 0x3238, 0x32A7, 
    0x32A6, 0x32A9, 0x322E, 0x322A, 0x3234, 0x3002, 0x329E, 0x3243, 
    0x3239, 0x323E, 0x32A8, 0x3299, 0x3242, 0x3233, 0x3000, 0x3235, 
    0x3231, 0x323B, 0x3230, 0x323C, 0x322C, 0x322D, 0x3007, 0x328E, 
    0x328A, 0x3294, 0x3290, 0x328C, 0x328D, 0x0907, 0x00EF, 0x1E2F, 
    0x04E5, 0x1ECB, 0x04D7, 0x0435, 0x3275, 0x3215, 0x3267, 0x3147, 
    0x3207, 0x00EC, 0x0A87, 0x0A07, 0x3044, 0x1EC9, 0x0988, 0x0438, 
    0x0908, 0x0A88, 0x0A08, 0x0A40, 0x020B, 0x0439, 0x09C0, 0x0940, 
    0x0AC0, 0x0133, 0x30A4, 0xFF72, 0x3163, 0x02DC, 0x05AC, 0x012B, 
    0x04E3, 0x2253, 0x0A3F, 0xFF49, 0x2206, 0x221E, 0x056B, 0x222B, 
    0x2321, 0x2321, 0xF8F5, 0x2320, 0x2320, 0x2229, 0x3305, 0x25D8, 
    0x25D9, 0x263B, 0x0451, 0x012F, 0x03B9, 0x03CA, 0x0390, 0x0269, 
    0x03AF, 0x24A4, 0x0A72, 0x3043, 0x30A3, 0xFF68, 0x09FA, 0x0268, 
    0xF6ED, 0x309D, 0x30FD, 0x0129, 0x1E2D, 0x3129, 0x044E, 0x09BF, 
    0x093F, 0x0ABF, 0x0475, 0x0477, 0x006A, 0x0571, 0x099C, 0x091C, 
    0x0A9C, 0x0A1C, 0x3110, 0x01F0, 0x24D9, 0x0135, 0x029D, 0x025F, 
    0x0458, 0x062C, 0xFE9E, 0xFE9F, 0xFEA0, 0x0698, 0xFB8B, 0x099D, 
    0x091D, 0x0A9D, 0x0A1D, 0x057B, 0x3004, 0xFF4A, 0x24A5, 0x02B2, 
    0x006B, 0x04A1, 0x0995, 0x1E31, 0x043A, 0x049B, 0x0915, 0x05DB, 
    0x0643, 0xFB3B, 0xFB3B, 0xFEDA, 0x05DB, 0xFEDB, 0xFEDC, 0xFB4D, 
    0x0A95, 0x0A15, 0x304B, 0x04C4, 0x30AB, 0xFF76, 0x03BA, 0x03F0, 
    0x3171, 0x3184, 0x3178, 0x3179, 0x330D, 0x0640, 0x0640, 0x30F5, 
    0x3384, 0x0650, 0x064D, 0x049F, 0xFF70, 0x049D, 0x310E, 0x3389, 
    0x01E9, 0x0137, 0x24DA, 0x0137, 0x1E33, 0x0584, 0x3051, 0x30B1, 
    0xFF79, 0x056F, 0x30F6, 0x0138, 0x0996, 0x0445, 0x0916, 0x0A96, 
    0x0A16, 0x062E, 0xFEA6, 0xFEA7, 0xFEA8, 0x03E7, 0x0959, 0x0A59, 
    0x3278, 0x3218, 0x326A, 0x314B, 0x320A, 0x0E02, 0x0E05, 0x0E03, 
    0x0E04, 0x0E5B, 0x0199, 0x0E06, 0x3391, 0x304D, 0x30AD, 0xFF77, 
    0x3315, 0x3316, 0x3314, 0x326E, 0x320E, 0x3260, 0x3131, 0x3200, 
    0x3133, 0x045C, 0x1E35, 0x3398, 0x33A6, 0xFF4B, 0x33A2, 0x3053, 
    0x33C0, 0x0E01, 0x30B3, 0xFF7A, 0x331E, 0x0481, 0x327F, 0x0343, 
    0x24A6, 0x33AA, 0x046F, 0x33CF, 0x029E, 0x304F, 0x30AF, 0xFF78, 
    0x33B8, 0x33BE, 0x006C, 0x09B2, 0x013A, 0x0932, 0x0AB2, 0x0A32, 
    0x0E45, 0xFEFC, 0xFEF8, 0xFEF7, 0xFEFA, 0xFEF9, 0xFEFB, 0xFEF6, 
    0xFEF5, 0x0644, 0x03BB, 0x019B, 0x05DC, 0xFB3C, 0xFB3C, 0x05DC, 
    0xFEDE, 0xFCCA, 0xFEDF, 0xFCC9, 0xFCCB, 0xFDF2, 0xFEE0, 0xFD88, 
    0xFCCC, 0x25EF, 0x019A, 0x026C, 0x310C, 0x013E, 0x013C, 0x24DB, 
    0x1E3D, 0x013C, 0x0140, 0x0140, 0x1E37, 0x1E39, 0x031A, 0x0318, 
    0x003C, 0x2264, 0x22DA, 0xFF1C, 0x2272, 0x2276, 0x2266, 0xFE64, 
    0x026E, 0x258C, 0x026D, 0x20A4, 0x056C, 0x01C9, 0x0459, 0xF6C0, 
    0x0933, 0x0AB3, 0x1E3B, 0x0934, 0x09E1, 0x0961, 0x09E3, 0x0963, 
    0x026B, 0xFF4C, 0x33D0, 0x0E2C, 0x2227, 0x00AC, 0x2310, 0x2228, 
    0x0E25, 0x017F, 0xFE4E, 0x0332, 0xFE4D, 0x25CA, 0x24A7, 0x0142, 
    0x2113, 0xF6EE, 0x2591, 0x0E26, 0x098C, 0x090C, 0x09E2, 0x0962, 
    0x33D3, 0x006D, 0x09AE, 0x00AF, 0x0331, 0x0304, 0x02CD, 0xFFE3, 
    0x1E3F, 0x092E, 0x0AAE, 0x0A2E, 0x05A4, 0x05A4, 0x307E, 0xF895, 
    0xF894, 0x0E4B, 0xF893, 0xF88C, 0xF88B, 0x0E48, 0xF88A, 0xF884, 
    0x0E31, 0xF889, 0x0E47, 0xF88F, 0xF88E, 0x0E49, 0xF88D, 0xF892, 
    0xF891, 0x0E4A, 0xF890, 0x0E46, 0x30DE, 0xFF8F, 0x2642, 0x3347, 
    0x05BE, 0x2642, 0x05AF, 0x3383, 0x3107, 0x33D4, 0x24DC, 0x33A5, 
    0x1E41, 0x1E43, 0x0645, 0xFEE2, 0xFEE3, 0xFEE4, 0xFCD1, 0xFC48, 
    0x334D, 0x3081, 0x337E, 0x30E1, 0xFF92, 0x05DE, 0xFB3E, 0xFB3E, 
    0x05DE, 0x0574, 0x05A5, 0x05A6, 0x05A6, 0x05A5, 0x0271, 0x3392, 
    0xFF65, 0x00B7, 0x3272, 0x3212, 0x3264, 0x3141, 0x3170, 0x3204, 
    0x316E, 0x316F, 0x307F, 0x30DF, 0xFF90, 0x2212, 0x0320, 0x2296, 
    0x02D7, 0x2213, 0x2032, 0x334A, 0x3349, 0x0270, 0x3396, 0x33A3, 
    0xFF4D, 0x339F, 0x3082, 0x33C1, 0x30E2, 0xFF93, 0x33D6, 0x0E21, 
    0x33A7, 0x33A8, 0x24A8, 0x33AB, 0x33B3, 0xF6EF, 0x026F, 0x00B5, 
    0x00B5, 0x3382, 0x226B, 0x226A, 0x338C, 0x03BC, 0x338D, 0x3080, 
    0x30E0, 0xFF91, 0x3395, 0x00D7, 0x339B, 0x05A3, 0x05A3, 0x266A, 
    0x266B, 0x266D, 0x266F, 0x33B2, 0x33B6, 0x33BC, 0x33B9, 0x33B7, 
    0x33BF, 0x33BD, 0x006E, 0x09A8, 0x2207, 0x0144, 0x0928, 0x0AA8, 
    0x0A28, 0x306A, 0x30CA, 0xFF85, 0x0149, 0x3381, 0x310B, 0x00A0, 
    0x0148, 0x0146, 0x24DD, 0x1E4B, 0x0146, 0x1E45, 0x1E47, 0x306D, 
    0x30CD, 0xFF88, 0x20AA, 0x338B, 0x0999, 0x0919, 0x0A99, 0x0A19, 
    0x0E07, 0x3093, 0x0272, 0x0273, 0x326F, 0x320F, 0x3135, 0x3261, 
    0x3136, 0x3134, 0x3168, 0x3201, 0x3167, 0x3166, 0x306B, 0x30CB, 
    0xFF86, 0xF899, 0x0E4D, 0x0039, 0x0669, 0x09EF, 0x2468, 0x2792, 
    0x096F, 0x0AEF, 0x0A6F, 0x0669, 0x3029, 0x3228, 0x2089, 0xFF19, 
    0xF739, 0x247C, 0x2490, 0x06F9, 0x2178, 0x2079, 0x2472, 0x2486, 
    0x249A, 0x0E59, 0x01CC, 0x045A, 0x30F3, 0xFF9D, 0x019E, 0x1E49, 
    0xFF4E, 0x339A, 0x09A3, 0x0923, 0x0AA3, 0x0A23, 0x0929, 0x306E, 
    0x30CE, 0xFF89, 0x00A0, 0x0E13, 0xFFFF, 0x0E19, 0x0646, 0xFEE6, 
    0x06BA, 0xFB9F, 0xFEE7, 0xFCD2, 0xFC4B, 0xFEE8, 0xFCD5, 0xFC4E, 
    0xFC8D, 0x220C, 0x2209, 0x2209, 0x2260, 0x226F, 0x2271, 0x2279, 
    0x2262, 0x226E, 0x2270, 0x2226, 0x2280, 0x2284, 0x2281, 0x2285, 
    0x0576, 0x24A9, 0x33B1, 0x207F, 0x00F1, 0x03BD, 0x306C, 0x30CC, 
    0xFF87, 0x09BC, 0x093C, 0x0ABC, 0x0A3C, 0x0023, 0xFF03, 0xFE5F, 
    0x0374, 0x0375, 0x2116, 0x05E0, 0xFB40, 0xFB40, 0x05E0, 0x33B5, 
    0x33BB, 0x099E, 0x091E, 0x0A9E, 0x0A1E, 0x006F, 0x00F3, 0x0E2D, 
    0x0275, 0x04E9, 0x04EB, 0x0993, 0x311B, 0x014F, 0x0911, 0x0A91, 
    0x0949, 0x0AC9, 0x01D2, 0x24DE, 0x00F4, 0x1ED1, 0x1ED9, 0x1ED3, 
    0x1ED5, 0x1ED7, 0x043E, 0x0151, 0x020D, 0x0913, 0x00F6, 0x04E7, 
    0x1ECD, 0x0153, 0x315A, 0x02DB, 0x0328, 0x00F2, 0x0A93, 0x0585, 
    0x304A, 0x1ECF, 0x01A1, 0x1EDB, 0x1EE3, 0x1EDD, 0x1EDF, 0x1EE1, 
    0x0151, 0x01A3, 0x020F, 0x30AA, 0xFF75, 0x3157, 0x05AB, 0x014D, 
    0x1E53, 0x1E51, 0x0950, 0x03C9, 0x03D6, 0x0461, 0x0277, 0x047B, 
    0x047D, 0x03CE, 0x0AD0, 0x03BF, 0x03CC, 0xFF4F, 0x0031, 0x0661, 
    0x09E7, 0x2460, 0x278A, 0x0967, 0x2024, 0x215B, 0xF6DC, 0x0AE7, 
    0x0A67, 0x0661, 0x00BD, 0x3021, 0x3220, 0x2081, 0xFF11, 0x09F4, 
    0xF731, 0x2474, 0x2488, 0x06F1, 0x00BC, 0x2170, 0x00B9, 0x0E51, 
    0x2153, 0x01EB, 0x01ED, 0x0A13, 0x0A4B, 0x0254, 0x24AA, 0x25E6, 
    0x2325, 0x00AA, 0x00BA, 0x221F, 0x0912, 0x094A, 0x00F8, 0x01FF, 
    0x3049, 0x30A9, 0xFF6B, 0x01FF, 0xF6F0, 0x047F, 0x00F5, 0x1E4D, 
    0x1E4F, 0x3121, 0x203E, 0xFE4A, 0x0305, 0xFE49, 0xFE4C, 0xFE4B, 
    0x00AF, 0x09CB, 0x094B, 0x0ACB, 0x0070, 0x3380, 0x332B, 0x09AA, 
    0x1E55, 0x092A, 0x21DF, 0x21DE, 0x0AAA, 0x0A2A, 0x3071, 0x0E2F, 
    0x30D1, 0x0484, 0x04C0, 0x317F, 0x00B6, 0x2225, 0x0028, 0xFD3E, 
    0xF8ED, 0xF8EC, 0x208D, 0xFF08, 0xFE59, 0x207D, 0xF8EB, 0xFE35, 
    0x0029, 0xFD3F, 0xF8F8, 0xF8F7, 0x208E, 0xFF09, 0xFE5A, 0x207E, 
    0xF8F6, 0xFE36, 0x2202, 0x05C0, 0x0599, 0x33A9, 0x05B7, 0x05B7, 
    0x05B7, 0x05B7, 0x05B7, 0x05B7, 0x05B7, 0x05B7, 0x05A1, 0x3106, 
    0x24DF, 0x1E57, 0x05E4, 0x043F, 0xFB44, 0xFB44, 0x333B, 0xFB43, 
    0x067E, 0x057A, 0x05E4, 0xFB57, 0xFB58, 0x307A, 0xFB59, 0x30DA, 
    0x04A7, 0xFB4E, 0x0025, 0x066A, 0xFF05, 0xFE6A, 0x002E, 0x0589, 
    0x00B7, 0xFF61, 0xF6E7, 0xFF0E, 0xFE52, 0xF6E8, 0x0342, 0x22A5, 
    0x2030, 0x20A7, 0x338A, 0x09AB, 0x092B, 0x0AAB, 0x0A2B, 0x03C6, 
    0x03D5, 0x327A, 0x321A, 0x326C, 0x314D, 0x320C, 0x0278, 0x0E3A, 
    0x03D5, 0x01A5, 0x0E1E, 0x0E1C, 0x0E20, 0x03C0, 0x3273, 0x3213, 
    0x3176, 0x3265, 0x3172, 0x3142, 0x3205, 0x3174, 0x3144, 0x3175, 
    0x3177, 0x3173, 0x3074, 0x30D4, 0x03D6, 0x0583, 0x002B, 0x031F, 
    0x2295, 0x00B1, 0x02D6, 0xFF0B, 0xFE62, 0x207A, 0xFF50, 0x33D8, 
    0x307D, 0x261F, 0x261C, 0x261E, 0x261D, 0x30DD, 0x0E1B, 0x3012, 
    0x3020, 0x24AB, 0x227A, 0x211E, 0x02B9, 0x2035, 0x220F, 0x2305, 
    0x30FC, 0x2318, 0x2282, 0x2283, 0x2237, 0x221D, 0x03C8, 0x0471, 
    0x0486, 0x33B0, 0x3077, 0x30D7, 0x33B4, 0x33BA, 0x0071, 0x0958, 
    0x05A8, 0x0642, 0xFED6, 0xFED7, 0xFED8, 0x05B8, 0x05B8, 0x05B8, 
    0x05B8, 0x05B8, 0x05B8, 0x05B8, 0x05B8, 0x05B8, 0x05B8, 0x05B8, 
    0x05B8, 0x05B8, 0x05B8, 0x05B8, 0x05B8, 0x059F, 0x3111, 0x24E0, 
    0x02A0, 0xFF51, 0x05E7, 0xFB47, 0xFB47, 0x05E7, 0x24AC, 0x2669, 
    0x05BB, 0x05BB, 0x05BB, 0x05BB, 0x05BB, 0x05BB, 0x05BB, 0x05BB, 
    0x003F, 0x061F, 0x055E, 0x00BF, 0xF7BF, 0x037E, 0xFF1F, 0xF73F, 
    0x0022, 0x201E, 0x201C, 0xFF02, 0x301E, 0x301D, 0x201D, 0x2018, 
    0x201B, 0x201B, 0x2019, 0x0149, 0x201A, 0x0027, 0xFF07, 0x0072, 
    0x057C, 0x09B0, 0x0155, 0x0930, 0x221A, 0xF8E5, 0x33AE, 0x33AF, 
    0x33AD, 0x05BF, 0x05BF, 0x0AB0, 0x0A30, 0x3089, 0x30E9, 0xFF97, 
    0x09F1, 0x09F0, 0x0264, 0x2236, 0x3116, 0x0159, 0x0157, 0x24E1, 
    0x0157, 0x0211, 0x1E59, 0x1E5B, 0x1E5D, 0x203B, 0x2286, 0x2287, 
    0x00AE, 0xF8E8, 0xF6DA, 0x0631, 0x0580, 0xFEAE, 0x308C, 0x30EC, 
    0xFF9A, 0x05E8, 0xFB48, 0x05E8, 0x223D, 0x0597, 0x0597, 0x2310, 
    0x027E, 0x027F, 0x09DD, 0x095D, 0x03C1, 0x027D, 0x027B, 0x02B5, 
    0x03F1, 0x02DE, 0x3271, 0x3211, 0x3263, 0x3140, 0x313A, 0x3169, 
    0x3139, 0x313B, 0x316C, 0x3203, 0x313F, 0x313C, 0x316B, 0x313D, 
    0x313E, 0x316A, 0x316D, 0x221F, 0x0319, 0x22BF, 0x308A, 0x30EA, 
    0xFF98, 0x02DA, 0x0325, 0x030A, 0x02BF, 0x0559, 0x031C, 0x02D3, 
    0x02BE, 0x0339, 0x02D2, 0x0213, 0x3351, 0x1E5F, 0x027C, 0x027A, 
    0xFF52, 0x308D, 0x30ED, 0xFF9B, 0x0E23, 0x24AD, 0x09DC, 0x0931, 
    0x0A5C, 0x0691, 0xFB8D, 0x09E0, 0x0960, 0x0AE0, 0x09C4, 0x0944, 
    0x0AC4, 0xF6F1, 0x2590, 0x0279, 0x02B4, 0x308B, 0x30EB, 0xFF99, 
    0x09F2, 0x09F3, 0xF6DD, 0x0E24, 0x098B, 0x090B, 0x0A8B, 0x09C3, 
    0x0943, 0x0AC3, 0x0073, 0x09B8, 0x015B, 0x1E65, 0x0635, 0x0938, 
    0xFEBA, 0xFEBB, 0xFEBC, 0x0AB8, 0x0A38, 0x3055, 0x30B5, 0xFF7B, 
    0xFDFA, 0x05E1, 0xFB41, 0xFB41, 0x05E1, 0x0E32, 0x0E41, 0x0E44, 
    0x0E43, 0x0E33, 0x0E30, 0x0E40, 0xF886, 0x0E35, 0xF885, 0x0E34, 
    0x0E42, 0xF888, 0x0E37, 0xF887, 0x0E36, 0x0E38, 0x0E39, 0x3119, 
    0x0161, 0x1E67, 0x015F, 0x0259, 0x04D9, 0x04DB, 0x025A, 0x24E2, 
    0x015D, 0x0219, 0x1E61, 0x1E63, 0x1E69, 0x033C, 0x2033, 0x02CA, 
    0x00A7, 0x0633, 0xFEB2, 0xFEB3, 0xFEB4, 0x05B6, 0x05B6, 0x05B6, 
    0x05B6, 0x05B6, 0x05B6, 0x05B6, 0x0592, 0x05B6, 0x057D, 0x305B, 
    0x30BB, 0xFF7E, 0x003B, 0x061B, 0xFF1B, 0xFE54, 0x309C, 0xFF9F, 
    0x3322, 0x3323, 0x0037, 0x0667, 0x09ED, 0x2466, 0x2790, 0x096D, 
    0x215E, 0x0AED, 0x0A6D, 0x0667, 0x3027, 0x3226, 0x2087, 0xFF17, 
    0xF737, 0x247A, 0x248E, 0x06F7, 0x2176, 0x2077, 0x2470, 0x2484, 
    0x2498, 0x0E57, 0x00AD, 0x0577, 0x09B6, 0x0448, 0x0651, 0xFC61, 
    0xFC5E, 0xFC60, 0xFC62, 0xFC5F, 0x2592, 0x2593, 0x2591, 0x2592, 
    0x0936, 0x0AB6, 0x0A36, 0x0593, 0x3115, 0x0449, 0x0634, 0xFEB6, 
    0xFEB7, 0xFEB8, 0x03E3, 0x20AA, 0x20AA, 0x05B0, 0x05B0, 0x05B0, 
    0x05B0, 0x05B0, 0x05B0, 0x05B0, 0x05B0, 0x05B0, 0x04BB, 0x03ED, 
    0x05E9, 0xFB49, 0xFB49, 0xFB2C, 0xFB2C, 0xFB2D, 0xFB2D, 0x05C1, 
    0x05E9, 0xFB2A, 0xFB2A, 0xFB2B, 0xFB2B, 0x0282, 0x03C3, 0x03C2, 
    0x03C2, 0x03F2, 0x3057, 0x30B7, 0xFF7C, 0x05BD, 0x05BD, 0x223C, 
    0x05C2, 0x3274, 0x3214, 0x317E, 0x3266, 0x317A, 0x3145, 0x317B, 
    0x3206, 0x317D, 0x317C, 0x0036, 0x0666, 0x09EC, 0x2465, 0x278F, 
    0x096C, 0x0AEC, 0x0A6C, 0x0666, 0x3026, 0x3225, 0x2086, 0xFF16, 
    0xF736, 0x2479, 0x248D, 0x06F6, 0x2175, 0x2076, 0x246F, 0x09F9, 
    0x2483, 0x2497, 0x0E56, 0x002F, 0xFF0F, 0x017F, 0x1E9B, 0x263A, 
    0xFF53, 0x05C3, 0x00AD, 0x044C, 0x305D, 0x30BD, 0xFF7F, 0x0338, 
    0x0337, 0x0E29, 0x0E28, 0x0E0B, 0x0E2A, 0x0020, 0x0020, 0x2660, 
    0x2660, 0x2664, 0x24AE, 0x033B, 0x33C4, 0x339D, 0x25A9, 0x25A4, 
    0x338F, 0x339E, 0x33CE, 0x33D1, 0x33D2, 0x338E, 0x33D5, 0x339C, 
    0x33A1, 0x25A6, 0x25A7, 0x25A8, 0x25A5, 0x25A3, 0x33DB, 0x09B7, 
    0x0937, 0x0AB7, 0x3149, 0x3185, 0x3180, 0x3132, 0x3165, 0x3143, 
    0x3146, 0x3138, 0xF6F2, 0x00A3, 0xFFE1, 0x0336, 0x0335, 0x2282, 
    0x228A, 0x2286, 0x227B, 0x220B, 0x3059, 0x30B9, 0xFF7D, 0x0652, 
    0x2211, 0x263C, 0x2283, 0x228B, 0x2287, 0x33DC, 0x337C, 0x0074, 
    0x09A4, 0x22A4, 0x22A3, 0x0924, 0x0AA4, 0x0A24, 0x0637, 0xFEC2, 
    0xFEC3, 0x305F, 0xFEC4, 0x337D, 0x30BF, 0xFF80, 0x0640, 0x03C4, 
    0x05EA, 0xFB4A, 0xFB4A, 0xFB4A, 0x05EA, 0x0167, 0x310A, 0x0165, 
    0x02A8, 0x0163, 0x0686, 0xFB7B, 0xFB7C, 0xFB7D, 0x24E3, 0x1E71, 
    0x0163, 0x1E97, 0x1E6B, 0x1E6D, 0x0442, 0x04AD, 0x062A, 0xFE96, 
    0xFCA2, 0xFC0C, 0xFE97, 0x3066, 0xFCA1, 0xFC0B, 0x0629, 0xFE94, 
    0xFE98, 0xFCA4, 0xFC0E, 0xFC73, 0x30C6, 0xFF83, 0x2121, 0x260E, 
    0x05A0, 0x05A9, 0x2469, 0x3229, 0x247D, 0x2491, 0x2179, 0x02A7, 
    0x05D8, 0xFB38, 0xFB38, 0x05D8, 0x04B5, 0x059B, 0x059B, 0x09A5, 
    0x0925, 0x0AA5, 0x0A25, 0x0630, 0xFEAC, 0xF898, 0xF897, 0x0E4C, 
    0xF896, 0x062B, 0xFE9A, 0xFE9B, 0xFE9C, 0x2203, 0x2234, 0x03B8, 
    0x03D1, 0x03D1, 0x3279, 0x3219, 0x326B, 0x314C, 0x320B, 0x246C, 
    0x2480, 0x2494, 0x0E11, 0x01AD, 0x0E12, 0x00FE, 0x0E17, 0x0E10, 
    0x0E18, 0x0E16, 0x0482, 0x066C, 0x066C, 0x0033, 0x0663, 0x09E9, 
    0x2462, 0x278C, 0x0969, 0x215C, 0x0AE9, 0x0A69, 0x0663, 0x3023, 
    0x3222, 0x2083, 0xFF13, 0x09F6, 0xF733, 0x2476, 0x248A, 0x06F3, 
    0x00BE, 0xF6DE, 0x2172, 0x00B3, 0x0E53, 0x3394, 0x3061, 0x30C1, 
    0xFF81, 0x3270, 0x3210, 0x3262, 0x3137, 0x3202, 0x02DC, 0x0330, 
    0x0303, 0x0303, 0x0360, 0x223C, 0x0334, 0x033E, 0x2297, 0x0596, 
    0x0596, 0x0A70, 0x0483, 0x057F, 0x1E6F, 0xFF54, 0x0569, 0x3068, 
    0x30C8, 0xFF84, 0x02E5, 0x02E9, 0x02E6, 0x02E8, 0x02E7, 0x01BD, 
    0x0185, 0x01A8, 0x0384, 0x3327, 0x0E0F, 0x3014, 0xFE5D, 0xFE39, 
    0x3015, 0xFE5E, 0xFE3A, 0x0E15, 0x01AB, 0x24AF, 0x2122, 0xF8EA, 
    0xF6DB, 0x0288, 0x25BC, 0x25C4, 0x25BA, 0x25B2, 0x02A6, 0x05E6, 
    0xFB46, 0xFB46, 0x05E6, 0x0446, 0x05B5, 0x05B5, 0x05B5, 0x05B5, 
    0x05B5, 0x05B5, 0x05B5, 0x05B5, 0x045B, 0xF6F3, 0x099F, 0x091F, 
    0x0A9F, 0x0A1F, 0x0679, 0xFB67, 0xFB68, 0xFB69, 0x09A0, 0x0920, 
    0x0AA0, 0x0A20, 0x0287, 0x3064, 0x30C4, 0xFF82, 0x3063, 0x30C3, 
    0xFF6F, 0x246B, 0x247F, 0x2493, 0x217B, 0x2473, 0x5344, 0x2487, 
    0x249B, 0x0032, 0x0662, 0x09E8, 0x2461, 0x278B, 0x0968, 0x2025, 
    0x2025, 0xFE30, 0x0AE8, 0x0A68, 0x0662, 0x3022, 0x3221, 0x2082, 
    0xFF12, 0x09F5, 0xF732, 0x2475, 0x2489, 0x06F2, 0x2171, 0x01BB, 
    0x00B2, 0x0E52, 0x2154, 0x0075, 0x00FA, 0x0289, 0x0989, 0x3128, 
    0x016D, 0x01D4, 0x24E4, 0x00FB, 0x1E77, 0x0443, 0x0951, 0x0171, 
    0x0215, 0x0909, 0x00FC, 0x01D8, 0x1E73, 0x01DA, 0x04F1, 0x01DC, 
    0x01D6, 0x1EE5, 0x00F9, 0x0A89, 0x0A09, 0x3046, 0x1EE7, 0x01B0, 
    0x1EE9, 0x1EF1, 0x1EEB, 0x1EED, 0x1EEF, 0x0171, 0x04F3, 0x0217, 
    0x30A6, 0xFF73, 0x0479, 0x315C, 0x016B, 0x04EF, 0x1E7B, 0x0A41, 
    0xFF55, 0x005F, 0x2017, 0xFF3F, 0xFE33, 0xFE4F, 0x222A, 0x2200, 
    0x0173, 0x24B0, 0x2580, 0x05C4, 0x03C5, 0x03CB, 0x03B0, 0x028A, 
    0x03CD, 0x031D, 0x02D4, 0x0A73, 0x016F, 0x045E, 0x3045, 0x30A5, 
    0xFF69, 0x04AF, 0x04B1, 0x0169, 0x1E79, 0x1E75, 0x098A, 0x090A, 
    0x0A8A, 0x0A0A, 0x0A42, 0x09C2, 0x0942, 0x0AC2, 0x09C1, 0x0941, 
    0x0AC1, 0x0076, 0x0935, 0x0AB5, 0x0A35, 0x30F7, 0x05D5, 0xFB35, 
    0xFB35, 0xFB35, 0x05D5, 0xFB4B, 0xFB4B, 0x05F0, 0x05F1, 0x24E5, 
    0x1E7F, 0x0432, 0x06A4, 0xFB6B, 0xFB6C, 0xFB6D, 0x30F9, 0x2640, 
    0x007C, 0x030D, 0x0329, 0x02CC, 0x02C8, 0x057E, 0x028B, 0x30F8, 
    0x09CD, 0x094D, 0x0ACD, 0x0983, 0x0903, 0x0A83, 0xFF56, 0x0578, 
    0x309E, 0x30FE, 0x309B, 0xFF9E, 0x30FA, 0x24B1, 0x1E7D, 0x028C, 
    0x3094, 0x30F4, 0x0077, 0x1E83, 0x3159, 0x308F, 0x30EF, 0xFF9C, 
    0x3158, 0x308E, 0x30EE, 0x3357, 0x301C, 0xFE34, 0x0648, 0xFEEE, 
    0x0624, 0xFE86, 0x33DD, 0x24E6, 0x0175, 0x1E85, 0x1E87, 0x1E89, 
    0x3091, 0x2118, 0x30F1, 0x315E, 0x315D, 0x1E81, 0x25E6, 0x25CB, 
    0x25D9, 0x300E, 0xFE43, 0x300F, 0xFE44, 0x25C7, 0x25C8, 0x25BF, 
    0x25BD, 0x25C3, 0x25C1, 0x3016, 0x3017, 0x25B9, 0x25B7, 0x25AB, 
    0x263A, 0x25A1, 0x2606, 0x260F, 0x3018, 0x3019, 0x25B5, 0x25B3, 
    0x3090, 0x30F0, 0x315F, 0xFF57, 0x3092, 0x30F2, 0xFF66, 0x20A9, 
    0xFFE6, 0x0E27, 0x24B2, 0x1E98, 0x02B7, 0x028D, 0x01BF, 0x0078, 
    0x033D, 0x3112, 0x24E7, 0x1E8D, 0x1E8B, 0x056D, 0x03BE, 0xFF58, 
    0x24B3, 0x02E3, 0x0079, 0x334E, 0x09AF, 0x00FD, 0x092F, 0x3152, 
    0x0AAF, 0x0A2F, 0x3084, 0x30E4, 0xFF94, 0x3151, 0x0E4E, 0x3083, 
    0x30E3, 0xFF6C, 0x0463, 0x24E8, 0x0177, 0x00FF, 0x1E8F, 0x1EF5, 
    0x064A, 0x06D2, 0xFBAF, 0xFEF2, 0x0626, 0xFE8A, 0xFE8B, 0xFE8C, 
    0xFEF3, 0xFEF4, 0xFCDD, 0xFC58, 0xFC94, 0x06D1, 0x3156, 0x00A5, 
    0xFFE5, 0x3155, 0x3186, 0x05AA, 0x05AA, 0x044B, 0x04F9, 0x3181, 
    0x3183, 0x3182, 0x059A, 0x1EF3, 0x01B4, 0x1EF7, 0x0575, 0x0457, 
    0x3162, 0x262F, 0x0582, 0xFF59, 0x05D9, 0xFB39, 0xFB39, 0x05D9, 
    0x05F2, 0xFB1F, 0x3088, 0x3189, 0x30E8, 0xFF96, 0x315B, 0x3087, 
    0x30E7, 0xFF6E, 0x03F3, 0x3188, 0x3187, 0x0E22, 0x0E0D, 0x24B4, 
    0x037A, 0x0345, 0x01A6, 0x1E99, 0x02B8, 0x1EF9, 0x028E, 0x3086, 
    0x318C, 0x30E6, 0xFF95, 0x3160, 0x046B, 0x046D, 0x0467, 0x0469, 
    0x3085, 0x30E5, 0xFF6D, 0x318B, 0x318A, 0x09DF, 0x095F, 0x007A, 
    0x0566, 0x017A, 0x095B, 0x0A5B, 0x0638, 0xFEC6, 0xFEC7, 0x3056, 
    0xFEC8, 0x0632, 0xFEB0, 0x30B6, 0x0595, 0x0594, 0x0598, 0x05D6, 
    0xFB36, 0xFB36, 0x05D6, 0x3117, 0x017E, 0x24E9, 0x1E91, 0x0291, 
    0x017C, 0x017C, 0x1E93, 0x0437, 0x0499, 0x04DF, 0x305C, 0x30BC, 
    0x0030, 0x0660, 0x09E6, 0x0966, 0x0AE6, 0x0A66, 0x0660, 0x2080, 
    0xFF10, 0xF730, 0x06F0, 0x2070, 0x0E50, 0xFEFF, 0x200C, 0x200B, 
    0x03B6, 0x3113, 0x056A, 0x04C2, 0x0436, 0x0497, 0x04DD, 0x3058, 
    0x30B8, 0x05AE, 0x1E95, 0xFF5A, 0x305E, 0x30BE, 0x24B5, 0x0290, 
    0x01B6, 0x305A, 0x30BA, 
};

wchar_t ps_glyph_to_unicode(glyph g)
{
    if (g < lenof(ps_codes_alphabetic))
	return ps_codes_alphabetic[g];
    return 0xFFFF;
}

/* ----------------------------------------------------------------------
 * Data about the standard fonts: available glyphs and font metrics.
 */

/*
 * Character set list extracted from the back of the PostScript
 * Language Reference Manual.
 * 
 * I'm going to include a bit of shell which creates a file
 * containing the list of characters, and then several further bits
 * of shell that process it in different ways.
 */

/*

tr -s ' \n' '\n' << EOF > stdchars.txt
A B C D E F G H I J K L M N O P Q R S T U V W X Y Z
a b c d e f g h i j k l m n o p q r s t u v w x y z
AE Aacute Acircumflex Adieresis Agrave Aring Atilde
Ccedilla
Eacute Ecircumflex Edieresis Egrave
Eth
Iacute Icircumflex Idieresis Igrave
Lslash
Ntilde
OE Oacute Ocircumflex Odieresis Ograve Oslash Otilde
Scaron
Thorn
Uacute Ucircumflex Udieresis Ugrave
Yacute Ydieresis
Zcaron
ae aacute acircumflex adieresis agrave aring atilde
ccedilla
eacute ecircumflex edieresis egrave
eth
iacute icircumflex idieresis igrave
lslash
ntilde
oe oacute ocircumflex odieresis ograve oslash otilde
scaron
thorn
uacute ucircumflex udieresis ugrave
yacute ydieresis
zcaron
acute ampersand asciicircum asciitilde asterisk at backslash bar
braceleft braceright bracketleft bracketright breve brokenbar bullet
caron cedilla cent circumflex colon comma copyright currency dagger
daggerdbl degree dieresis divide dollar dotaccent dotlessi eight
ellipsis emdash endash equal exclam exclamdown fi five fl florin
four fraction germandbls grave greater guillemotleft guillemotright
guilsinglleft guilsinglright hungarumlaut hyphen less logicalnot
macron minus mu multiply nine numbersign ogonek one onehalf onequarter
onesuperior ordfeminine ordmasculine paragraph parenleft parenright
percent period periodcentered perthousand plus plusminus question
questiondown quotedbl quotedblbase quotedblleft quotedblright quoteleft
quoteright quotesinglbase quotesingle registered ring section semicolon
seven six slash space sterling three threequarters threesuperior tilde
trademark two twosuperior underscore yen zero
EOF

 */

/*
 * A simple string array giving the glyph list.
 */

/*

perl -ne 'chomp; print "\"$_\", "; END { print "NULL" }' stdchars.txt | \
    fold -s -w68 | sed 's/^/    /'

*/

const char *const ps_std_glyphs[] = {
    "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", 
    "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z", 
    "a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", 
    "n", "o", "p", "q", "r", "s", "t", "u", "v", "w", "x", "y", "z", 
    "AE", "Aacute", "Acircumflex", "Adieresis", "Agrave", "Aring", 
    "Atilde", "Ccedilla", "Eacute", "Ecircumflex", "Edieresis", 
    "Egrave", "Eth", "Iacute", "Icircumflex", "Idieresis", "Igrave", 
    "Lslash", "Ntilde", "OE", "Oacute", "Ocircumflex", "Odieresis", 
    "Ograve", "Oslash", "Otilde", "Scaron", "Thorn", "Uacute", 
    "Ucircumflex", "Udieresis", "Ugrave", "Yacute", "Ydieresis", 
    "Zcaron", "ae", "aacute", "acircumflex", "adieresis", "agrave", 
    "aring", "atilde", "ccedilla", "eacute", "ecircumflex", 
    "edieresis", "egrave", "eth", "iacute", "icircumflex", "idieresis", 
    "igrave", "lslash", "ntilde", "oe", "oacute", "ocircumflex", 
    "odieresis", "ograve", "oslash", "otilde", "scaron", "thorn", 
    "uacute", "ucircumflex", "udieresis", "ugrave", "yacute", 
    "ydieresis", "zcaron", "acute", "ampersand", "asciicircum", 
    "asciitilde", "asterisk", "at", "backslash", "bar", "braceleft", 
    "braceright", "bracketleft", "bracketright", "breve", "brokenbar", 
    "bullet", "caron", "cedilla", "cent", "circumflex", "colon", 
    "comma", "copyright", "currency", "dagger", "daggerdbl", "degree", 
    "dieresis", "divide", "dollar", "dotaccent", "dotlessi", "eight", 
    "ellipsis", "emdash", "endash", "equal", "exclam", "exclamdown", 
    "fi", "five", "fl", "florin", "four", "fraction", "germandbls", 
    "grave", "greater", "guillemotleft", "guillemotright", 
    "guilsinglleft", "guilsinglright", "hungarumlaut", "hyphen", 
    "less", "logicalnot", "macron", "minus", "mu", "multiply", "nine", 
    "numbersign", "ogonek", "one", "onehalf", "onequarter", 
    "onesuperior", "ordfeminine", "ordmasculine", "paragraph", 
    "parenleft", "parenright", "percent", "period", "periodcentered", 
    "perthousand", "plus", "plusminus", "question", "questiondown", 
    "quotedbl", "quotedblbase", "quotedblleft", "quotedblright", 
    "quoteleft", "quoteright", "quotesinglbase", "quotesingle", 
    "registered", "ring", "section", "semicolon", "seven", "six", 
    "slash", "space", "sterling", "three", "threequarters", 
    "threesuperior", "tilde", "trademark", "two", "twosuperior", 
    "underscore", "yen", "zero", NULL
};

/*
 * Standard TrueType glyph mapping, from the TrueType reference manual.
 */

/*

wget http://developer.apple.com/textfonts/TTRefMan/RM06/Chap6post.html
< Chap6post.html \
sed -n '/<td>\.notdef/,/<\/table>/s/^.*<td>\([^0-9].*\)<\/td>$/\1/p' |
perl -e '
open G, "glyphnames.txt" or die;
chomp(@g = <G>); %g = map(($_, $i++), @g);
while(<>){chomp;print"$g{$_}, "}
print "NOGLYPH\n";' | fold -sw68 | sed 's/^/    /'

*/

glyph const tt_std_glyphs[] = {
    0, 1, 2740, 3501, 1776, 3136, 2781, 1597, 2994, 1079, 3149, 2942, 
    2952, 1165, 3046, 1411, 2128, 2998, 3483, 4168, 2862, 3793, 3669, 
    1860, 1831, 3459, 3362, 1671, 2699, 1404, 3354, 2464, 1738, 1972, 
    3128, 1174, 2, 53, 66, 95, 129, 194, 205, 234, 258, 303, 311, 339, 
    361, 374, 395, 461, 476, 480, 501, 572, 605, 660, 670, 680, 688, 
    713, 1286, 1196, 1291, 1156, 3865, 1964, 736, 1194, 1314, 1483, 
    1632, 1789, 1888, 1990, 2134, 2276, 2304, 2418, 2513, 2650, 2797, 
    2924, 3086, 3151, 3282, 3559, 3819, 3905, 3954, 4023, 4034, 4135, 
    1272, 1204, 1279, 1158, 29, 45, 72, 130, 392, 416, 617, 738, 1032, 
    760, 777, 1175, 1113, 1334, 1633, 1664, 1647, 1657, 2135, 2209, 
    2142, 2198, 2772, 2798, 2829, 2812, 2822, 2910, 3820, 3842, 3827, 
    3834, 1493, 1554, 1344, 3539, 3336, 1310, 2940, 1920, 3184, 1461, 
    3742, 766, 1574, 2756, 3, 451, 2237, 3049, 2465, 1973, 4071, 2623, 
    2962, 3552, 3070, 3029, 2239, 2897, 2898, 438, 782, 2902, 3131, 
    1779, 2493, 3156, 1853, 1106, 109, 1983, 1984, 1707, 2738, 35, 50, 
    457, 396, 2825, 1720, 1713, 3138, 3142, 3143, 3146, 1583, 2501, 
    4053, 694, 1886, 1478, 1985, 1986, 1809, 1852, 1494, 3000, 3148, 
    3137, 3008, 18, 137, 7, 147, 153, 262, 267, 271, 281, 398, 406, 
    1104, 421, 606, 611, 626, 1610, 1389, 3702, 2515, 1296, 1605, 3233, 
    1342, 2125, 2827, 1328, 358, 2503, 545, 3320, 716, 4156, 1304, 186, 
    1763, 689, 4037, 588, 3661, 2597, 2635, 2886, 3816, 3691, 2874, 
    2884, 3688, 1887, 211, 1905, 276, 548, 3322, 68, 1317, 71, 1333, 
    1537, NOGLYPH
};

/*
 * Character width information from the standard PS fonts.
 * Units are UNITS_PER_PT defined in paper.h.
 * Extracted from the AFM files in
 * <http://partners.adobe.com/public/developer/en/pdf/Core14_AFMs.tar>
 * by the following script:

fonts="Times-Roman Times-Italic Times-Bold Times-BoldItalic \
       Helvetica Helvetica-Oblique Helvetica-Bold Helvetica-BoldOblique \
       Courier Courier-Oblique Courier-Bold Courier-BoldOblique"

for i in $fonts; do
  printf 'static const kern_pair %s_kerns[] = {\n' $(echo $i | tr 'A-Z\-' a-z_)
  perl -e '
open S, "stdchars.txt" or die;
chomp(@s = <S>);
open G, "glyphnames.txt" or die;
chomp(@g = <G>); %g = map(($_, $i++), @g); %g = map(($_, $g{$_}), @s); 
open M, "$ARGV[0].afm" or die;
while (<M>) { /KPX (\S+) (\S+) (\S+)/ and exists $g{$1} and exists $g{$2} and
  print "{$g{$1},$g{$2},$3}, "; } print "\n"' $i |\
    fold -sw 68 | sed 's/^/    /'
  printf '    {NOGLYPH,NOGLYPH,0}\n};\n'
  printf 'static const ligature %s_ligs[] = {\n' $(echo $i | tr 'A-Z\-' a-z_)
  perl -e '
open S, "stdchars.txt" or die;
chomp(@s = <S>);
open G, "glyphnames.txt" or die;
chomp(@g = <G>); %g = map(($_, $i++), @g); %g = map(($_, $g{$_}), @s); 
open M, "$ARGV[0].afm" or die;
while (<M>) { / N (\S+) / and $l = $1;
  while (/ L (\S+) (\S+) /g) { exists $g{$l} and exists $g{$1} and
    exists $g{$2} and print "{$g{$l},$g{$1},$g{$2}}, "; } } print "\n"' $i |\
    fold -sw 68 | sed 's/^/    /'
  printf '    {NOGLYPH,NOGLYPH,NOGLYPH}\n};\n'
done

cat <<DECL

static const struct ps_std_font_data {
    char const *name;
    kern_pair const *kerns;
    ligature const *ligs;
    int widths[lenof(ps_std_glyphs)-1];
} ps_std_fonts[] = {
DECL
for i in $fonts; do
  printf '    { "%s",\n\t%s_kerns, %s_ligs, {\n' $i \
    $(echo $i | tr 'A-Z\-' a-z_) $(echo $i | tr 'A-Z\-' a-z_)
  perl -e '
open M, "$ARGV[0].afm" or die;
while (<M>) { /WX (\d+) ; N (\S+)/ and $m{$2} = $1; }
open G, "stdchars.txt" or die;
while (<G>) { chomp; print "$m{$_}, " }; print "\n"' $i |\
    fold -sw 68 | sed 's/^/	/'
  printf '    }},\n'
done
printf '};\n'

 * The AFM files have the following copyright and licence: 
 *
 * Copyright (c) 1985, 1987, 1989, 1990, 1991, 1992, 1993, 1997
 * Adobe Systems Incorporated.  All Rights Reserved.
 *
 * This file and the 14 PostScript(R) AFM files it accompanies may be
 * used, copied, and distributed for any purpose and without charge,
 * with or without modification, provided that all copyright notices
 * are retained; that the AFM files are not distributed without this
 * file; that all modifications to this file or any of the AFM files
 * are prominently noted in the modified file(s); and that this
 * paragraph is not modified. Adobe Systems has no responsibility or
 * obligation to support the use of the AFM files.
 */

static const kern_pair times_roman_kerns[] = {
    {2,66,-40}, {2,72,-40}, {2,205,-40}, {2,395,-55}, {2,398,-55}, 
    {2,406,-55}, {2,416,-55}, {2,421,-55}, {2,451,-55}, {2,457,-55}, 
    {2,476,-55}, {2,572,-111}, {2,605,-55}, {2,606,-55}, {2,611,-55}, 
    {2,617,-55}, {2,626,-55}, {2,660,-135}, {2,670,-90}, {2,688,-105}, 
    {2,689,-105}, {2,694,-105}, {2,3146,-111}, {2,3905,-74}, 
    {2,3954,-92}, {2,4034,-92}, {2,4037,-92}, {2,4053,-92}, {7,66,-40}, 
    {7,72,-40}, {7,205,-40}, {7,395,-55}, {7,398,-55}, {7,406,-55}, 
    {7,416,-55}, {7,421,-55}, {7,451,-55}, {7,457,-55}, {7,476,-55}, 
    {7,572,-111}, {7,605,-55}, {7,606,-55}, {7,611,-55}, {7,617,-55}, 
    {7,626,-55}, {7,660,-135}, {7,670,-90}, {7,688,-105}, {7,689,-105}, 
    {7,694,-105}, {7,3146,-111}, {7,3905,-74}, {7,3954,-92}, 
    {7,4034,-92}, {7,4037,-92}, {7,4053,-92}, {18,66,-40}, {18,72,-40}, 
    {18,205,-40}, {18,395,-55}, {18,398,-55}, {18,406,-55}, 
    {18,416,-55}, {18,421,-55}, {18,451,-55}, {18,457,-55}, 
    {18,476,-55}, {18,572,-111}, {18,605,-55}, {18,606,-55}, 
    {18,611,-55}, {18,617,-55}, {18,626,-55}, {18,660,-135}, 
    {18,670,-90}, {18,688,-105}, {18,689,-105}, {18,694,-105}, 
    {18,3146,-111}, {18,3905,-74}, {18,3954,-92}, {18,4034,-92}, 
    {18,4037,-92}, {18,4053,-92}, {29,66,-40}, {29,72,-40}, 
    {29,205,-40}, {29,395,-55}, {29,398,-55}, {29,406,-55}, 
    {29,416,-55}, {29,421,-55}, {29,451,-55}, {29,457,-55}, 
    {29,476,-55}, {29,572,-111}, {29,605,-55}, {29,606,-55}, 
    {29,611,-55}, {29,617,-55}, {29,626,-55}, {29,660,-135}, 
    {29,670,-90}, {29,688,-105}, {29,689,-105}, {29,694,-105}, 
    {29,3146,-111}, {29,3905,-74}, {29,3954,-92}, {29,4034,-92}, 
    {29,4037,-92}, {29,4053,-92}, {35,66,-40}, {35,72,-40}, 
    {35,205,-40}, {35,395,-55}, {35,398,-55}, {35,406,-55}, 
    {35,416,-55}, {35,421,-55}, {35,451,-55}, {35,457,-55}, 
    {35,476,-55}, {35,572,-111}, {35,605,-55}, {35,606,-55}, 
    {35,611,-55}, {35,617,-55}, {35,626,-55}, {35,660,-135}, 
    {35,670,-90}, {35,688,-105}, {35,689,-105}, {35,694,-105}, 
    {35,3146,-111}, {35,3905,-74}, {35,3954,-92}, {35,4034,-92}, 
    {35,4037,-92}, {35,4053,-92}, {45,66,-40}, {45,72,-40}, 
    {45,205,-40}, {45,395,-55}, {45,398,-55}, {45,406,-55}, 
    {45,416,-55}, {45,421,-55}, {45,451,-55}, {45,457,-55}, 
    {45,476,-55}, {45,572,-111}, {45,605,-55}, {45,606,-55}, 
    {45,611,-55}, {45,617,-55}, {45,626,-55}, {45,660,-135}, 
    {45,670,-90}, {45,688,-105}, {45,689,-105}, {45,694,-105}, 
    {45,3146,-111}, {45,3905,-74}, {45,3954,-92}, {45,4034,-92}, 
    {45,4037,-92}, {45,4053,-92}, {50,66,-40}, {50,72,-40}, 
    {50,205,-40}, {50,395,-55}, {50,398,-55}, {50,406,-55}, 
    {50,416,-55}, {50,421,-55}, {50,451,-55}, {50,457,-55}, 
    {50,476,-55}, {50,572,-111}, {50,605,-55}, {50,606,-55}, 
    {50,611,-55}, {50,617,-55}, {50,626,-55}, {50,660,-135}, 
    {50,670,-90}, {50,688,-105}, {50,689,-105}, {50,694,-105}, 
    {50,3146,-111}, {50,3905,-74}, {50,3954,-92}, {50,4034,-92}, 
    {50,4037,-92}, {50,4053,-92}, {53,2,-35}, {53,7,-35}, {53,18,-35}, 
    {53,29,-35}, {53,35,-35}, {53,45,-35}, {53,50,-35}, {53,605,-10}, 
    {53,606,-10}, {53,611,-10}, {53,617,-10}, {53,626,-10}, {95,2,-40}, 
    {95,7,-40}, {95,18,-40}, {95,29,-40}, {95,35,-40}, {95,45,-40}, 
    {95,50,-40}, {95,660,-40}, {95,670,-30}, {95,688,-55}, 
    {95,689,-55}, {95,694,-55}, {194,2,-74}, {194,7,-74}, {194,18,-74}, 
    {194,29,-74}, {194,35,-74}, {194,45,-74}, {194,50,-74}, 
    {194,736,-15}, {194,738,-15}, {194,760,-15}, {194,777,-15}, 
    {194,1032,-15}, {194,1113,-15}, {194,1175,-15}, {194,1411,-80}, 
    {194,2797,-15}, {194,2798,-15}, {194,2812,-15}, {194,2822,-15}, 
    {194,2829,-15}, {194,2902,-15}, {194,2910,-15}, {194,2998,-80}, 
    {303,2,-60}, {303,7,-60}, {303,18,-60}, {303,29,-60}, {303,35,-60}, 
    {303,45,-60}, {303,50,-60}, {311,395,-30}, {311,398,-30}, 
    {311,406,-30}, {311,416,-30}, {311,421,-30}, {311,451,-30}, 
    {311,457,-30}, {311,1632,-25}, {311,1633,-25}, {311,1647,-25}, 
    {311,1657,-25}, {311,1664,-25}, {311,2797,-35}, {311,2798,-35}, 
    {311,2812,-35}, {311,2822,-35}, {311,2829,-35}, {311,2902,-35}, 
    {311,2910,-35}, {311,3819,-15}, {311,3820,-15}, {311,3827,-15}, 
    {311,3834,-15}, {311,3842,-15}, {311,4034,-25}, {311,4037,-25}, 
    {311,4053,-25}, {339,572,-92}, {339,660,-100}, {339,670,-74}, 
    {339,688,-100}, {339,689,-100}, {339,694,-100}, {339,3146,-92}, 
    {339,4034,-55}, {339,4037,-55}, {339,4053,-55}, {358,572,-92}, 
    {358,660,-100}, {358,670,-74}, {358,688,-100}, {358,689,-100}, 
    {358,694,-100}, {358,3146,-92}, {358,4034,-55}, {358,4037,-55}, 
    {358,4053,-55}, {374,2,-35}, {374,7,-35}, {374,18,-35}, 
    {374,29,-35}, {374,35,-35}, {374,45,-35}, {374,50,-35}, 
    {392,2,-35}, {392,7,-35}, {392,18,-35}, {392,29,-35}, {392,35,-35}, 
    {392,45,-35}, {392,50,-35}, {395,2,-35}, {395,7,-35}, {395,18,-35}, 
    {395,29,-35}, {395,35,-35}, {395,45,-35}, {395,50,-35}, 
    {395,572,-40}, {395,660,-50}, {395,670,-35}, {395,680,-40}, 
    {395,688,-50}, {395,689,-50}, {395,694,-50}, {398,2,-35}, 
    {398,7,-35}, {398,18,-35}, {398,29,-35}, {398,35,-35}, 
    {398,45,-35}, {398,50,-35}, {398,572,-40}, {398,660,-50}, 
    {398,670,-35}, {398,680,-40}, {398,688,-50}, {398,689,-50}, 
    {398,694,-50}, {406,2,-35}, {406,7,-35}, {406,18,-35}, 
    {406,29,-35}, {406,35,-35}, {406,45,-35}, {406,50,-35}, 
    {406,572,-40}, {406,660,-50}, {406,670,-35}, {406,680,-40}, 
    {406,688,-50}, {406,689,-50}, {406,694,-50}, {416,2,-35}, 
    {416,7,-35}, {416,18,-35}, {416,29,-35}, {416,35,-35}, 
    {416,45,-35}, {416,50,-35}, {416,572,-40}, {416,660,-50}, 
    {416,670,-35}, {416,680,-40}, {416,688,-50}, {416,689,-50}, 
    {416,694,-50}, {421,2,-35}, {421,7,-35}, {421,18,-35}, 
    {421,29,-35}, {421,35,-35}, {421,45,-35}, {421,50,-35}, 
    {421,572,-40}, {421,660,-50}, {421,670,-35}, {421,680,-40}, 
    {421,688,-50}, {421,689,-50}, {421,694,-50}, {451,2,-35}, 
    {451,7,-35}, {451,18,-35}, {451,29,-35}, {451,35,-35}, 
    {451,45,-35}, {451,50,-35}, {451,572,-40}, {451,660,-50}, 
    {451,670,-35}, {451,680,-40}, {451,688,-50}, {451,689,-50}, 
    {451,694,-50}, {457,2,-35}, {457,7,-35}, {457,18,-35}, 
    {457,29,-35}, {457,35,-35}, {457,45,-35}, {457,50,-35}, 
    {457,572,-40}, {457,660,-50}, {457,670,-35}, {457,680,-40}, 
    {457,688,-50}, {457,689,-50}, {457,694,-50}, {461,2,-92}, 
    {461,7,-92}, {461,18,-92}, {461,29,-92}, {461,35,-92}, 
    {461,45,-92}, {461,50,-92}, {461,736,-15}, {461,738,-15}, 
    {461,760,-15}, {461,777,-15}, {461,1032,-15}, {461,1113,-15}, 
    {461,1175,-15}, {461,1411,-111}, {461,2998,-111}, {476,605,-10}, 
    {476,606,-10}, {476,611,-10}, {476,617,-10}, {476,626,-10}, 
    {480,395,-40}, {480,398,-40}, {480,406,-40}, {480,416,-40}, 
    {480,421,-40}, {480,451,-40}, {480,457,-40}, {480,572,-60}, 
    {480,605,-40}, {480,606,-40}, {480,611,-40}, {480,617,-40}, 
    {480,626,-40}, {480,660,-80}, {480,670,-55}, {480,688,-65}, 
    {480,689,-65}, {480,694,-65}, {572,2,-93}, {572,7,-93}, 
    {572,18,-93}, {572,29,-93}, {572,35,-93}, {572,45,-93}, 
    {572,50,-93}, {572,395,-18}, {572,398,-18}, {572,406,-18}, 
    {572,416,-18}, {572,421,-18}, {572,451,-18}, {572,457,-18}, 
    {572,736,-80}, {572,738,-80}, {572,760,-80}, {572,777,-40}, 
    {572,1032,-40}, {572,1113,-80}, {572,1175,-40}, {572,1404,-50}, 
    {572,1411,-74}, {572,1632,-70}, {572,1633,-70}, {572,1647,-70}, 
    {572,1657,-30}, {572,1664,-70}, {572,2128,-92}, {572,2134,-35}, 
    {572,2135,-35}, {572,2797,-80}, {572,2798,-80}, {572,2812,-80}, 
    {572,2822,-80}, {572,2829,-80}, {572,2902,-80}, {572,2910,-80}, 
    {572,2998,-74}, {572,3151,-35}, {572,3354,-55}, {572,3819,-45}, 
    {572,3820,-45}, {572,3827,-45}, {572,3834,-45}, {572,3842,-45}, 
    {572,3954,-80}, {572,4034,-80}, {572,4037,-80}, {572,4053,-80}, 
    {605,2,-40}, {605,7,-40}, {605,18,-40}, {605,29,-40}, {605,35,-40}, 
    {605,45,-40}, {605,50,-40}, {606,2,-40}, {606,7,-40}, {606,18,-40}, 
    {606,29,-40}, {606,35,-40}, {606,45,-40}, {606,50,-40}, 
    {611,2,-40}, {611,7,-40}, {611,18,-40}, {611,29,-40}, {611,35,-40}, 
    {611,45,-40}, {611,50,-40}, {617,2,-40}, {617,7,-40}, {617,18,-40}, 
    {617,29,-40}, {617,35,-40}, {617,45,-40}, {617,50,-40}, 
    {626,2,-40}, {626,7,-40}, {626,18,-40}, {626,29,-40}, {626,35,-40}, 
    {626,45,-40}, {626,50,-40}, {660,2,-135}, {660,7,-135}, 
    {660,18,-135}, {660,29,-135}, {660,35,-135}, {660,45,-135}, 
    {660,50,-135}, {660,205,-15}, {660,395,-40}, {660,398,-40}, 
    {660,406,-40}, {660,416,-40}, {660,421,-40}, {660,451,-40}, 
    {660,457,-40}, {660,736,-111}, {660,738,-111}, {660,760,-71}, 
    {660,777,-71}, {660,1032,-71}, {660,1113,-111}, {660,1175,-71}, 
    {660,1404,-74}, {660,1411,-129}, {660,1632,-111}, {660,1633,-111}, 
    {660,1647,-71}, {660,1657,-71}, {660,1664,-71}, {660,2128,-100}, 
    {660,2134,-60}, {660,2135,-60}, {660,2142,-20}, {660,2198,-20}, 
    {660,2209,-20}, {660,2797,-129}, {660,2798,-129}, {660,2812,-129}, 
    {660,2822,-89}, {660,2829,-89}, {660,2902,-129}, {660,2910,-89}, 
    {660,2998,-129}, {660,3354,-74}, {660,3819,-75}, {660,3820,-75}, 
    {660,3827,-75}, {660,3834,-75}, {660,3842,-75}, {670,2,-120}, 
    {670,7,-120}, {670,18,-120}, {670,29,-120}, {670,35,-120}, 
    {670,45,-120}, {670,50,-120}, {670,395,-10}, {670,398,-10}, 
    {670,406,-10}, {670,416,-10}, {670,421,-10}, {670,451,-10}, 
    {670,457,-10}, {670,736,-80}, {670,738,-80}, {670,760,-80}, 
    {670,777,-80}, {670,1032,-80}, {670,1113,-80}, {670,1175,-80}, 
    {670,1404,-37}, {670,1411,-92}, {670,1632,-80}, {670,1633,-80}, 
    {670,1647,-80}, {670,1657,-40}, {670,1664,-40}, {670,2128,-65}, 
    {670,2134,-40}, {670,2135,-40}, {670,2797,-80}, {670,2798,-80}, 
    {670,2812,-80}, {670,2822,-80}, {670,2829,-80}, {670,2902,-80}, 
    {670,2910,-80}, {670,2998,-92}, {670,3354,-37}, {670,3819,-50}, 
    {670,3820,-50}, {670,3827,-50}, {670,3834,-50}, {670,3842,-50}, 
    {670,4034,-73}, {670,4037,-73}, {670,4053,-73}, {688,2,-120}, 
    {688,7,-120}, {688,18,-120}, {688,29,-120}, {688,35,-120}, 
    {688,45,-120}, {688,50,-120}, {688,395,-30}, {688,398,-30}, 
    {688,406,-30}, {688,416,-30}, {688,421,-30}, {688,451,-30}, 
    {688,457,-30}, {688,736,-100}, {688,738,-100}, {688,760,-100}, 
    {688,777,-60}, {688,1032,-60}, {688,1113,-100}, {688,1175,-60}, 
    {688,1404,-92}, {688,1411,-129}, {688,1632,-100}, {688,1633,-100}, 
    {688,1647,-100}, {688,1657,-60}, {688,1664,-60}, {688,2128,-111}, 
    {688,2134,-55}, {688,2135,-55}, {688,2797,-110}, {688,2798,-110}, 
    {688,2812,-110}, {688,2822,-70}, {688,2829,-70}, {688,2902,-110}, 
    {688,2910,-70}, {688,2998,-129}, {688,3354,-92}, {688,3819,-111}, 
    {688,3820,-111}, {688,3827,-111}, {688,3834,-71}, {688,3842,-71}, 
    {689,2,-120}, {689,7,-120}, {689,18,-120}, {689,29,-120}, 
    {689,35,-120}, {689,45,-120}, {689,50,-120}, {689,395,-30}, 
    {689,398,-30}, {689,406,-30}, {689,416,-30}, {689,421,-30}, 
    {689,451,-30}, {689,457,-30}, {689,736,-100}, {689,738,-100}, 
    {689,760,-100}, {689,777,-60}, {689,1032,-60}, {689,1113,-100}, 
    {689,1175,-60}, {689,1404,-92}, {689,1411,-129}, {689,1632,-100}, 
    {689,1633,-100}, {689,1647,-100}, {689,1657,-60}, {689,1664,-60}, 
    {689,2128,-111}, {689,2134,-55}, {689,2135,-55}, {689,2797,-110}, 
    {689,2798,-110}, {689,2812,-110}, {689,2822,-70}, {689,2829,-70}, 
    {689,2902,-110}, {689,2910,-70}, {689,2998,-129}, {689,3354,-92}, 
    {689,3819,-111}, {689,3820,-111}, {689,3827,-111}, {689,3834,-71}, 
    {689,3842,-71}, {694,2,-120}, {694,7,-120}, {694,18,-120}, 
    {694,29,-120}, {694,35,-120}, {694,45,-120}, {694,50,-120}, 
    {694,395,-30}, {694,398,-30}, {694,406,-30}, {694,416,-30}, 
    {694,421,-30}, {694,451,-30}, {694,457,-30}, {694,736,-100}, 
    {694,738,-100}, {694,760,-100}, {694,777,-60}, {694,1032,-60}, 
    {694,1113,-100}, {694,1175,-100}, {694,1404,-92}, {694,1411,-129}, 
    {694,1632,-100}, {694,1633,-100}, {694,1647,-100}, {694,1657,-60}, 
    {694,1664,-60}, {694,2128,-111}, {694,2134,-55}, {694,2135,-55}, 
    {694,2797,-110}, {694,2798,-110}, {694,2812,-110}, {694,2822,-70}, 
    {694,2829,-70}, {694,2902,-110}, {694,2910,-70}, {694,2998,-129}, 
    {694,3354,-92}, {694,3819,-111}, {694,3820,-111}, {694,3827,-111}, 
    {694,3834,-71}, {694,3842,-71}, {736,3905,-20}, {736,3954,-15}, 
    {738,3905,-20}, {738,3954,-15}, {760,3905,-20}, {760,3954,-15}, 
    {777,3905,-20}, {777,3954,-15}, {1032,3905,-20}, {1032,3954,-15}, 
    {1113,3905,-20}, {1113,3954,-15}, {1175,3905,-20}, {1175,3954,-15}, 
    {1194,2998,-40}, {1194,3819,-20}, {1194,3820,-20}, {1194,3827,-20}, 
    {1194,3834,-20}, {1194,3842,-20}, {1194,3905,-15}, {1314,4034,-15}, 
    {1314,4037,-15}, {1314,4053,-15}, {1334,4034,-15}, {1334,4037,-15}, 
    {1334,4053,-15}, {1411,3142,-70}, {1411,3146,-70}, {1632,1888,-15}, 
    {1632,3905,-25}, {1632,3954,-25}, {1632,4023,-15}, {1632,4034,-15}, 
    {1632,4037,-15}, {1632,4053,-15}, {1633,1888,-15}, {1633,3905,-25}, 
    {1633,3954,-25}, {1633,4023,-15}, {1633,4034,-15}, {1633,4037,-15}, 
    {1633,4053,-15}, {1647,1888,-15}, {1647,3905,-25}, {1647,3954,-25}, 
    {1647,4023,-15}, {1647,4034,-15}, {1647,4037,-15}, {1647,4053,-15}, 
    {1657,1888,-15}, {1657,3905,-25}, {1657,3954,-25}, {1657,4023,-15}, 
    {1657,4034,-15}, {1657,4037,-15}, {1657,4053,-15}, {1664,1888,-15}, 
    {1664,3905,-25}, {1664,3954,-25}, {1664,4023,-15}, {1664,4034,-15}, 
    {1664,4037,-15}, {1664,4053,-15}, {1789,736,-10}, {1789,738,-10}, 
    {1789,760,-10}, {1789,777,-10}, {1789,1032,-10}, {1789,1113,-10}, 
    {1789,1175,-10}, {1789,1610,-50}, {1789,1789,-25}, {1789,2134,-20}, 
    {1789,2135,-20}, {1789,3146,55}, {1888,736,-5}, {1888,738,-5}, 
    {1888,760,-5}, {1888,777,-5}, {1888,1032,-5}, {1888,1113,-5}, 
    {1888,1175,-5}, {1990,4034,-5}, {1990,4037,-5}, {1990,4053,-5}, 
    {2134,3905,-25}, {2135,3905,-25}, {2142,3905,-25}, {2198,3905,-25}, 
    {2209,3905,-25}, {2304,1632,-10}, {2304,1633,-10}, {2304,1647,-10}, 
    {2304,1657,-10}, {2304,1664,-10}, {2304,2797,-10}, {2304,2798,-10}, 
    {2304,2812,-10}, {2304,2822,-10}, {2304,2829,-10}, {2304,2902,-10}, 
    {2304,2910,-10}, {2304,4034,-15}, {2304,4037,-15}, {2304,4053,-15}, 
    {2418,3954,-10}, {2503,3954,-10}, {2650,3905,-40}, {2650,4034,-15}, 
    {2650,4037,-15}, {2650,4053,-15}, {2772,3905,-40}, {2772,4034,-15}, 
    {2772,4037,-15}, {2772,4053,-15}, {2797,3905,-15}, {2797,3954,-25}, 
    {2797,4034,-10}, {2797,4037,-10}, {2797,4053,-10}, {2798,3905,-15}, 
    {2798,3954,-25}, {2798,4034,-10}, {2798,4037,-10}, {2798,4053,-10}, 
    {2812,3905,-15}, {2812,3954,-25}, {2812,4034,-10}, {2812,4037,-10}, 
    {2812,4053,-10}, {2822,3905,-15}, {2822,3954,-25}, {2822,4034,-10}, 
    {2822,4037,-10}, {2822,4053,-10}, {2829,3905,-15}, {2829,3954,-25}, 
    {2829,4034,-10}, {2829,4037,-10}, {2829,4053,-10}, {2902,3905,-15}, 
    {2902,3954,-25}, {2902,4034,-10}, {2902,4037,-10}, {2902,4053,-10}, 
    {2910,3905,-15}, {2910,3954,-25}, {2910,4034,-10}, {2910,4037,-10}, 
    {2910,4053,-10}, {2924,4034,-10}, {2924,4037,-10}, {2924,4053,-10}, 
    {2998,3142,-70}, {2998,3146,-70}, {3138,2,-80}, {3138,7,-80}, 
    {3138,18,-80}, {3138,29,-80}, {3138,35,-80}, {3138,45,-80}, 
    {3138,50,-80}, {3143,2,-80}, {3143,7,-80}, {3143,18,-80}, 
    {3143,29,-80}, {3143,35,-80}, {3143,45,-80}, {3143,50,-80}, 
    {3143,3143,-74}, {3146,1483,-50}, {3146,2418,-10}, {3146,2503,-10}, 
    {3146,3146,-74}, {3146,3151,-50}, {3146,3282,-55}, {3146,3320,-55}, 
    {3146,3501,-74}, {3146,3559,-18}, {3146,3905,-50}, {3151,1411,-40}, 
    {3151,1888,-18}, {3151,2128,-20}, {3151,2998,-55}, {3501,2,-55}, 
    {3501,7,-55}, {3501,18,-55}, {3501,29,-55}, {3501,35,-55}, 
    {3501,45,-55}, {3501,50,-55}, {3501,572,-18}, {3501,660,-50}, 
    {3501,670,-30}, {3501,688,-90}, {3501,689,-90}, {3501,694,-90}, 
    {3905,736,-25}, {3905,738,-25}, {3905,760,-25}, {3905,777,-25}, 
    {3905,1032,-25}, {3905,1113,-25}, {3905,1175,-25}, {3905,1411,-65}, 
    {3905,1632,-15}, {3905,1633,-15}, {3905,1647,-15}, {3905,1657,-15}, 
    {3905,1664,-15}, {3905,2797,-20}, {3905,2798,-20}, {3905,2812,-20}, 
    {3905,2822,-20}, {3905,2829,-20}, {3905,2902,-20}, {3905,2910,-20}, 
    {3905,2998,-65}, {3954,736,-10}, {3954,738,-10}, {3954,760,-10}, 
    {3954,777,-10}, {3954,1032,-10}, {3954,1113,-10}, {3954,1175,-10}, 
    {3954,1411,-65}, {3954,2797,-10}, {3954,2798,-10}, {3954,2812,-10}, 
    {3954,2822,-10}, {3954,2829,-10}, {3954,2902,-10}, {3954,2910,-10}, 
    {3954,2998,-65}, {4023,1632,-15}, {4023,1633,-15}, {4023,1647,-15}, 
    {4023,1657,-15}, {4023,1664,-15}, {4034,1411,-65}, {4034,2998,-65}, 
    {4037,1411,-65}, {4037,2998,-65}, {4053,1411,-65}, {4053,2998,-65}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature times_roman_ligs[] = {
    {1789,2134,1809}, {1789,2418,1852}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair times_italic_kerns[] = {
    {2,66,-30}, {2,72,-30}, {2,205,-35}, {2,395,-40}, {2,398,-40}, 
    {2,406,-40}, {2,416,-40}, {2,421,-40}, {2,451,-40}, {2,457,-40}, 
    {2,476,-40}, {2,572,-37}, {2,605,-50}, {2,606,-50}, {2,611,-50}, 
    {2,617,-50}, {2,626,-50}, {2,660,-105}, {2,670,-95}, {2,688,-55}, 
    {2,689,-55}, {2,694,-55}, {2,3146,-37}, {2,3819,-20}, {2,3820,-20}, 
    {2,3827,-20}, {2,3834,-20}, {2,3842,-20}, {2,3905,-55}, 
    {2,3954,-55}, {2,4034,-55}, {2,4037,-55}, {2,4053,-55}, {7,66,-30}, 
    {7,72,-30}, {7,205,-35}, {7,395,-40}, {7,398,-40}, {7,406,-40}, 
    {7,416,-40}, {7,421,-40}, {7,451,-40}, {7,457,-40}, {7,476,-40}, 
    {7,572,-37}, {7,605,-50}, {7,606,-50}, {7,611,-50}, {7,617,-50}, 
    {7,626,-50}, {7,660,-105}, {7,670,-95}, {7,688,-55}, {7,689,-55}, 
    {7,694,-55}, {7,3146,-37}, {7,3819,-20}, {7,3820,-20}, 
    {7,3827,-20}, {7,3834,-20}, {7,3842,-20}, {7,3905,-55}, 
    {7,3954,-55}, {7,4034,-55}, {7,4037,-55}, {7,4053,-55}, 
    {18,66,-30}, {18,72,-30}, {18,205,-35}, {18,395,-40}, {18,398,-40}, 
    {18,406,-40}, {18,416,-40}, {18,421,-40}, {18,451,-40}, 
    {18,457,-40}, {18,476,-40}, {18,572,-37}, {18,605,-50}, 
    {18,606,-50}, {18,611,-50}, {18,617,-50}, {18,626,-50}, 
    {18,660,-105}, {18,670,-95}, {18,688,-55}, {18,689,-55}, 
    {18,694,-55}, {18,3146,-37}, {18,3819,-20}, {18,3820,-20}, 
    {18,3827,-20}, {18,3834,-20}, {18,3842,-20}, {18,3905,-55}, 
    {18,3954,-55}, {18,4034,-55}, {18,4037,-55}, {18,4053,-55}, 
    {29,66,-30}, {29,72,-30}, {29,205,-35}, {29,395,-40}, {29,398,-40}, 
    {29,406,-40}, {29,416,-40}, {29,421,-40}, {29,451,-40}, 
    {29,457,-40}, {29,476,-40}, {29,572,-37}, {29,605,-50}, 
    {29,606,-50}, {29,611,-50}, {29,617,-50}, {29,626,-50}, 
    {29,660,-105}, {29,670,-95}, {29,688,-55}, {29,689,-55}, 
    {29,694,-55}, {29,3146,-37}, {29,3819,-20}, {29,3820,-20}, 
    {29,3827,-20}, {29,3834,-20}, {29,3842,-20}, {29,3905,-55}, 
    {29,3954,-55}, {29,4034,-55}, {29,4037,-55}, {29,4053,-55}, 
    {35,66,-30}, {35,72,-30}, {35,205,-35}, {35,395,-40}, {35,398,-40}, 
    {35,406,-40}, {35,416,-40}, {35,421,-40}, {35,451,-40}, 
    {35,457,-40}, {35,476,-40}, {35,572,-37}, {35,605,-50}, 
    {35,606,-50}, {35,611,-50}, {35,617,-50}, {35,626,-50}, 
    {35,660,-105}, {35,670,-95}, {35,688,-55}, {35,689,-55}, 
    {35,694,-55}, {35,3146,-37}, {35,3819,-20}, {35,3820,-20}, 
    {35,3827,-20}, {35,3834,-20}, {35,3842,-20}, {35,3905,-55}, 
    {35,3954,-55}, {35,4034,-55}, {35,4037,-55}, {35,4053,-55}, 
    {45,66,-30}, {45,72,-30}, {45,205,-35}, {45,395,-40}, {45,398,-40}, 
    {45,406,-40}, {45,416,-40}, {45,421,-40}, {45,451,-40}, 
    {45,457,-40}, {45,476,-40}, {45,572,-37}, {45,605,-50}, 
    {45,606,-50}, {45,611,-50}, {45,617,-50}, {45,626,-50}, 
    {45,660,-105}, {45,670,-95}, {45,688,-55}, {45,689,-55}, 
    {45,694,-55}, {45,3146,-37}, {45,3819,-20}, {45,3820,-20}, 
    {45,3827,-20}, {45,3834,-20}, {45,3842,-20}, {45,3905,-55}, 
    {45,3954,-55}, {45,4034,-55}, {45,4037,-55}, {45,4053,-55}, 
    {50,66,-30}, {50,72,-30}, {50,205,-35}, {50,395,-40}, {50,398,-40}, 
    {50,406,-40}, {50,416,-40}, {50,421,-40}, {50,451,-40}, 
    {50,457,-40}, {50,476,-40}, {50,572,-37}, {50,605,-50}, 
    {50,606,-50}, {50,611,-50}, {50,617,-50}, {50,626,-50}, 
    {50,660,-105}, {50,670,-95}, {50,688,-55}, {50,689,-55}, 
    {50,694,-55}, {50,3146,-37}, {50,3819,-20}, {50,3820,-20}, 
    {50,3827,-20}, {50,3834,-20}, {50,3842,-20}, {50,3905,-55}, 
    {50,3954,-55}, {50,4034,-55}, {50,4037,-55}, {50,4053,-55}, 
    {53,2,-25}, {53,7,-25}, {53,18,-25}, {53,29,-25}, {53,35,-25}, 
    {53,45,-25}, {53,50,-25}, {53,605,-10}, {53,606,-10}, {53,611,-10}, 
    {53,617,-10}, {53,626,-10}, {95,2,-35}, {95,7,-35}, {95,18,-35}, 
    {95,29,-35}, {95,35,-35}, {95,45,-35}, {95,50,-35}, {95,660,-40}, 
    {95,670,-40}, {95,688,-40}, {95,689,-40}, {95,694,-40}, 
    {194,2,-115}, {194,7,-115}, {194,18,-115}, {194,29,-115}, 
    {194,35,-115}, {194,45,-115}, {194,50,-115}, {194,736,-75}, 
    {194,738,-75}, {194,760,-75}, {194,777,-75}, {194,1032,-75}, 
    {194,1113,-75}, {194,1175,-75}, {194,1411,-135}, {194,1632,-75}, 
    {194,1633,-75}, {194,1647,-75}, {194,1657,-75}, {194,1664,-75}, 
    {194,2134,-45}, {194,2135,-45}, {194,2142,-45}, {194,2198,-45}, 
    {194,2209,-45}, {194,2797,-105}, {194,2798,-105}, {194,2812,-105}, 
    {194,2822,-105}, {194,2829,-105}, {194,2902,-105}, {194,2910,-105}, 
    {194,2998,-135}, {194,3151,-55}, {303,2,-40}, {303,7,-40}, 
    {303,18,-40}, {303,29,-40}, {303,35,-40}, {303,45,-40}, 
    {303,50,-40}, {303,736,-35}, {303,738,-35}, {303,760,-35}, 
    {303,777,-35}, {303,1032,-35}, {303,1113,-35}, {303,1175,-35}, 
    {303,1411,-25}, {303,1632,-25}, {303,1633,-25}, {303,1647,-25}, 
    {303,1657,-25}, {303,1664,-25}, {303,2797,-25}, {303,2798,-25}, 
    {303,2812,-25}, {303,2822,-25}, {303,2829,-25}, {303,2902,-25}, 
    {303,2910,-25}, {303,2998,-25}, {303,3819,-35}, {303,3820,-35}, 
    {303,3827,-35}, {303,3834,-35}, {303,3842,-35}, {311,395,-50}, 
    {311,398,-50}, {311,406,-50}, {311,416,-50}, {311,421,-50}, 
    {311,451,-50}, {311,457,-50}, {311,1632,-35}, {311,1633,-35}, 
    {311,1647,-35}, {311,1657,-35}, {311,1664,-35}, {311,2797,-40}, 
    {311,2798,-40}, {311,2812,-40}, {311,2822,-40}, {311,2829,-40}, 
    {311,2902,-40}, {311,2910,-40}, {311,3819,-40}, {311,3820,-40}, 
    {311,3827,-40}, {311,3834,-40}, {311,3842,-40}, {311,4034,-40}, 
    {311,4037,-40}, {311,4053,-40}, {339,572,-20}, {339,660,-55}, 
    {339,670,-55}, {339,688,-20}, {339,689,-20}, {339,694,-20}, 
    {339,3146,-37}, {339,4034,-30}, {339,4037,-30}, {339,4053,-30}, 
    {358,572,-20}, {358,660,-55}, {358,670,-55}, {358,688,-20}, 
    {358,689,-20}, {358,694,-20}, {358,3146,-37}, {358,4034,-30}, 
    {358,4037,-30}, {358,4053,-30}, {374,2,-27}, {374,7,-27}, 
    {374,18,-27}, {374,29,-27}, {374,35,-27}, {374,45,-27}, 
    {374,50,-27}, {392,2,-27}, {392,7,-27}, {392,18,-27}, {392,29,-27}, 
    {392,35,-27}, {392,45,-27}, {392,50,-27}, {395,2,-55}, {395,7,-55}, 
    {395,18,-55}, {395,29,-55}, {395,35,-55}, {395,45,-55}, 
    {395,50,-55}, {395,572,-40}, {395,660,-50}, {395,670,-50}, 
    {395,680,-40}, {395,688,-50}, {395,689,-50}, {395,694,-50}, 
    {398,2,-55}, {398,7,-55}, {398,18,-55}, {398,29,-55}, {398,35,-55}, 
    {398,45,-55}, {398,50,-55}, {398,572,-40}, {398,660,-50}, 
    {398,670,-50}, {398,680,-40}, {398,688,-50}, {398,689,-50}, 
    {398,694,-50}, {406,2,-55}, {406,7,-55}, {406,18,-55}, 
    {406,29,-55}, {406,35,-55}, {406,45,-55}, {406,50,-55}, 
    {406,572,-40}, {406,660,-50}, {406,670,-50}, {406,680,-40}, 
    {406,688,-50}, {406,689,-50}, {406,694,-50}, {416,2,-55}, 
    {416,7,-55}, {416,18,-55}, {416,29,-55}, {416,35,-55}, 
    {416,45,-55}, {416,50,-55}, {416,572,-40}, {416,660,-50}, 
    {416,670,-50}, {416,680,-40}, {416,688,-50}, {416,689,-50}, 
    {416,694,-50}, {421,2,-55}, {421,7,-55}, {421,18,-55}, 
    {421,29,-55}, {421,35,-55}, {421,45,-55}, {421,50,-55}, 
    {421,572,-40}, {421,660,-50}, {421,670,-50}, {421,680,-40}, 
    {421,688,-50}, {421,689,-50}, {421,694,-50}, {451,2,-55}, 
    {451,7,-55}, {451,18,-55}, {451,29,-55}, {451,35,-55}, 
    {451,45,-55}, {451,50,-55}, {451,572,-40}, {451,660,-50}, 
    {451,670,-50}, {451,680,-40}, {451,688,-50}, {451,689,-50}, 
    {451,694,-50}, {457,2,-55}, {457,7,-55}, {457,18,-55}, 
    {457,29,-55}, {457,35,-55}, {457,45,-55}, {457,50,-55}, 
    {457,572,-40}, {457,660,-50}, {457,670,-50}, {457,680,-40}, 
    {457,688,-50}, {457,689,-50}, {457,694,-50}, {461,2,-90}, 
    {461,7,-90}, {461,18,-90}, {461,29,-90}, {461,35,-90}, 
    {461,45,-90}, {461,50,-90}, {461,736,-80}, {461,738,-80}, 
    {461,760,-80}, {461,777,-80}, {461,1032,-80}, {461,1113,-80}, 
    {461,1175,-80}, {461,1411,-135}, {461,1632,-80}, {461,1633,-80}, 
    {461,1647,-80}, {461,1657,-80}, {461,1664,-80}, {461,2797,-80}, 
    {461,2798,-80}, {461,2812,-80}, {461,2822,-80}, {461,2829,-80}, 
    {461,2902,-80}, {461,2910,-80}, {461,2998,-135}, {476,605,-10}, 
    {476,606,-10}, {476,611,-10}, {476,617,-10}, {476,626,-10}, 
    {480,395,-40}, {480,398,-40}, {480,406,-40}, {480,416,-40}, 
    {480,421,-40}, {480,451,-40}, {480,457,-40}, {480,605,-40}, 
    {480,606,-40}, {480,611,-40}, {480,617,-40}, {480,626,-40}, 
    {480,660,-18}, {480,670,-18}, {480,688,-18}, {480,689,-18}, 
    {480,694,-18}, {572,2,-50}, {572,7,-50}, {572,18,-50}, 
    {572,29,-50}, {572,35,-50}, {572,45,-50}, {572,50,-50}, 
    {572,395,-18}, {572,398,-18}, {572,406,-18}, {572,416,-18}, 
    {572,421,-18}, {572,451,-18}, {572,457,-18}, {572,736,-92}, 
    {572,738,-92}, {572,760,-92}, {572,777,-92}, {572,1032,-92}, 
    {572,1113,-92}, {572,1175,-92}, {572,1404,-55}, {572,1411,-74}, 
    {572,1632,-92}, {572,1633,-92}, {572,1647,-52}, {572,1657,-52}, 
    {572,1664,-52}, {572,2128,-74}, {572,2134,-55}, {572,2135,-55}, 
    {572,2797,-92}, {572,2798,-92}, {572,2812,-92}, {572,2822,-92}, 
    {572,2829,-92}, {572,2902,-92}, {572,2910,-92}, {572,2998,-74}, 
    {572,3151,-55}, {572,3354,-65}, {572,3819,-55}, {572,3820,-55}, 
    {572,3827,-55}, {572,3834,-55}, {572,3842,-55}, {572,3954,-74}, 
    {572,4034,-74}, {572,4037,-74}, {572,4053,-34}, {605,2,-40}, 
    {605,7,-40}, {605,18,-40}, {605,29,-40}, {605,35,-40}, 
    {605,45,-40}, {605,50,-40}, {605,1411,-25}, {605,2998,-25}, 
    {606,2,-40}, {606,7,-40}, {606,18,-40}, {606,29,-40}, {606,35,-40}, 
    {606,45,-40}, {606,50,-40}, {606,1411,-25}, {606,2998,-25}, 
    {611,2,-40}, {611,7,-40}, {611,18,-40}, {611,29,-40}, {611,35,-40}, 
    {611,45,-40}, {611,50,-40}, {611,1411,-25}, {611,2998,-25}, 
    {617,2,-40}, {617,7,-40}, {617,18,-40}, {617,29,-40}, {617,35,-40}, 
    {617,45,-40}, {617,50,-40}, {617,1411,-25}, {617,2998,-25}, 
    {626,2,-40}, {626,7,-40}, {626,18,-40}, {626,29,-40}, {626,35,-40}, 
    {626,45,-40}, {626,50,-40}, {626,1411,-25}, {626,2998,-25}, 
    {660,2,-60}, {660,7,-60}, {660,18,-60}, {660,29,-60}, {660,35,-60}, 
    {660,45,-60}, {660,50,-60}, {660,395,-30}, {660,398,-30}, 
    {660,406,-30}, {660,416,-30}, {660,421,-30}, {660,451,-30}, 
    {660,457,-30}, {660,736,-111}, {660,738,-111}, {660,760,-111}, 
    {660,777,-111}, {660,1032,-111}, {660,1113,-111}, {660,1175,-111}, 
    {660,1404,-65}, {660,1411,-129}, {660,1632,-111}, {660,1633,-111}, 
    {660,1647,-111}, {660,1657,-71}, {660,1664,-71}, {660,2128,-55}, 
    {660,2134,-74}, {660,2135,-74}, {660,2142,-34}, {660,2198,-34}, 
    {660,2209,-34}, {660,2797,-111}, {660,2798,-111}, {660,2812,-111}, 
    {660,2822,-111}, {660,2829,-111}, {660,2902,-111}, {660,2910,-111}, 
    {660,2998,-129}, {660,3354,-74}, {660,3819,-74}, {660,3820,-74}, 
    {660,3827,-74}, {660,3834,-74}, {660,3842,-74}, {670,2,-60}, 
    {670,7,-60}, {670,18,-60}, {670,29,-60}, {670,35,-60}, 
    {670,45,-60}, {670,50,-60}, {670,395,-25}, {670,398,-25}, 
    {670,406,-25}, {670,416,-25}, {670,421,-25}, {670,451,-25}, 
    {670,457,-25}, {670,736,-92}, {670,738,-92}, {670,760,-92}, 
    {670,777,-92}, {670,1032,-92}, {670,1113,-92}, {670,1175,-92}, 
    {670,1404,-65}, {670,1411,-92}, {670,1632,-92}, {670,1633,-92}, 
    {670,1647,-92}, {670,1657,-52}, {670,1664,-52}, {670,2128,-37}, 
    {670,2134,-55}, {670,2135,-55}, {670,2797,-92}, {670,2798,-92}, 
    {670,2812,-92}, {670,2822,-92}, {670,2829,-92}, {670,2902,-92}, 
    {670,2910,-92}, {670,2998,-92}, {670,3354,-65}, {670,3819,-55}, 
    {670,3820,-55}, {670,3827,-55}, {670,3834,-55}, {670,3842,-55}, 
    {670,4034,-70}, {670,4037,-70}, {670,4053,-70}, {688,2,-50}, 
    {688,7,-50}, {688,18,-50}, {688,29,-50}, {688,35,-50}, 
    {688,45,-50}, {688,50,-50}, {688,395,-15}, {688,398,-15}, 
    {688,406,-15}, {688,416,-15}, {688,421,-15}, {688,451,-15}, 
    {688,457,-15}, {688,736,-92}, {688,738,-92}, {688,760,-92}, 
    {688,777,-92}, {688,1032,-92}, {688,1113,-92}, {688,1175,-92}, 
    {688,1404,-65}, {688,1411,-92}, {688,1632,-92}, {688,1633,-92}, 
    {688,1647,-92}, {688,1657,-52}, {688,1664,-52}, {688,2128,-74}, 
    {688,2134,-74}, {688,2135,-74}, {688,2142,-34}, {688,2198,-34}, 
    {688,2209,-34}, {688,2797,-92}, {688,2798,-92}, {688,2812,-92}, 
    {688,2822,-92}, {688,2829,-92}, {688,2902,-92}, {688,2910,-92}, 
    {688,2998,-92}, {688,3354,-65}, {688,3819,-92}, {688,3820,-92}, 
    {688,3827,-92}, {688,3834,-92}, {688,3842,-92}, {689,2,-50}, 
    {689,7,-50}, {689,18,-50}, {689,29,-50}, {689,35,-50}, 
    {689,45,-50}, {689,50,-50}, {689,395,-15}, {689,398,-15}, 
    {689,406,-15}, {689,416,-15}, {689,421,-15}, {689,451,-15}, 
    {689,457,-15}, {689,736,-92}, {689,738,-92}, {689,760,-92}, 
    {689,777,-92}, {689,1032,-92}, {689,1113,-92}, {689,1175,-92}, 
    {689,1404,-65}, {689,1411,-92}, {689,1632,-92}, {689,1633,-92}, 
    {689,1647,-92}, {689,1657,-52}, {689,1664,-52}, {689,2128,-74}, 
    {689,2134,-74}, {689,2135,-74}, {689,2142,-34}, {689,2198,-34}, 
    {689,2209,-34}, {689,2797,-92}, {689,2798,-92}, {689,2812,-92}, 
    {689,2822,-92}, {689,2829,-92}, {689,2902,-92}, {689,2910,-92}, 
    {689,2998,-92}, {689,3354,-65}, {689,3819,-92}, {689,3820,-92}, 
    {689,3827,-92}, {689,3834,-92}, {689,3842,-92}, {694,2,-50}, 
    {694,7,-50}, {694,18,-50}, {694,29,-50}, {694,35,-50}, 
    {694,45,-50}, {694,50,-50}, {694,395,-15}, {694,398,-15}, 
    {694,406,-15}, {694,416,-15}, {694,421,-15}, {694,451,-15}, 
    {694,457,-15}, {694,736,-92}, {694,738,-92}, {694,760,-92}, 
    {694,777,-92}, {694,1032,-92}, {694,1113,-92}, {694,1175,-92}, 
    {694,1404,-65}, {694,1411,-92}, {694,1632,-92}, {694,1633,-92}, 
    {694,1647,-92}, {694,1657,-52}, {694,1664,-52}, {694,2128,-74}, 
    {694,2134,-74}, {694,2135,-74}, {694,2142,-34}, {694,2198,-34}, 
    {694,2209,-34}, {694,2797,-92}, {694,2798,-92}, {694,2812,-92}, 
    {694,2822,-92}, {694,2829,-92}, {694,2902,-92}, {694,2910,-92}, 
    {694,2998,-92}, {694,3354,-65}, {694,3819,-92}, {694,3820,-92}, 
    {694,3827,-92}, {694,3834,-92}, {694,3842,-92}, {736,1888,-10}, 
    {738,1888,-10}, {760,1888,-10}, {777,1888,-10}, {1032,1888,-10}, 
    {1113,1888,-10}, {1175,1888,-10}, {1194,2998,-40}, {1194,3819,-20}, 
    {1194,3820,-20}, {1194,3827,-20}, {1194,3834,-20}, {1194,3842,-20}, 
    {1314,1990,-15}, {1314,2304,-20}, {1334,1990,-15}, {1334,2304,-20}, 
    {1411,3142,-140}, {1411,3146,-140}, {1632,1411,-10}, 
    {1632,1888,-40}, {1632,2998,-15}, {1632,3905,-15}, {1632,3954,-15}, 
    {1632,4023,-20}, {1632,4034,-30}, {1632,4037,-30}, {1632,4053,-30}, 
    {1633,1411,-10}, {1633,1888,-40}, {1633,2998,-15}, {1633,3905,-15}, 
    {1633,3954,-15}, {1633,4023,-20}, {1633,4034,-30}, {1633,4037,-30}, 
    {1633,4053,-30}, {1647,1411,-10}, {1647,1888,-40}, {1647,2998,-15}, 
    {1647,3905,-15}, {1647,3954,-15}, {1647,4023,-20}, {1647,4034,-30}, 
    {1647,4037,-30}, {1647,4053,-30}, {1657,1411,-10}, {1657,1888,-40}, 
    {1657,2998,-15}, {1657,3905,-15}, {1657,3954,-15}, {1657,4023,-20}, 
    {1657,4034,-30}, {1657,4037,-30}, {1657,4053,-30}, {1664,1411,-10}, 
    {1664,1888,-40}, {1664,2998,-15}, {1664,3905,-15}, {1664,3954,-15}, 
    {1664,4023,-20}, {1664,4034,-30}, {1664,4037,-30}, {1664,4053,-30}, 
    {1789,1411,-10}, {1789,1610,-60}, {1789,1789,-18}, {1789,2134,-20}, 
    {1789,2998,-15}, {1789,3146,92}, {1888,1411,-10}, {1888,1632,-10}, 
    {1888,1633,-10}, {1888,1647,-10}, {1888,1657,-10}, {1888,1664,-10}, 
    {1888,1888,-10}, {1888,2998,-15}, {2304,1632,-10}, {2304,1633,-10}, 
    {2304,1647,-10}, {2304,1657,-10}, {2304,1664,-10}, {2304,2797,-10}, 
    {2304,2798,-10}, {2304,2812,-10}, {2304,2822,-10}, {2304,2829,-10}, 
    {2304,2902,-10}, {2304,2910,-10}, {2304,4034,-10}, {2304,4037,-10}, 
    {2304,4053,-10}, {2650,3905,-40}, {2772,3905,-40}, {2797,1888,-10}, 
    {2797,3905,-10}, {2798,1888,-10}, {2798,3905,-10}, {2812,1888,-10}, 
    {2812,3905,-10}, {2822,1888,-10}, {2822,3905,-10}, {2829,1888,-10}, 
    {2829,3905,-10}, {2902,1888,-10}, {2902,3905,-10}, {2910,1888,-10}, 
    {2910,3905,-10}, {2998,3142,-140}, {2998,3146,-140}, 
    {3143,3143,-111}, {3146,1483,-25}, {3146,3146,-111}, 
    {3146,3151,-25}, {3146,3282,-40}, {3146,3320,-40}, 
    {3146,3501,-111}, {3146,3559,-30}, {3146,3905,-10}, {3151,736,-15}, 
    {3151,738,-15}, {3151,760,-15}, {3151,777,-15}, {3151,1032,-15}, 
    {3151,1113,-15}, {3151,1175,-15}, {3151,1314,-37}, {3151,1334,-37}, 
    {3151,1411,-111}, {3151,1483,-37}, {3151,1632,-37}, 
    {3151,1633,-37}, {3151,1647,-37}, {3151,1657,-37}, {3151,1664,-37}, 
    {3151,1888,-37}, {3151,2128,-20}, {3151,2797,-45}, {3151,2798,-45}, 
    {3151,2812,-45}, {3151,2822,-45}, {3151,2829,-45}, {3151,2902,-45}, 
    {3151,2910,-45}, {3151,2998,-111}, {3151,3086,-37}, 
    {3151,3282,-10}, {3151,3320,-10}, {3501,2,-18}, {3501,7,-18}, 
    {3501,18,-18}, {3501,29,-18}, {3501,35,-18}, {3501,45,-18}, 
    {3501,50,-18}, {3501,572,-18}, {3501,660,-35}, {3501,670,-40}, 
    {3501,688,-75}, {3501,689,-75}, {3501,694,-75}, {3905,1411,-74}, 
    {3905,2998,-74}, {3954,1411,-74}, {3954,2998,-74}, {4034,1411,-55}, 
    {4034,2998,-55}, {4037,1411,-55}, {4037,2998,-55}, {4053,1411,-55}, 
    {4053,2998,-55}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature times_italic_ligs[] = {
    {1789,2134,1809}, {1789,2418,1852}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair times_bold_kerns[] = {
    {2,66,-55}, {2,72,-55}, {2,205,-55}, {2,395,-45}, {2,398,-45}, 
    {2,406,-45}, {2,416,-45}, {2,421,-45}, {2,451,-45}, {2,457,-45}, 
    {2,476,-45}, {2,572,-95}, {2,605,-50}, {2,606,-50}, {2,611,-50}, 
    {2,617,-50}, {2,626,-50}, {2,660,-145}, {2,670,-130}, {2,688,-100}, 
    {2,689,-100}, {2,694,-100}, {2,2924,-25}, {2,3146,-74}, 
    {2,3819,-50}, {2,3820,-50}, {2,3827,-50}, {2,3834,-50}, 
    {2,3842,-50}, {2,3905,-100}, {2,3954,-90}, {2,4034,-74}, 
    {2,4037,-74}, {2,4053,-74}, {7,66,-55}, {7,72,-55}, {7,205,-55}, 
    {7,395,-45}, {7,398,-45}, {7,406,-45}, {7,416,-45}, {7,421,-45}, 
    {7,451,-45}, {7,457,-45}, {7,476,-45}, {7,572,-95}, {7,605,-50}, 
    {7,606,-50}, {7,611,-50}, {7,617,-50}, {7,626,-50}, {7,660,-145}, 
    {7,670,-130}, {7,688,-100}, {7,689,-100}, {7,694,-100}, 
    {7,2924,-25}, {7,3146,-74}, {7,3819,-50}, {7,3820,-50}, 
    {7,3827,-50}, {7,3834,-50}, {7,3842,-50}, {7,3905,-100}, 
    {7,3954,-90}, {7,4034,-74}, {7,4037,-74}, {7,4053,-74}, 
    {18,66,-55}, {18,72,-55}, {18,205,-55}, {18,395,-45}, {18,398,-45}, 
    {18,406,-45}, {18,416,-45}, {18,421,-45}, {18,451,-45}, 
    {18,457,-45}, {18,476,-45}, {18,572,-95}, {18,605,-50}, 
    {18,606,-50}, {18,611,-50}, {18,617,-50}, {18,626,-50}, 
    {18,660,-145}, {18,670,-130}, {18,688,-100}, {18,689,-100}, 
    {18,694,-100}, {18,2924,-25}, {18,3146,-74}, {18,3819,-50}, 
    {18,3820,-50}, {18,3827,-50}, {18,3834,-50}, {18,3842,-50}, 
    {18,3905,-100}, {18,3954,-90}, {18,4034,-74}, {18,4037,-74}, 
    {18,4053,-74}, {29,66,-55}, {29,72,-55}, {29,205,-55}, 
    {29,395,-45}, {29,398,-45}, {29,406,-45}, {29,416,-45}, 
    {29,421,-45}, {29,451,-45}, {29,457,-45}, {29,476,-45}, 
    {29,572,-95}, {29,605,-50}, {29,606,-50}, {29,611,-50}, 
    {29,617,-50}, {29,626,-50}, {29,660,-145}, {29,670,-130}, 
    {29,688,-100}, {29,689,-100}, {29,694,-100}, {29,2924,-25}, 
    {29,3146,-74}, {29,3819,-50}, {29,3820,-50}, {29,3827,-50}, 
    {29,3834,-50}, {29,3842,-50}, {29,3905,-100}, {29,3954,-90}, 
    {29,4034,-74}, {29,4037,-74}, {29,4053,-74}, {35,66,-55}, 
    {35,72,-55}, {35,205,-55}, {35,395,-45}, {35,398,-45}, 
    {35,406,-45}, {35,416,-45}, {35,421,-45}, {35,451,-45}, 
    {35,457,-45}, {35,476,-45}, {35,572,-95}, {35,605,-50}, 
    {35,606,-50}, {35,611,-50}, {35,617,-50}, {35,626,-50}, 
    {35,660,-145}, {35,670,-130}, {35,688,-100}, {35,689,-100}, 
    {35,694,-100}, {35,2924,-25}, {35,3146,-74}, {35,3819,-50}, 
    {35,3820,-50}, {35,3827,-50}, {35,3834,-50}, {35,3842,-50}, 
    {35,3905,-100}, {35,3954,-90}, {35,4034,-74}, {35,4037,-74}, 
    {35,4053,-74}, {45,66,-55}, {45,72,-55}, {45,205,-55}, 
    {45,395,-45}, {45,398,-45}, {45,406,-45}, {45,416,-45}, 
    {45,421,-45}, {45,451,-45}, {45,457,-45}, {45,476,-45}, 
    {45,572,-95}, {45,605,-50}, {45,606,-50}, {45,611,-50}, 
    {45,617,-50}, {45,626,-50}, {45,660,-145}, {45,670,-130}, 
    {45,688,-100}, {45,689,-100}, {45,694,-100}, {45,2924,-25}, 
    {45,3146,-74}, {45,3819,-50}, {45,3820,-50}, {45,3827,-50}, 
    {45,3834,-50}, {45,3842,-50}, {45,3905,-100}, {45,3954,-90}, 
    {45,4034,-74}, {45,4037,-74}, {45,4053,-74}, {50,66,-55}, 
    {50,72,-55}, {50,205,-55}, {50,395,-45}, {50,398,-45}, 
    {50,406,-45}, {50,416,-45}, {50,421,-45}, {50,451,-45}, 
    {50,457,-45}, {50,476,-45}, {50,572,-95}, {50,605,-50}, 
    {50,606,-50}, {50,611,-50}, {50,617,-50}, {50,626,-50}, 
    {50,660,-145}, {50,670,-130}, {50,688,-100}, {50,689,-100}, 
    {50,694,-100}, {50,2924,-25}, {50,3146,-74}, {50,3819,-50}, 
    {50,3820,-50}, {50,3827,-50}, {50,3834,-50}, {50,3842,-50}, 
    {50,3905,-100}, {50,3954,-90}, {50,4034,-74}, {50,4037,-74}, 
    {50,4053,-74}, {53,2,-30}, {53,7,-30}, {53,18,-30}, {53,29,-30}, 
    {53,35,-30}, {53,45,-30}, {53,50,-30}, {53,605,-10}, {53,606,-10}, 
    {53,611,-10}, {53,617,-10}, {53,626,-10}, {95,2,-35}, {95,7,-35}, 
    {95,18,-35}, {95,29,-35}, {95,35,-35}, {95,45,-35}, {95,50,-35}, 
    {95,660,-40}, {95,670,-40}, {95,688,-40}, {95,689,-40}, 
    {95,694,-40}, {95,2998,-20}, {194,2,-90}, {194,7,-90}, 
    {194,18,-90}, {194,29,-90}, {194,35,-90}, {194,45,-90}, 
    {194,50,-90}, {194,736,-25}, {194,738,-25}, {194,760,-25}, 
    {194,777,-25}, {194,1032,-25}, {194,1113,-25}, {194,1175,-25}, 
    {194,1411,-92}, {194,1632,-25}, {194,1633,-25}, {194,1647,-25}, 
    {194,1657,-25}, {194,1664,-25}, {194,2797,-25}, {194,2798,-25}, 
    {194,2812,-25}, {194,2822,-25}, {194,2829,-25}, {194,2902,-25}, 
    {194,2910,-25}, {194,2998,-110}, {303,2,-30}, {303,7,-30}, 
    {303,18,-30}, {303,29,-30}, {303,35,-30}, {303,45,-30}, 
    {303,50,-30}, {303,736,-15}, {303,738,-15}, {303,760,-15}, 
    {303,777,-15}, {303,1032,-15}, {303,1113,-15}, {303,1175,-15}, 
    {303,1632,-15}, {303,1633,-15}, {303,1647,-15}, {303,1657,-15}, 
    {303,1664,-15}, {303,2797,-15}, {303,2798,-15}, {303,2812,-15}, 
    {303,2822,-15}, {303,2829,-15}, {303,2902,-15}, {303,2910,-15}, 
    {303,2998,-20}, {303,3819,-15}, {303,3820,-15}, {303,3827,-15}, 
    {303,3834,-15}, {303,3842,-15}, {311,395,-30}, {311,398,-30}, 
    {311,406,-30}, {311,416,-30}, {311,421,-30}, {311,451,-30}, 
    {311,457,-30}, {311,1632,-25}, {311,1633,-25}, {311,1647,-25}, 
    {311,1657,-25}, {311,1664,-25}, {311,2797,-25}, {311,2798,-25}, 
    {311,2812,-25}, {311,2822,-25}, {311,2829,-25}, {311,2902,-25}, 
    {311,2910,-25}, {311,3819,-15}, {311,3820,-15}, {311,3827,-15}, 
    {311,3834,-15}, {311,3842,-15}, {311,4034,-45}, {311,4037,-45}, 
    {311,4053,-45}, {339,572,-92}, {339,660,-92}, {339,670,-92}, 
    {339,688,-92}, {339,689,-92}, {339,694,-92}, {339,3142,-20}, 
    {339,3146,-110}, {339,4034,-55}, {339,4037,-55}, {339,4053,-55}, 
    {358,572,-92}, {358,660,-92}, {358,670,-92}, {358,688,-92}, 
    {358,689,-92}, {358,694,-92}, {358,3142,-20}, {358,3146,-110}, 
    {358,4034,-55}, {358,4037,-55}, {358,4053,-55}, {374,2,-20}, 
    {374,7,-20}, {374,18,-20}, {374,29,-20}, {374,35,-20}, 
    {374,45,-20}, {374,50,-20}, {392,2,-20}, {392,7,-20}, {392,18,-20}, 
    {392,29,-20}, {392,35,-20}, {392,45,-20}, {392,50,-20}, 
    {395,2,-40}, {395,7,-40}, {395,18,-40}, {395,29,-40}, {395,35,-40}, 
    {395,45,-40}, {395,50,-40}, {395,572,-40}, {395,660,-50}, 
    {395,670,-50}, {395,680,-40}, {395,688,-50}, {395,689,-50}, 
    {395,694,-50}, {398,2,-40}, {398,7,-40}, {398,18,-40}, 
    {398,29,-40}, {398,35,-40}, {398,45,-40}, {398,50,-40}, 
    {398,572,-40}, {398,660,-50}, {398,670,-50}, {398,680,-40}, 
    {398,688,-50}, {398,689,-50}, {398,694,-50}, {406,2,-40}, 
    {406,7,-40}, {406,18,-40}, {406,29,-40}, {406,35,-40}, 
    {406,45,-40}, {406,50,-40}, {406,572,-40}, {406,660,-50}, 
    {406,670,-50}, {406,680,-40}, {406,688,-50}, {406,689,-50}, 
    {406,694,-50}, {416,2,-40}, {416,7,-40}, {416,18,-40}, 
    {416,29,-40}, {416,35,-40}, {416,45,-40}, {416,50,-40}, 
    {416,572,-40}, {416,660,-50}, {416,670,-50}, {416,680,-40}, 
    {416,688,-50}, {416,689,-50}, {416,694,-50}, {421,2,-40}, 
    {421,7,-40}, {421,18,-40}, {421,29,-40}, {421,35,-40}, 
    {421,45,-40}, {421,50,-40}, {421,572,-40}, {421,660,-50}, 
    {421,670,-50}, {421,680,-40}, {421,688,-50}, {421,689,-50}, 
    {421,694,-50}, {451,2,-40}, {451,7,-40}, {451,18,-40}, 
    {451,29,-40}, {451,35,-40}, {451,45,-40}, {451,50,-40}, 
    {451,572,-40}, {451,660,-50}, {451,670,-50}, {451,680,-40}, 
    {451,688,-50}, {451,689,-50}, {451,694,-50}, {457,2,-40}, 
    {457,7,-40}, {457,18,-40}, {457,29,-40}, {457,35,-40}, 
    {457,45,-40}, {457,50,-40}, {457,572,-40}, {457,660,-50}, 
    {457,670,-50}, {457,680,-40}, {457,688,-50}, {457,689,-50}, 
    {457,694,-50}, {461,2,-74}, {461,7,-74}, {461,18,-74}, 
    {461,29,-74}, {461,35,-74}, {461,45,-74}, {461,50,-74}, 
    {461,736,-10}, {461,738,-10}, {461,760,-10}, {461,777,-10}, 
    {461,1032,-10}, {461,1113,-10}, {461,1175,-10}, {461,1411,-92}, 
    {461,1632,-20}, {461,1633,-20}, {461,1647,-20}, {461,1657,-20}, 
    {461,1664,-20}, {461,2797,-20}, {461,2798,-20}, {461,2812,-20}, 
    {461,2822,-20}, {461,2829,-20}, {461,2902,-20}, {461,2910,-20}, 
    {461,2998,-110}, {476,605,-10}, {476,606,-10}, {476,611,-10}, 
    {476,617,-10}, {476,626,-10}, {476,2998,-20}, {480,395,-30}, 
    {480,398,-30}, {480,406,-30}, {480,416,-30}, {480,421,-30}, 
    {480,451,-30}, {480,457,-30}, {480,572,-40}, {480,605,-30}, 
    {480,606,-30}, {480,611,-30}, {480,617,-30}, {480,626,-30}, 
    {480,660,-55}, {480,670,-35}, {480,688,-35}, {480,689,-35}, 
    {480,694,-35}, {572,2,-90}, {572,7,-90}, {572,18,-90}, 
    {572,29,-90}, {572,35,-90}, {572,45,-90}, {572,50,-90}, 
    {572,395,-18}, {572,398,-18}, {572,406,-18}, {572,416,-18}, 
    {572,421,-18}, {572,451,-18}, {572,457,-18}, {572,736,-92}, 
    {572,738,-92}, {572,760,-52}, {572,777,-52}, {572,1032,-52}, 
    {572,1113,-92}, {572,1175,-52}, {572,1404,-74}, {572,1411,-74}, 
    {572,1632,-92}, {572,1633,-92}, {572,1647,-92}, {572,1657,-52}, 
    {572,1664,-52}, {572,2128,-92}, {572,2134,-18}, {572,2135,-18}, 
    {572,2797,-92}, {572,2798,-92}, {572,2812,-92}, {572,2822,-92}, 
    {572,2829,-92}, {572,2902,-92}, {572,2910,-92}, {572,2998,-90}, 
    {572,3151,-74}, {572,3354,-74}, {572,3819,-92}, {572,3820,-92}, 
    {572,3827,-92}, {572,3834,-92}, {572,3842,-92}, {572,3954,-74}, 
    {572,4034,-34}, {572,4037,-34}, {572,4053,-34}, {605,2,-60}, 
    {605,7,-60}, {605,18,-60}, {605,29,-60}, {605,35,-60}, 
    {605,45,-60}, {605,50,-60}, {605,1411,-50}, {605,2998,-50}, 
    {606,2,-60}, {606,7,-60}, {606,18,-60}, {606,29,-60}, {606,35,-60}, 
    {606,45,-60}, {606,50,-60}, {606,1411,-50}, {606,2998,-50}, 
    {611,2,-60}, {611,7,-60}, {611,18,-60}, {611,29,-60}, {611,35,-60}, 
    {611,45,-60}, {611,50,-60}, {611,1411,-50}, {611,2998,-50}, 
    {617,2,-60}, {617,7,-60}, {617,18,-60}, {617,29,-60}, {617,35,-60}, 
    {617,45,-60}, {617,50,-60}, {617,1411,-50}, {617,2998,-50}, 
    {626,2,-60}, {626,7,-60}, {626,18,-60}, {626,29,-60}, {626,35,-60}, 
    {626,45,-60}, {626,50,-60}, {626,1411,-50}, {626,2998,-50}, 
    {660,2,-135}, {660,7,-135}, {660,18,-135}, {660,29,-135}, 
    {660,35,-135}, {660,45,-135}, {660,50,-135}, {660,205,-30}, 
    {660,395,-45}, {660,398,-45}, {660,406,-45}, {660,416,-45}, 
    {660,421,-45}, {660,451,-45}, {660,457,-45}, {660,736,-92}, 
    {660,738,-92}, {660,760,-92}, {660,777,-92}, {660,1032,-92}, 
    {660,1113,-92}, {660,1175,-92}, {660,1404,-92}, {660,1411,-129}, 
    {660,1632,-100}, {660,1633,-100}, {660,1647,-100}, {660,1657,-100}, 
    {660,1664,-100}, {660,2128,-74}, {660,2134,-37}, {660,2135,-37}, 
    {660,2142,-37}, {660,2198,-37}, {660,2209,-37}, {660,2797,-100}, 
    {660,2798,-100}, {660,2812,-100}, {660,2822,-100}, {660,2829,-100}, 
    {660,2902,-100}, {660,2910,-100}, {660,2998,-145}, {660,3354,-92}, 
    {660,3819,-92}, {660,3820,-92}, {660,3827,-92}, {660,3834,-92}, 
    {660,3842,-92}, {670,2,-120}, {670,7,-120}, {670,18,-120}, 
    {670,29,-120}, {670,35,-120}, {670,45,-120}, {670,50,-120}, 
    {670,395,-10}, {670,398,-10}, {670,406,-10}, {670,416,-10}, 
    {670,421,-10}, {670,451,-10}, {670,457,-10}, {670,736,-65}, 
    {670,738,-65}, {670,760,-65}, {670,777,-65}, {670,1032,-65}, 
    {670,1113,-65}, {670,1175,-65}, {670,1404,-55}, {670,1411,-92}, 
    {670,1632,-65}, {670,1633,-65}, {670,1647,-65}, {670,1657,-65}, 
    {670,1664,-65}, {670,2128,-37}, {670,2134,-18}, {670,2135,-18}, 
    {670,2797,-75}, {670,2798,-75}, {670,2812,-75}, {670,2822,-75}, 
    {670,2829,-75}, {670,2902,-75}, {670,2910,-75}, {670,2998,-92}, 
    {670,3354,-55}, {670,3819,-50}, {670,3820,-50}, {670,3827,-50}, 
    {670,3834,-50}, {670,3842,-50}, {670,4034,-60}, {670,4037,-60}, 
    {670,4053,-60}, {688,2,-110}, {688,7,-110}, {688,18,-110}, 
    {688,29,-110}, {688,35,-110}, {688,45,-110}, {688,50,-110}, 
    {688,395,-35}, {688,398,-35}, {688,406,-35}, {688,416,-35}, 
    {688,421,-35}, {688,451,-35}, {688,457,-35}, {688,736,-85}, 
    {688,738,-85}, {688,760,-85}, {688,777,-85}, {688,1032,-85}, 
    {688,1113,-85}, {688,1175,-85}, {688,1404,-92}, {688,1411,-92}, 
    {688,1632,-111}, {688,1633,-111}, {688,1647,-111}, {688,1657,-71}, 
    {688,1664,-71}, {688,2128,-92}, {688,2134,-37}, {688,2135,-37}, 
    {688,2797,-111}, {688,2798,-111}, {688,2812,-111}, {688,2822,-111}, 
    {688,2829,-111}, {688,2902,-111}, {688,2910,-111}, {688,2998,-92}, 
    {688,3354,-92}, {688,3819,-92}, {688,3820,-92}, {688,3827,-92}, 
    {688,3834,-92}, {688,3842,-92}, {689,2,-110}, {689,7,-110}, 
    {689,18,-110}, {689,29,-110}, {689,35,-110}, {689,45,-110}, 
    {689,50,-110}, {689,395,-35}, {689,398,-35}, {689,406,-35}, 
    {689,416,-35}, {689,421,-35}, {689,451,-35}, {689,457,-35}, 
    {689,736,-85}, {689,738,-85}, {689,760,-85}, {689,777,-85}, 
    {689,1032,-85}, {689,1113,-85}, {689,1175,-85}, {689,1404,-92}, 
    {689,1411,-92}, {689,1632,-111}, {689,1633,-111}, {689,1647,-111}, 
    {689,1657,-71}, {689,1664,-71}, {689,2128,-92}, {689,2134,-37}, 
    {689,2135,-37}, {689,2797,-111}, {689,2798,-111}, {689,2812,-111}, 
    {689,2822,-111}, {689,2829,-111}, {689,2902,-111}, {689,2910,-111}, 
    {689,2998,-92}, {689,3354,-92}, {689,3819,-92}, {689,3820,-92}, 
    {689,3827,-92}, {689,3834,-92}, {689,3842,-92}, {694,2,-110}, 
    {694,7,-110}, {694,18,-110}, {694,29,-110}, {694,35,-110}, 
    {694,45,-110}, {694,50,-110}, {694,395,-35}, {694,398,-35}, 
    {694,406,-35}, {694,416,-35}, {694,421,-35}, {694,451,-35}, 
    {694,457,-35}, {694,736,-85}, {694,738,-85}, {694,760,-85}, 
    {694,777,-85}, {694,1032,-85}, {694,1113,-85}, {694,1175,-85}, 
    {694,1404,-92}, {694,1411,-92}, {694,1632,-111}, {694,1633,-111}, 
    {694,1647,-111}, {694,1657,-71}, {694,1664,-71}, {694,2128,-92}, 
    {694,2134,-37}, {694,2135,-37}, {694,2797,-111}, {694,2798,-111}, 
    {694,2812,-111}, {694,2822,-111}, {694,2829,-111}, {694,2902,-111}, 
    {694,2910,-111}, {694,2998,-92}, {694,3354,-92}, {694,3819,-92}, 
    {694,3820,-92}, {694,3827,-92}, {694,3834,-92}, {694,3842,-92}, 
    {736,3905,-25}, {738,3905,-25}, {760,3905,-25}, {777,3905,-25}, 
    {1032,3905,-25}, {1113,3905,-25}, {1175,3905,-25}, {1194,1194,-10}, 
    {1194,2998,-40}, {1194,3819,-20}, {1194,3820,-20}, {1194,3827,-20}, 
    {1194,3834,-20}, {1194,3842,-20}, {1194,3905,-15}, {1411,3142,-45}, 
    {1411,3146,-55}, {1483,3954,-15}, {1632,3905,-15}, {1633,3905,-15}, 
    {1647,3905,-15}, {1657,3905,-15}, {1664,3905,-15}, {1789,1411,-15}, 
    {1789,1610,-35}, {1789,2134,-25}, {1789,2797,-25}, {1789,2798,-25}, 
    {1789,2812,-25}, {1789,2822,-25}, {1789,2829,-25}, {1789,2902,-25}, 
    {1789,2910,-25}, {1789,2998,-15}, {1789,3142,50}, {1789,3146,55}, 
    {1888,2998,-15}, {1990,4034,-15}, {1990,4037,-15}, {1990,4053,-15}, 
    {2134,3905,-10}, {2135,3905,-10}, {2142,3905,-10}, {2198,3905,-10}, 
    {2209,3905,-10}, {2304,1632,-10}, {2304,1633,-10}, {2304,1647,-10}, 
    {2304,1657,-10}, {2304,1664,-10}, {2304,2797,-15}, {2304,2798,-15}, 
    {2304,2812,-15}, {2304,2822,-15}, {2304,2829,-15}, {2304,2902,-15}, 
    {2304,2910,-15}, {2304,4034,-15}, {2304,4037,-15}, {2304,4053,-15}, 
    {2650,3905,-40}, {2772,3905,-40}, {2797,3905,-10}, {2797,3954,-10}, 
    {2798,3905,-10}, {2798,3954,-10}, {2812,3905,-10}, {2812,3954,-10}, 
    {2822,3905,-10}, {2822,3954,-10}, {2829,3905,-10}, {2829,3954,-10}, 
    {2902,3905,-10}, {2902,3954,-10}, {2910,3905,-10}, {2910,3954,-10}, 
    {2998,3142,-55}, {2998,3146,-55}, {3138,2,-10}, {3138,7,-10}, 
    {3138,18,-10}, {3138,29,-10}, {3138,35,-10}, {3138,45,-10}, 
    {3138,50,-10}, {3143,2,-10}, {3143,7,-10}, {3143,18,-10}, 
    {3143,29,-10}, {3143,35,-10}, {3143,45,-10}, {3143,50,-10}, 
    {3143,3143,-63}, {3146,1483,-20}, {3146,3146,-63}, {3146,3151,-20}, 
    {3146,3282,-37}, {3146,3320,-37}, {3146,3501,-74}, {3146,3905,-20}, 
    {3151,1314,-18}, {3151,1334,-18}, {3151,1411,-92}, {3151,1632,-18}, 
    {3151,1633,-18}, {3151,1647,-18}, {3151,1657,-18}, {3151,1664,-18}, 
    {3151,1888,-10}, {3151,2128,-37}, {3151,2650,-15}, {3151,2772,-15}, 
    {3151,2797,-18}, {3151,2798,-18}, {3151,2812,-18}, {3151,2822,-18}, 
    {3151,2829,-18}, {3151,2902,-18}, {3151,2910,-18}, {3151,2924,-10}, 
    {3151,2998,-100}, {3151,3086,-18}, {3151,3905,-10}, {3501,2,-55}, 
    {3501,7,-55}, {3501,18,-55}, {3501,29,-55}, {3501,35,-55}, 
    {3501,45,-55}, {3501,50,-55}, {3501,572,-30}, {3501,660,-45}, 
    {3501,670,-30}, {3501,688,-55}, {3501,689,-55}, {3501,694,-55}, 
    {3905,736,-10}, {3905,738,-10}, {3905,760,-10}, {3905,777,-10}, 
    {3905,1032,-10}, {3905,1113,-10}, {3905,1175,-10}, {3905,1411,-55}, 
    {3905,1632,-10}, {3905,1633,-10}, {3905,1647,-10}, {3905,1657,-10}, 
    {3905,1664,-10}, {3905,2797,-10}, {3905,2798,-10}, {3905,2812,-10}, 
    {3905,2822,-10}, {3905,2829,-10}, {3905,2902,-10}, {3905,2910,-10}, 
    {3905,2998,-70}, {3954,1411,-55}, {3954,2797,-10}, {3954,2798,-10}, 
    {3954,2812,-10}, {3954,2822,-10}, {3954,2829,-10}, {3954,2902,-10}, 
    {3954,2910,-10}, {3954,2998,-70}, {4034,1411,-55}, {4034,1632,-10}, 
    {4034,1633,-10}, {4034,1647,-10}, {4034,1657,-10}, {4034,1664,-10}, 
    {4034,2797,-25}, {4034,2798,-25}, {4034,2812,-25}, {4034,2822,-25}, 
    {4034,2829,-25}, {4034,2902,-25}, {4034,2910,-25}, {4034,2998,-70}, 
    {4037,1411,-55}, {4037,1632,-10}, {4037,1633,-10}, {4037,1647,-10}, 
    {4037,1657,-10}, {4037,1664,-10}, {4037,2797,-25}, {4037,2798,-25}, 
    {4037,2812,-25}, {4037,2822,-25}, {4037,2829,-25}, {4037,2902,-25}, 
    {4037,2910,-25}, {4037,2998,-70}, {4053,1411,-55}, {4053,1632,-10}, 
    {4053,1633,-10}, {4053,1647,-10}, {4053,1657,-10}, {4053,1664,-10}, 
    {4053,2797,-25}, {4053,2798,-25}, {4053,2812,-25}, {4053,2822,-25}, 
    {4053,2829,-25}, {4053,2902,-25}, {4053,2910,-25}, {4053,2998,-70}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature times_bold_ligs[] = {
    {1789,2134,1809}, {1789,2418,1852}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair times_bolditalic_kerns[] = {
    {2,66,-65}, {2,72,-65}, {2,205,-60}, {2,395,-50}, {2,398,-50}, 
    {2,406,-50}, {2,416,-50}, {2,421,-50}, {2,451,-50}, {2,457,-50}, 
    {2,476,-55}, {2,572,-55}, {2,605,-50}, {2,606,-50}, {2,611,-50}, 
    {2,617,-50}, {2,626,-50}, {2,660,-95}, {2,670,-100}, {2,688,-70}, 
    {2,689,-70}, {2,694,-70}, {2,3146,-74}, {2,3819,-30}, {2,3820,-30}, 
    {2,3827,-30}, {2,3834,-30}, {2,3842,-30}, {2,3905,-74}, 
    {2,3954,-74}, {2,4034,-74}, {2,4037,-74}, {2,4053,-74}, {7,66,-65}, 
    {7,72,-65}, {7,205,-60}, {7,395,-50}, {7,398,-50}, {7,406,-50}, 
    {7,416,-50}, {7,421,-50}, {7,451,-50}, {7,457,-50}, {7,476,-55}, 
    {7,572,-55}, {7,605,-50}, {7,606,-50}, {7,611,-50}, {7,617,-50}, 
    {7,626,-50}, {7,660,-95}, {7,670,-100}, {7,688,-70}, {7,689,-70}, 
    {7,694,-70}, {7,3146,-74}, {7,3819,-30}, {7,3820,-30}, 
    {7,3827,-30}, {7,3834,-30}, {7,3842,-30}, {7,3905,-74}, 
    {7,3954,-74}, {7,4034,-74}, {7,4037,-74}, {7,4053,-74}, 
    {18,66,-65}, {18,72,-65}, {18,205,-60}, {18,395,-50}, {18,398,-50}, 
    {18,406,-50}, {18,416,-50}, {18,421,-50}, {18,451,-50}, 
    {18,457,-50}, {18,476,-55}, {18,572,-55}, {18,605,-50}, 
    {18,606,-50}, {18,611,-50}, {18,617,-50}, {18,626,-50}, 
    {18,660,-95}, {18,670,-100}, {18,688,-70}, {18,689,-70}, 
    {18,694,-70}, {18,3146,-74}, {18,3819,-30}, {18,3820,-30}, 
    {18,3827,-30}, {18,3834,-30}, {18,3842,-30}, {18,3905,-74}, 
    {18,3954,-74}, {18,4034,-74}, {18,4037,-74}, {18,4053,-74}, 
    {29,66,-65}, {29,72,-65}, {29,205,-60}, {29,395,-50}, {29,398,-50}, 
    {29,406,-50}, {29,416,-50}, {29,421,-50}, {29,451,-50}, 
    {29,457,-50}, {29,476,-55}, {29,572,-55}, {29,605,-50}, 
    {29,606,-50}, {29,611,-50}, {29,617,-50}, {29,626,-50}, 
    {29,660,-95}, {29,670,-100}, {29,688,-70}, {29,689,-70}, 
    {29,694,-70}, {29,3146,-74}, {29,3819,-30}, {29,3820,-30}, 
    {29,3827,-30}, {29,3834,-30}, {29,3842,-30}, {29,3905,-74}, 
    {29,3954,-74}, {29,4034,-74}, {29,4037,-74}, {29,4053,-74}, 
    {35,66,-65}, {35,72,-65}, {35,205,-60}, {35,395,-50}, {35,398,-50}, 
    {35,406,-50}, {35,416,-50}, {35,421,-50}, {35,451,-50}, 
    {35,457,-50}, {35,476,-55}, {35,572,-55}, {35,605,-50}, 
    {35,606,-50}, {35,611,-50}, {35,617,-50}, {35,626,-50}, 
    {35,660,-95}, {35,670,-100}, {35,688,-70}, {35,689,-70}, 
    {35,694,-70}, {35,3146,-74}, {35,3819,-30}, {35,3820,-30}, 
    {35,3827,-30}, {35,3834,-30}, {35,3842,-30}, {35,3905,-74}, 
    {35,3954,-74}, {35,4034,-74}, {35,4037,-74}, {35,4053,-74}, 
    {45,66,-65}, {45,72,-65}, {45,205,-60}, {45,395,-50}, {45,398,-50}, 
    {45,406,-50}, {45,416,-50}, {45,421,-50}, {45,451,-50}, 
    {45,457,-50}, {45,476,-55}, {45,572,-55}, {45,605,-50}, 
    {45,606,-50}, {45,611,-50}, {45,617,-50}, {45,626,-50}, 
    {45,660,-95}, {45,670,-100}, {45,688,-70}, {45,689,-70}, 
    {45,694,-70}, {45,3146,-74}, {45,3819,-30}, {45,3820,-30}, 
    {45,3827,-30}, {45,3834,-30}, {45,3842,-30}, {45,3905,-74}, 
    {45,3954,-74}, {45,4034,-74}, {45,4037,-74}, {45,4053,-74}, 
    {50,66,-65}, {50,72,-65}, {50,205,-60}, {50,395,-50}, {50,398,-50}, 
    {50,406,-50}, {50,416,-50}, {50,421,-50}, {50,451,-50}, 
    {50,457,-50}, {50,476,-55}, {50,572,-55}, {50,605,-50}, 
    {50,606,-50}, {50,611,-50}, {50,617,-50}, {50,626,-50}, 
    {50,660,-95}, {50,670,-100}, {50,688,-70}, {50,689,-70}, 
    {50,694,-70}, {50,3146,-74}, {50,3819,-30}, {50,3820,-30}, 
    {50,3827,-30}, {50,3834,-30}, {50,3842,-30}, {50,3905,-74}, 
    {50,3954,-74}, {50,4034,-74}, {50,4037,-74}, {50,4053,-74}, 
    {53,2,-25}, {53,7,-25}, {53,18,-25}, {53,29,-25}, {53,35,-25}, 
    {53,45,-25}, {53,50,-25}, {53,605,-10}, {53,606,-10}, {53,611,-10}, 
    {53,617,-10}, {53,626,-10}, {95,2,-25}, {95,7,-25}, {95,18,-25}, 
    {95,29,-25}, {95,35,-25}, {95,45,-25}, {95,50,-25}, {95,660,-50}, 
    {95,670,-40}, {95,688,-50}, {95,689,-50}, {95,694,-50}, 
    {194,2,-100}, {194,7,-100}, {194,18,-100}, {194,29,-100}, 
    {194,35,-100}, {194,45,-100}, {194,50,-100}, {194,736,-95}, 
    {194,738,-95}, {194,760,-95}, {194,777,-95}, {194,1032,-95}, 
    {194,1113,-95}, {194,1175,-95}, {194,1411,-129}, {194,1632,-100}, 
    {194,1633,-100}, {194,1647,-100}, {194,1657,-100}, {194,1664,-100}, 
    {194,2134,-40}, {194,2135,-40}, {194,2142,-40}, {194,2198,-40}, 
    {194,2209,-40}, {194,2797,-70}, {194,2798,-70}, {194,2812,-70}, 
    {194,2822,-70}, {194,2829,-70}, {194,2902,-70}, {194,2910,-70}, 
    {194,2998,-129}, {194,3151,-50}, {303,2,-25}, {303,7,-25}, 
    {303,18,-25}, {303,29,-25}, {303,35,-25}, {303,45,-25}, 
    {303,50,-25}, {303,736,-40}, {303,738,-40}, {303,760,-40}, 
    {303,777,-40}, {303,1032,-40}, {303,1113,-40}, {303,1175,-40}, 
    {303,1411,-10}, {303,1632,-40}, {303,1633,-40}, {303,1647,-40}, 
    {303,1657,-40}, {303,1664,-40}, {303,2797,-40}, {303,2798,-40}, 
    {303,2812,-40}, {303,2822,-40}, {303,2829,-40}, {303,2902,-40}, 
    {303,2910,-40}, {303,2998,-10}, {303,3819,-40}, {303,3820,-40}, 
    {303,3827,-40}, {303,3834,-40}, {303,3842,-40}, {311,395,-30}, 
    {311,398,-30}, {311,406,-30}, {311,416,-30}, {311,421,-30}, 
    {311,451,-30}, {311,457,-30}, {311,1632,-25}, {311,1633,-25}, 
    {311,1647,-25}, {311,1657,-25}, {311,1664,-25}, {311,2797,-25}, 
    {311,2798,-25}, {311,2812,-25}, {311,2822,-25}, {311,2829,-25}, 
    {311,2902,-25}, {311,2910,-25}, {311,3819,-20}, {311,3820,-20}, 
    {311,3827,-20}, {311,3834,-20}, {311,3842,-20}, {311,4034,-20}, 
    {311,4037,-20}, {311,4053,-20}, {339,572,-18}, {339,660,-37}, 
    {339,670,-37}, {339,688,-37}, {339,689,-37}, {339,694,-37}, 
    {339,3146,-55}, {339,4034,-37}, {339,4037,-37}, {339,4053,-37}, 
    {358,572,-18}, {358,660,-37}, {358,670,-37}, {358,688,-37}, 
    {358,689,-37}, {358,694,-37}, {358,3146,-55}, {358,4034,-37}, 
    {358,4037,-37}, {358,4053,-37}, {374,2,-30}, {374,7,-30}, 
    {374,18,-30}, {374,29,-30}, {374,35,-30}, {374,45,-30}, 
    {374,50,-30}, {392,2,-30}, {392,7,-30}, {392,18,-30}, {392,29,-30}, 
    {392,35,-30}, {392,45,-30}, {392,50,-30}, {395,2,-40}, {395,7,-40}, 
    {395,18,-40}, {395,29,-40}, {395,35,-40}, {395,45,-40}, 
    {395,50,-40}, {395,572,-40}, {395,660,-50}, {395,670,-50}, 
    {395,680,-40}, {395,688,-50}, {395,689,-50}, {395,694,-50}, 
    {398,2,-40}, {398,7,-40}, {398,18,-40}, {398,29,-40}, {398,35,-40}, 
    {398,45,-40}, {398,50,-40}, {398,572,-40}, {398,660,-50}, 
    {398,670,-50}, {398,680,-40}, {398,688,-50}, {398,689,-50}, 
    {398,694,-50}, {406,2,-40}, {406,7,-40}, {406,18,-40}, 
    {406,29,-40}, {406,35,-40}, {406,45,-40}, {406,50,-40}, 
    {406,572,-40}, {406,660,-50}, {406,670,-50}, {406,680,-40}, 
    {406,688,-50}, {406,689,-50}, {406,694,-50}, {416,2,-40}, 
    {416,7,-40}, {416,18,-40}, {416,29,-40}, {416,35,-40}, 
    {416,45,-40}, {416,50,-40}, {416,572,-40}, {416,660,-50}, 
    {416,670,-50}, {416,680,-40}, {416,688,-50}, {416,689,-50}, 
    {416,694,-50}, {421,2,-40}, {421,7,-40}, {421,18,-40}, 
    {421,29,-40}, {421,35,-40}, {421,45,-40}, {421,50,-40}, 
    {421,572,-40}, {421,660,-50}, {421,670,-50}, {421,680,-40}, 
    {421,688,-50}, {421,689,-50}, {421,694,-50}, {451,2,-40}, 
    {451,7,-40}, {451,18,-40}, {451,29,-40}, {451,35,-40}, 
    {451,45,-40}, {451,50,-40}, {451,572,-40}, {451,660,-50}, 
    {451,670,-50}, {451,680,-40}, {451,688,-50}, {451,689,-50}, 
    {451,694,-50}, {457,2,-40}, {457,7,-40}, {457,18,-40}, 
    {457,29,-40}, {457,35,-40}, {457,45,-40}, {457,50,-40}, 
    {457,572,-40}, {457,660,-50}, {457,670,-50}, {457,680,-40}, 
    {457,688,-50}, {457,689,-50}, {457,694,-50}, {461,2,-85}, 
    {461,7,-85}, {461,18,-85}, {461,29,-85}, {461,35,-85}, 
    {461,45,-85}, {461,50,-85}, {461,736,-40}, {461,738,-40}, 
    {461,760,-40}, {461,777,-40}, {461,1032,-40}, {461,1113,-40}, 
    {461,1175,-40}, {461,1411,-129}, {461,1632,-50}, {461,1633,-50}, 
    {461,1647,-50}, {461,1657,-50}, {461,1664,-50}, {461,2797,-55}, 
    {461,2798,-55}, {461,2812,-55}, {461,2822,-55}, {461,2829,-55}, 
    {461,2902,-55}, {461,2910,-55}, {461,2998,-129}, {476,605,-10}, 
    {476,606,-10}, {476,611,-10}, {476,617,-10}, {476,626,-10}, 
    {480,395,-40}, {480,398,-40}, {480,406,-40}, {480,416,-40}, 
    {480,421,-40}, {480,451,-40}, {480,457,-40}, {480,572,-30}, 
    {480,605,-40}, {480,606,-40}, {480,611,-40}, {480,617,-40}, 
    {480,626,-40}, {480,660,-18}, {480,670,-18}, {480,688,-18}, 
    {480,689,-18}, {480,694,-18}, {572,2,-55}, {572,7,-55}, 
    {572,18,-55}, {572,29,-55}, {572,35,-55}, {572,45,-55}, 
    {572,50,-55}, {572,395,-18}, {572,398,-18}, {572,406,-18}, 
    {572,416,-18}, {572,421,-18}, {572,451,-18}, {572,457,-18}, 
    {572,736,-92}, {572,738,-92}, {572,760,-92}, {572,777,-92}, 
    {572,1032,-92}, {572,1113,-92}, {572,1175,-92}, {572,1404,-74}, 
    {572,1411,-92}, {572,1632,-92}, {572,1633,-92}, {572,1647,-92}, 
    {572,1657,-52}, {572,1664,-52}, {572,2128,-92}, {572,2134,-37}, 
    {572,2135,-37}, {572,2797,-95}, {572,2798,-95}, {572,2812,-95}, 
    {572,2822,-95}, {572,2829,-95}, {572,2902,-95}, {572,2910,-95}, 
    {572,2998,-92}, {572,3151,-37}, {572,3354,-74}, {572,3819,-37}, 
    {572,3820,-37}, {572,3827,-37}, {572,3834,-37}, {572,3842,-37}, 
    {572,3954,-37}, {572,4034,-37}, {572,4037,-37}, {572,4053,-37}, 
    {605,2,-45}, {605,7,-45}, {605,18,-45}, {605,29,-45}, {605,35,-45}, 
    {605,45,-45}, {605,50,-45}, {606,2,-45}, {606,7,-45}, {606,18,-45}, 
    {606,29,-45}, {606,35,-45}, {606,45,-45}, {606,50,-45}, 
    {611,2,-45}, {611,7,-45}, {611,18,-45}, {611,29,-45}, {611,35,-45}, 
    {611,45,-45}, {611,50,-45}, {617,2,-45}, {617,7,-45}, {617,18,-45}, 
    {617,29,-45}, {617,35,-45}, {617,45,-45}, {617,50,-45}, 
    {626,2,-45}, {626,7,-45}, {626,18,-45}, {626,29,-45}, {626,35,-45}, 
    {626,45,-45}, {626,50,-45}, {660,2,-85}, {660,7,-85}, {660,18,-85}, 
    {660,29,-85}, {660,35,-85}, {660,45,-85}, {660,50,-85}, 
    {660,205,-10}, {660,395,-30}, {660,398,-30}, {660,406,-30}, 
    {660,416,-30}, {660,421,-30}, {660,451,-30}, {660,457,-30}, 
    {660,736,-111}, {660,738,-111}, {660,760,-111}, {660,777,-111}, 
    {660,1032,-111}, {660,1113,-111}, {660,1175,-111}, {660,1404,-74}, 
    {660,1411,-129}, {660,1632,-111}, {660,1633,-111}, {660,1647,-111}, 
    {660,1657,-71}, {660,1664,-71}, {660,2128,-70}, {660,2134,-55}, 
    {660,2135,-55}, {660,2797,-111}, {660,2798,-111}, {660,2812,-111}, 
    {660,2822,-111}, {660,2829,-111}, {660,2902,-111}, {660,2910,-111}, 
    {660,2998,-129}, {660,3354,-74}, {660,3819,-55}, {660,3820,-55}, 
    {660,3827,-55}, {660,3834,-55}, {660,3842,-55}, {670,2,-74}, 
    {670,7,-74}, {670,18,-74}, {670,29,-74}, {670,35,-74}, 
    {670,45,-74}, {670,50,-74}, {670,395,-15}, {670,398,-15}, 
    {670,406,-15}, {670,416,-15}, {670,421,-15}, {670,451,-15}, 
    {670,457,-15}, {670,736,-85}, {670,738,-85}, {670,760,-85}, 
    {670,777,-85}, {670,1032,-85}, {670,1113,-85}, {670,1175,-85}, 
    {670,1404,-55}, {670,1411,-74}, {670,1632,-90}, {670,1633,-90}, 
    {670,1647,-90}, {670,1657,-50}, {670,1664,-50}, {670,2128,-50}, 
    {670,2134,-37}, {670,2135,-37}, {670,2797,-80}, {670,2798,-80}, 
    {670,2812,-80}, {670,2822,-80}, {670,2829,-80}, {670,2902,-80}, 
    {670,2910,-80}, {670,2998,-74}, {670,3354,-55}, {670,3819,-55}, 
    {670,3820,-55}, {670,3827,-55}, {670,3834,-55}, {670,3842,-55}, 
    {670,4034,-55}, {670,4037,-55}, {670,4053,-55}, {688,2,-74}, 
    {688,7,-74}, {688,18,-74}, {688,29,-74}, {688,35,-74}, 
    {688,45,-74}, {688,50,-74}, {688,395,-25}, {688,398,-25}, 
    {688,406,-25}, {688,416,-25}, {688,421,-25}, {688,451,-25}, 
    {688,457,-25}, {688,736,-92}, {688,738,-92}, {688,760,-92}, 
    {688,777,-92}, {688,1032,-92}, {688,1113,-92}, {688,1175,-92}, 
    {688,1404,-92}, {688,1411,-92}, {688,1632,-111}, {688,1633,-111}, 
    {688,1647,-71}, {688,1657,-71}, {688,1664,-71}, {688,2128,-92}, 
    {688,2134,-55}, {688,2135,-55}, {688,2797,-111}, {688,2798,-111}, 
    {688,2812,-111}, {688,2822,-111}, {688,2829,-111}, {688,2902,-111}, 
    {688,2910,-111}, {688,2998,-74}, {688,3354,-92}, {688,3819,-92}, 
    {688,3820,-92}, {688,3827,-92}, {688,3834,-92}, {688,3842,-92}, 
    {689,2,-74}, {689,7,-74}, {689,18,-74}, {689,29,-74}, {689,35,-74}, 
    {689,45,-74}, {689,50,-74}, {689,395,-25}, {689,398,-25}, 
    {689,406,-25}, {689,416,-25}, {689,421,-25}, {689,451,-25}, 
    {689,457,-25}, {689,736,-92}, {689,738,-92}, {689,760,-92}, 
    {689,777,-92}, {689,1032,-92}, {689,1113,-92}, {689,1175,-92}, 
    {689,1404,-92}, {689,1411,-92}, {689,1632,-111}, {689,1633,-111}, 
    {689,1647,-71}, {689,1657,-71}, {689,1664,-71}, {689,2128,-92}, 
    {689,2134,-55}, {689,2135,-55}, {689,2797,-111}, {689,2798,-111}, 
    {689,2812,-111}, {689,2822,-111}, {689,2829,-111}, {689,2902,-111}, 
    {689,2910,-111}, {689,2998,-74}, {689,3354,-92}, {689,3819,-92}, 
    {689,3820,-92}, {689,3827,-92}, {689,3834,-92}, {689,3842,-92}, 
    {694,2,-74}, {694,7,-74}, {694,18,-74}, {694,29,-74}, {694,35,-74}, 
    {694,45,-74}, {694,50,-74}, {694,395,-25}, {694,398,-25}, 
    {694,406,-25}, {694,416,-25}, {694,421,-25}, {694,451,-25}, 
    {694,457,-25}, {694,736,-92}, {694,738,-92}, {694,760,-92}, 
    {694,777,-92}, {694,1032,-92}, {694,1113,-92}, {694,1175,-92}, 
    {694,1404,-92}, {694,1411,-92}, {694,1632,-111}, {694,1633,-111}, 
    {694,1647,-71}, {694,1657,-71}, {694,1664,-71}, {694,2128,-92}, 
    {694,2134,-55}, {694,2135,-55}, {694,2797,-111}, {694,2798,-111}, 
    {694,2812,-111}, {694,2822,-111}, {694,2829,-111}, {694,2902,-111}, 
    {694,2910,-111}, {694,2998,-74}, {694,3354,-92}, {694,3819,-92}, 
    {694,3820,-92}, {694,3827,-92}, {694,3834,-92}, {694,3842,-92}, 
    {1194,1194,-10}, {1194,2998,-40}, {1194,3819,-20}, {1194,3820,-20}, 
    {1194,3827,-20}, {1194,3834,-20}, {1194,3842,-20}, {1314,1990,-10}, 
    {1314,2304,-10}, {1334,1990,-10}, {1334,2304,-10}, {1411,3142,-95}, 
    {1411,3146,-95}, {1632,1194,-10}, {1633,1194,-10}, {1647,1194,-10}, 
    {1657,1194,-10}, {1664,1194,-10}, {1789,1411,-10}, {1789,1610,-30}, 
    {1789,1632,-10}, {1789,1633,-10}, {1789,1789,-18}, {1789,2797,-10}, 
    {1789,2798,-10}, {1789,2812,-10}, {1789,2829,-10}, {1789,2902,-10}, 
    {1789,2910,-10}, {1789,2998,-10}, {1789,3146,55}, {2304,1632,-30}, 
    {2304,1633,-30}, {2304,1647,-30}, {2304,1657,-30}, {2304,1664,-30}, 
    {2304,2797,-10}, {2304,2798,-10}, {2304,2812,-10}, {2304,2822,-10}, 
    {2304,2829,-10}, {2304,2902,-10}, {2304,2910,-10}, {2650,3905,-40}, 
    {2772,3905,-40}, {2797,3905,-15}, {2797,3954,-25}, {2797,4023,-10}, 
    {2797,4034,-10}, {2797,4037,-10}, {2797,4053,-10}, {2798,3905,-15}, 
    {2798,3954,-25}, {2798,4023,-10}, {2798,4034,-10}, {2798,4037,-10}, 
    {2798,4053,-10}, {2812,3905,-15}, {2812,3954,-25}, {2812,4023,-10}, 
    {2812,4034,-10}, {2812,4037,-10}, {2812,4053,-10}, {2822,3905,-15}, 
    {2822,3954,-25}, {2822,4023,-10}, {2822,4034,-10}, {2822,4037,-10}, 
    {2822,4053,-10}, {2829,3905,-15}, {2829,3954,-25}, {2829,4023,-10}, 
    {2829,4034,-10}, {2829,4037,-10}, {2829,4053,-10}, {2902,3905,-15}, 
    {2902,3954,-25}, {2902,4023,-10}, {2902,4034,-10}, {2902,4037,-10}, 
    {2902,4053,-10}, {2910,3905,-15}, {2910,3954,-25}, {2910,4023,-10}, 
    {2910,4034,-10}, {2910,4037,-10}, {2910,4053,-10}, {2998,3142,-95}, 
    {2998,3146,-95}, {3143,3143,-74}, {3146,1483,-15}, {3146,3146,-74}, 
    {3146,3151,-15}, {3146,3282,-74}, {3146,3320,-74}, {3146,3501,-74}, 
    {3146,3559,-37}, {3146,3905,-15}, {3151,1411,-65}, {3151,2998,-65}, 
    {3501,2,-37}, {3501,7,-37}, {3501,18,-37}, {3501,29,-37}, 
    {3501,35,-37}, {3501,45,-37}, {3501,50,-37}, {3501,660,-70}, 
    {3501,670,-70}, {3501,688,-70}, {3501,689,-70}, {3501,694,-70}, 
    {3905,1411,-37}, {3905,1632,-15}, {3905,1633,-15}, {3905,1647,-15}, 
    {3905,1657,-15}, {3905,1664,-15}, {3905,2797,-15}, {3905,2798,-15}, 
    {3905,2812,-15}, {3905,2822,-15}, {3905,2829,-15}, {3905,2902,-15}, 
    {3905,2910,-15}, {3905,2998,-37}, {3954,736,-10}, {3954,738,-10}, 
    {3954,760,-10}, {3954,777,-10}, {3954,1032,-10}, {3954,1113,-10}, 
    {3954,1175,-10}, {3954,1411,-37}, {3954,1632,-10}, {3954,1633,-10}, 
    {3954,1647,-10}, {3954,1657,-10}, {3954,1664,-10}, {3954,2797,-15}, 
    {3954,2798,-15}, {3954,2812,-15}, {3954,2822,-15}, {3954,2829,-15}, 
    {3954,2902,-15}, {3954,2910,-15}, {3954,2998,-37}, {4023,1632,-10}, 
    {4023,1633,-10}, {4023,1647,-10}, {4023,1657,-10}, {4023,1664,-10}, 
    {4034,1411,-37}, {4034,2998,-37}, {4037,1411,-37}, {4037,2998,-37}, 
    {4053,1411,-37}, {4053,2998,-37}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature times_bolditalic_ligs[] = {
    {1789,2134,1809}, {1789,2418,1852}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair helvetica_kerns[] = {
    {2,66,-30}, {2,72,-30}, {2,205,-30}, {2,395,-30}, {2,398,-30}, 
    {2,406,-30}, {2,416,-30}, {2,421,-30}, {2,451,-30}, {2,457,-30}, 
    {2,476,-30}, {2,572,-120}, {2,605,-50}, {2,606,-50}, {2,611,-50}, 
    {2,617,-50}, {2,626,-50}, {2,660,-70}, {2,670,-50}, {2,688,-100}, 
    {2,689,-100}, {2,694,-100}, {2,3819,-30}, {2,3820,-30}, 
    {2,3827,-30}, {2,3834,-30}, {2,3842,-30}, {2,3905,-40}, 
    {2,3954,-40}, {2,4034,-40}, {2,4037,-40}, {2,4053,-40}, {7,66,-30}, 
    {7,72,-30}, {7,205,-30}, {7,395,-30}, {7,398,-30}, {7,406,-30}, 
    {7,416,-30}, {7,421,-30}, {7,451,-30}, {7,457,-30}, {7,476,-30}, 
    {7,572,-120}, {7,605,-50}, {7,606,-50}, {7,611,-50}, {7,617,-50}, 
    {7,626,-50}, {7,660,-70}, {7,670,-50}, {7,688,-100}, {7,689,-100}, 
    {7,694,-100}, {7,3819,-30}, {7,3820,-30}, {7,3827,-30}, 
    {7,3834,-30}, {7,3842,-30}, {7,3905,-40}, {7,3954,-40}, 
    {7,4034,-40}, {7,4037,-40}, {7,4053,-40}, {18,66,-30}, {18,72,-30}, 
    {18,205,-30}, {18,395,-30}, {18,398,-30}, {18,406,-30}, 
    {18,416,-30}, {18,421,-30}, {18,451,-30}, {18,457,-30}, 
    {18,476,-30}, {18,572,-120}, {18,605,-50}, {18,606,-50}, 
    {18,611,-50}, {18,617,-50}, {18,626,-50}, {18,660,-70}, 
    {18,670,-50}, {18,688,-100}, {18,689,-100}, {18,694,-100}, 
    {18,3819,-30}, {18,3820,-30}, {18,3827,-30}, {18,3834,-30}, 
    {18,3842,-30}, {18,3905,-40}, {18,3954,-40}, {18,4034,-40}, 
    {18,4037,-40}, {18,4053,-40}, {29,66,-30}, {29,72,-30}, 
    {29,205,-30}, {29,395,-30}, {29,398,-30}, {29,406,-30}, 
    {29,416,-30}, {29,421,-30}, {29,451,-30}, {29,457,-30}, 
    {29,476,-30}, {29,572,-120}, {29,605,-50}, {29,606,-50}, 
    {29,611,-50}, {29,617,-50}, {29,626,-50}, {29,660,-70}, 
    {29,670,-50}, {29,688,-100}, {29,689,-100}, {29,694,-100}, 
    {29,3819,-30}, {29,3820,-30}, {29,3827,-30}, {29,3834,-30}, 
    {29,3842,-30}, {29,3905,-40}, {29,3954,-40}, {29,4034,-40}, 
    {29,4037,-40}, {29,4053,-40}, {35,66,-30}, {35,72,-30}, 
    {35,205,-30}, {35,395,-30}, {35,398,-30}, {35,406,-30}, 
    {35,416,-30}, {35,421,-30}, {35,451,-30}, {35,457,-30}, 
    {35,476,-30}, {35,572,-120}, {35,605,-50}, {35,606,-50}, 
    {35,611,-50}, {35,617,-50}, {35,626,-50}, {35,660,-70}, 
    {35,670,-50}, {35,688,-100}, {35,689,-100}, {35,694,-100}, 
    {35,3819,-30}, {35,3820,-30}, {35,3827,-30}, {35,3834,-30}, 
    {35,3842,-30}, {35,3905,-40}, {35,3954,-40}, {35,4034,-40}, 
    {35,4037,-40}, {35,4053,-40}, {45,66,-30}, {45,72,-30}, 
    {45,205,-30}, {45,395,-30}, {45,398,-30}, {45,406,-30}, 
    {45,416,-30}, {45,421,-30}, {45,451,-30}, {45,457,-30}, 
    {45,476,-30}, {45,572,-120}, {45,605,-50}, {45,606,-50}, 
    {45,611,-50}, {45,617,-50}, {45,626,-50}, {45,660,-70}, 
    {45,670,-50}, {45,688,-100}, {45,689,-100}, {45,694,-100}, 
    {45,3819,-30}, {45,3820,-30}, {45,3827,-30}, {45,3834,-30}, 
    {45,3842,-30}, {45,3905,-40}, {45,3954,-40}, {45,4034,-40}, 
    {45,4037,-40}, {45,4053,-40}, {50,66,-30}, {50,72,-30}, 
    {50,205,-30}, {50,395,-30}, {50,398,-30}, {50,406,-30}, 
    {50,416,-30}, {50,421,-30}, {50,451,-30}, {50,457,-30}, 
    {50,476,-30}, {50,572,-120}, {50,605,-50}, {50,606,-50}, 
    {50,611,-50}, {50,617,-50}, {50,626,-50}, {50,660,-70}, 
    {50,670,-50}, {50,688,-100}, {50,689,-100}, {50,694,-100}, 
    {50,3819,-30}, {50,3820,-30}, {50,3827,-30}, {50,3834,-30}, 
    {50,3842,-30}, {50,3905,-40}, {50,3954,-40}, {50,4034,-40}, 
    {50,4037,-40}, {50,4053,-40}, {53,605,-10}, {53,606,-10}, 
    {53,611,-10}, {53,617,-10}, {53,626,-10}, {53,1411,-20}, 
    {53,2998,-20}, {66,1411,-30}, {66,2998,-30}, {72,1411,-30}, 
    {72,2998,-30}, {95,2,-40}, {95,7,-40}, {95,18,-40}, {95,29,-40}, 
    {95,35,-40}, {95,45,-40}, {95,50,-40}, {95,660,-70}, {95,670,-40}, 
    {95,688,-90}, {95,689,-90}, {95,694,-90}, {95,1411,-70}, 
    {95,2998,-70}, {194,2,-80}, {194,7,-80}, {194,18,-80}, 
    {194,29,-80}, {194,35,-80}, {194,45,-80}, {194,50,-80}, 
    {194,736,-50}, {194,738,-50}, {194,760,-50}, {194,777,-50}, 
    {194,1032,-50}, {194,1113,-50}, {194,1175,-50}, {194,1411,-150}, 
    {194,1632,-30}, {194,1633,-30}, {194,1647,-30}, {194,1657,-30}, 
    {194,1664,-30}, {194,2797,-30}, {194,2798,-30}, {194,2812,-30}, 
    {194,2822,-30}, {194,2829,-30}, {194,2902,-30}, {194,2910,-30}, 
    {194,2998,-150}, {194,3151,-45}, {303,2,-20}, {303,7,-20}, 
    {303,18,-20}, {303,29,-20}, {303,35,-20}, {303,45,-20}, 
    {303,50,-20}, {303,736,-20}, {303,738,-20}, {303,760,-20}, 
    {303,777,-20}, {303,1032,-20}, {303,1113,-20}, {303,1175,-20}, 
    {303,1411,-30}, {303,2998,-30}, {303,3819,-20}, {303,3820,-20}, 
    {303,3827,-20}, {303,3834,-20}, {303,3842,-20}, {311,395,-50}, 
    {311,398,-50}, {311,406,-50}, {311,416,-50}, {311,421,-50}, 
    {311,451,-50}, {311,457,-50}, {311,1632,-40}, {311,1633,-40}, 
    {311,1647,-40}, {311,1657,-40}, {311,1664,-40}, {311,2797,-40}, 
    {311,2798,-40}, {311,2812,-40}, {311,2822,-40}, {311,2829,-40}, 
    {311,2902,-40}, {311,2910,-40}, {311,3819,-30}, {311,3820,-30}, 
    {311,3827,-30}, {311,3834,-30}, {311,3842,-30}, {311,4034,-50}, 
    {311,4037,-50}, {311,4053,-50}, {339,572,-110}, {339,660,-110}, 
    {339,670,-70}, {339,688,-140}, {339,689,-140}, {339,694,-140}, 
    {339,3142,-140}, {339,3146,-160}, {339,4034,-30}, {339,4037,-30}, 
    {339,4053,-30}, {358,572,-110}, {358,660,-110}, {358,670,-70}, 
    {358,688,-140}, {358,689,-140}, {358,694,-140}, {358,3142,-140}, 
    {358,3146,-160}, {358,4034,-30}, {358,4037,-30}, {358,4053,-30}, 
    {395,2,-20}, {395,7,-20}, {395,18,-20}, {395,29,-20}, {395,35,-20}, 
    {395,45,-20}, {395,50,-20}, {395,572,-40}, {395,660,-50}, 
    {395,670,-30}, {395,680,-60}, {395,688,-70}, {395,689,-70}, 
    {395,694,-70}, {395,1411,-40}, {395,2998,-40}, {398,2,-20}, 
    {398,7,-20}, {398,18,-20}, {398,29,-20}, {398,35,-20}, 
    {398,45,-20}, {398,50,-20}, {398,572,-40}, {398,660,-50}, 
    {398,670,-30}, {398,680,-60}, {398,688,-70}, {398,689,-70}, 
    {398,694,-70}, {398,1411,-40}, {398,2998,-40}, {406,2,-20}, 
    {406,7,-20}, {406,18,-20}, {406,29,-20}, {406,35,-20}, 
    {406,45,-20}, {406,50,-20}, {406,572,-40}, {406,660,-50}, 
    {406,670,-30}, {406,680,-60}, {406,688,-70}, {406,689,-70}, 
    {406,694,-70}, {406,1411,-40}, {406,2998,-40}, {416,2,-20}, 
    {416,7,-20}, {416,18,-20}, {416,29,-20}, {416,35,-20}, 
    {416,45,-20}, {416,50,-20}, {416,572,-40}, {416,660,-50}, 
    {416,670,-30}, {416,680,-60}, {416,688,-70}, {416,689,-70}, 
    {416,694,-70}, {416,1411,-40}, {416,2998,-40}, {421,2,-20}, 
    {421,7,-20}, {421,18,-20}, {421,29,-20}, {421,35,-20}, 
    {421,45,-20}, {421,50,-20}, {421,572,-40}, {421,660,-50}, 
    {421,670,-30}, {421,680,-60}, {421,688,-70}, {421,689,-70}, 
    {421,694,-70}, {421,1411,-40}, {421,2998,-40}, {451,2,-20}, 
    {451,7,-20}, {451,18,-20}, {451,29,-20}, {451,35,-20}, 
    {451,45,-20}, {451,50,-20}, {451,572,-40}, {451,660,-50}, 
    {451,670,-30}, {451,680,-60}, {451,688,-70}, {451,689,-70}, 
    {451,694,-70}, {451,1411,-40}, {451,2998,-40}, {457,2,-20}, 
    {457,7,-20}, {457,18,-20}, {457,29,-20}, {457,35,-20}, 
    {457,45,-20}, {457,50,-20}, {457,572,-40}, {457,660,-50}, 
    {457,670,-30}, {457,680,-60}, {457,688,-70}, {457,689,-70}, 
    {457,694,-70}, {457,1411,-40}, {457,2998,-40}, {461,2,-120}, 
    {461,7,-120}, {461,18,-120}, {461,29,-120}, {461,35,-120}, 
    {461,45,-120}, {461,50,-120}, {461,736,-40}, {461,738,-40}, 
    {461,760,-40}, {461,777,-40}, {461,1032,-40}, {461,1113,-40}, 
    {461,1175,-40}, {461,1411,-180}, {461,1632,-50}, {461,1633,-50}, 
    {461,1647,-50}, {461,1657,-50}, {461,1664,-50}, {461,2797,-50}, 
    {461,2798,-50}, {461,2812,-50}, {461,2822,-50}, {461,2829,-50}, 
    {461,2902,-50}, {461,2910,-50}, {461,2998,-180}, {476,605,-10}, 
    {476,606,-10}, {476,611,-10}, {476,617,-10}, {476,626,-10}, 
    {480,395,-20}, {480,398,-20}, {480,406,-20}, {480,416,-20}, 
    {480,421,-20}, {480,451,-20}, {480,457,-20}, {480,572,-30}, 
    {480,605,-40}, {480,606,-40}, {480,611,-40}, {480,617,-40}, 
    {480,626,-40}, {480,660,-50}, {480,670,-30}, {480,688,-50}, 
    {480,689,-50}, {480,694,-50}, {501,1411,-20}, {501,2998,-20}, 
    {545,1411,-20}, {545,2998,-20}, {572,2,-120}, {572,7,-120}, 
    {572,18,-120}, {572,29,-120}, {572,35,-120}, {572,45,-120}, 
    {572,50,-120}, {572,395,-40}, {572,398,-40}, {572,406,-40}, 
    {572,416,-40}, {572,421,-40}, {572,451,-40}, {572,457,-40}, 
    {572,736,-120}, {572,738,-120}, {572,760,-120}, {572,777,-120}, 
    {572,1032,-120}, {572,1113,-120}, {572,1175,-60}, {572,1404,-20}, 
    {572,1411,-120}, {572,1632,-120}, {572,1633,-120}, {572,1647,-120}, 
    {572,1657,-120}, {572,1664,-60}, {572,2128,-140}, {572,2797,-120}, 
    {572,2798,-120}, {572,2812,-120}, {572,2822,-120}, {572,2829,-120}, 
    {572,2902,-120}, {572,2910,-60}, {572,2998,-120}, {572,3151,-120}, 
    {572,3354,-20}, {572,3819,-120}, {572,3820,-120}, {572,3827,-120}, 
    {572,3834,-120}, {572,3842,-120}, {572,3954,-120}, {572,4034,-120}, 
    {572,4037,-120}, {572,4053,-60}, {605,2,-40}, {605,7,-40}, 
    {605,18,-40}, {605,29,-40}, {605,35,-40}, {605,45,-40}, 
    {605,50,-40}, {605,1411,-40}, {605,2998,-40}, {606,2,-40}, 
    {606,7,-40}, {606,18,-40}, {606,29,-40}, {606,35,-40}, 
    {606,45,-40}, {606,50,-40}, {606,1411,-40}, {606,2998,-40}, 
    {611,2,-40}, {611,7,-40}, {611,18,-40}, {611,29,-40}, {611,35,-40}, 
    {611,45,-40}, {611,50,-40}, {611,1411,-40}, {611,2998,-40}, 
    {617,2,-40}, {617,7,-40}, {617,18,-40}, {617,29,-40}, {617,35,-40}, 
    {617,45,-40}, {617,50,-40}, {617,1411,-40}, {617,2998,-40}, 
    {626,2,-40}, {626,7,-40}, {626,18,-40}, {626,29,-40}, {626,35,-40}, 
    {626,45,-40}, {626,50,-40}, {626,1411,-40}, {626,2998,-40}, 
    {660,2,-80}, {660,7,-80}, {660,18,-80}, {660,29,-80}, {660,35,-80}, 
    {660,45,-80}, {660,50,-80}, {660,205,-40}, {660,395,-40}, 
    {660,398,-40}, {660,406,-40}, {660,416,-40}, {660,421,-40}, 
    {660,451,-40}, {660,457,-40}, {660,736,-70}, {660,738,-70}, 
    {660,760,-70}, {660,777,-70}, {660,1032,-70}, {660,1113,-70}, 
    {660,1175,-70}, {660,1404,-40}, {660,1411,-125}, {660,1632,-80}, 
    {660,1633,-80}, {660,1647,-80}, {660,1657,-80}, {660,1664,-80}, 
    {660,2128,-80}, {660,2797,-80}, {660,2798,-80}, {660,2812,-80}, 
    {660,2822,-80}, {660,2829,-80}, {660,2902,-80}, {660,2910,-80}, 
    {660,2998,-125}, {660,3354,-40}, {660,3819,-70}, {660,3820,-70}, 
    {660,3827,-70}, {660,3834,-70}, {660,3842,-70}, {670,2,-50}, 
    {670,7,-50}, {670,18,-50}, {670,29,-50}, {670,35,-50}, 
    {670,45,-50}, {670,50,-50}, {670,395,-20}, {670,398,-20}, 
    {670,406,-20}, {670,416,-20}, {670,421,-20}, {670,451,-20}, 
    {670,457,-20}, {670,736,-40}, {670,738,-40}, {670,760,-40}, 
    {670,777,-40}, {670,1032,-40}, {670,1113,-40}, {670,1175,-40}, 
    {670,1411,-80}, {670,1632,-30}, {670,1633,-30}, {670,1647,-30}, 
    {670,1657,-30}, {670,1664,-30}, {670,2128,-40}, {670,2797,-30}, 
    {670,2798,-30}, {670,2812,-30}, {670,2822,-30}, {670,2829,-30}, 
    {670,2902,-30}, {670,2910,-30}, {670,2998,-80}, {670,3819,-30}, 
    {670,3820,-30}, {670,3827,-30}, {670,3834,-30}, {670,3842,-30}, 
    {670,4034,-20}, {670,4037,-20}, {670,4053,-20}, {688,2,-110}, 
    {688,7,-110}, {688,18,-110}, {688,29,-110}, {688,35,-110}, 
    {688,45,-110}, {688,50,-110}, {688,395,-85}, {688,398,-85}, 
    {688,406,-85}, {688,416,-85}, {688,421,-85}, {688,451,-85}, 
    {688,457,-85}, {688,736,-140}, {688,738,-140}, {688,760,-140}, 
    {688,777,-140}, {688,1032,-140}, {688,1113,-140}, {688,1175,-140}, 
    {688,1404,-60}, {688,1411,-140}, {688,1632,-140}, {688,1633,-140}, 
    {688,1647,-140}, {688,1657,-140}, {688,1664,-140}, {688,2128,-140}, 
    {688,2134,-20}, {688,2135,-20}, {688,2797,-140}, {688,2798,-140}, 
    {688,2812,-140}, {688,2822,-140}, {688,2829,-140}, {688,2902,-140}, 
    {688,2910,-140}, {688,2998,-140}, {688,3354,-60}, {688,3819,-110}, 
    {688,3820,-110}, {688,3827,-110}, {688,3834,-110}, {688,3842,-110}, 
    {689,2,-110}, {689,7,-110}, {689,18,-110}, {689,29,-110}, 
    {689,35,-110}, {689,45,-110}, {689,50,-110}, {689,395,-85}, 
    {689,398,-85}, {689,406,-85}, {689,416,-85}, {689,421,-85}, 
    {689,451,-85}, {689,457,-85}, {689,736,-140}, {689,738,-140}, 
    {689,760,-140}, {689,777,-140}, {689,1032,-140}, {689,1113,-140}, 
    {689,1175,-70}, {689,1404,-60}, {689,1411,-140}, {689,1632,-140}, 
    {689,1633,-140}, {689,1647,-140}, {689,1657,-140}, {689,1664,-140}, 
    {689,2128,-140}, {689,2134,-20}, {689,2135,-20}, {689,2797,-140}, 
    {689,2798,-140}, {689,2812,-140}, {689,2822,-140}, {689,2829,-140}, 
    {689,2902,-140}, {689,2910,-140}, {689,2998,-140}, {689,3354,-60}, 
    {689,3819,-110}, {689,3820,-110}, {689,3827,-110}, {689,3834,-110}, 
    {689,3842,-110}, {694,2,-110}, {694,7,-110}, {694,18,-110}, 
    {694,29,-110}, {694,35,-110}, {694,45,-110}, {694,50,-110}, 
    {694,395,-85}, {694,398,-85}, {694,406,-85}, {694,416,-85}, 
    {694,421,-85}, {694,451,-85}, {694,457,-85}, {694,736,-140}, 
    {694,738,-140}, {694,760,-140}, {694,777,-140}, {694,1032,-140}, 
    {694,1113,-140}, {694,1175,-70}, {694,1404,-60}, {694,1411,-140}, 
    {694,1632,-140}, {694,1633,-140}, {694,1647,-140}, {694,1657,-140}, 
    {694,1664,-140}, {694,2128,-140}, {694,2134,-20}, {694,2135,-20}, 
    {694,2797,-140}, {694,2798,-140}, {694,2812,-140}, {694,2822,-140}, 
    {694,2829,-140}, {694,2902,-140}, {694,2910,-140}, {694,2998,-140}, 
    {694,3354,-60}, {694,3819,-110}, {694,3820,-110}, {694,3827,-110}, 
    {694,3834,-110}, {694,3842,-110}, {736,3905,-20}, {736,3954,-20}, 
    {736,4034,-30}, {736,4037,-30}, {736,4053,-30}, {738,3905,-20}, 
    {738,3954,-20}, {738,4034,-30}, {738,4037,-30}, {738,4053,-30}, 
    {760,3905,-20}, {760,3954,-20}, {760,4034,-30}, {760,4037,-30}, 
    {760,4053,-30}, {777,3905,-20}, {777,3954,-20}, {777,4034,-30}, 
    {777,4037,-30}, {777,4053,-30}, {1032,3905,-20}, {1032,3954,-20}, 
    {1032,4034,-30}, {1032,4037,-30}, {1032,4053,-30}, {1113,3905,-20}, 
    {1113,3954,-20}, {1113,4034,-30}, {1113,4037,-30}, {1113,4053,-30}, 
    {1175,3905,-20}, {1175,3954,-20}, {1175,4034,-30}, {1175,4037,-30}, 
    {1175,4053,-30}, {1194,1194,-10}, {1194,1411,-40}, {1194,2418,-20}, 
    {1194,2503,-20}, {1194,2998,-40}, {1194,3819,-20}, {1194,3820,-20}, 
    {1194,3827,-20}, {1194,3834,-20}, {1194,3842,-20}, {1194,3905,-20}, 
    {1194,4034,-20}, {1194,4037,-20}, {1194,4053,-20}, {1314,1411,-15}, 
    {1314,2304,-20}, {1334,1411,-15}, {1334,2304,-20}, {1404,3501,-50}, 
    {1411,3142,-100}, {1411,3146,-100}, {1632,1411,-15}, 
    {1632,2998,-15}, {1632,3905,-30}, {1632,3954,-20}, {1632,4023,-30}, 
    {1632,4034,-20}, {1632,4037,-20}, {1632,4053,-20}, {1633,1411,-15}, 
    {1633,2998,-15}, {1633,3905,-30}, {1633,3954,-20}, {1633,4023,-30}, 
    {1633,4034,-20}, {1633,4037,-20}, {1633,4053,-20}, {1647,1411,-15}, 
    {1647,2998,-15}, {1647,3905,-30}, {1647,3954,-20}, {1647,4023,-30}, 
    {1647,4034,-20}, {1647,4037,-20}, {1647,4053,-20}, {1657,1411,-15}, 
    {1657,2998,-15}, {1657,3905,-30}, {1657,3954,-20}, {1657,4023,-30}, 
    {1657,4034,-20}, {1657,4037,-20}, {1657,4053,-20}, {1664,1411,-15}, 
    {1664,2998,-15}, {1664,3905,-30}, {1664,3954,-20}, {1664,4023,-30}, 
    {1664,4034,-20}, {1664,4037,-20}, {1664,4053,-20}, {1789,736,-30}, 
    {1789,738,-30}, {1789,760,-30}, {1789,777,-30}, {1789,1032,-30}, 
    {1789,1113,-30}, {1789,1175,-30}, {1789,1411,-30}, {1789,1610,-28}, 
    {1789,1632,-30}, {1789,1633,-30}, {1789,1647,-30}, {1789,1657,-30}, 
    {1789,1664,-30}, {1789,2797,-30}, {1789,2798,-30}, {1789,2812,-30}, 
    {1789,2822,-30}, {1789,2829,-30}, {1789,2902,-30}, {1789,2910,-30}, 
    {1789,2998,-30}, {1789,3142,60}, {1789,3146,50}, {1888,3151,-10}, 
    {1990,4034,-30}, {1990,4037,-30}, {1990,4053,-30}, {2304,1632,-20}, 
    {2304,1633,-20}, {2304,1647,-20}, {2304,1657,-20}, {2304,1664,-20}, 
    {2304,2797,-20}, {2304,2798,-20}, {2304,2812,-20}, {2304,2822,-20}, 
    {2304,2829,-20}, {2304,2902,-20}, {2304,2910,-20}, {2513,3819,-10}, 
    {2513,3820,-10}, {2513,3827,-10}, {2513,3834,-10}, {2513,3842,-10}, 
    {2513,4034,-15}, {2513,4037,-15}, {2513,4053,-15}, {2650,3819,-10}, 
    {2650,3820,-10}, {2650,3827,-10}, {2650,3834,-10}, {2650,3842,-10}, 
    {2650,3905,-20}, {2650,4034,-15}, {2650,4037,-15}, {2650,4053,-15}, 
    {2772,3819,-10}, {2772,3820,-10}, {2772,3827,-10}, {2772,3834,-10}, 
    {2772,3842,-10}, {2772,3905,-20}, {2772,4034,-15}, {2772,4037,-15}, 
    {2772,4053,-15}, {2797,1411,-40}, {2797,2998,-40}, {2797,3905,-15}, 
    {2797,3954,-15}, {2797,4023,-30}, {2797,4034,-30}, {2797,4037,-30}, 
    {2797,4053,-30}, {2798,1411,-40}, {2798,2998,-40}, {2798,3905,-15}, 
    {2798,3954,-15}, {2798,4023,-30}, {2798,4034,-30}, {2798,4037,-30}, 
    {2798,4053,-30}, {2812,1411,-40}, {2812,2998,-40}, {2812,3905,-15}, 
    {2812,3954,-15}, {2812,4023,-30}, {2812,4034,-30}, {2812,4037,-30}, 
    {2812,4053,-30}, {2822,1411,-40}, {2822,2998,-40}, {2822,3905,-15}, 
    {2822,3954,-15}, {2822,4023,-30}, {2822,4034,-30}, {2822,4037,-30}, 
    {2822,4053,-30}, {2829,1411,-40}, {2829,2998,-40}, {2829,3905,-15}, 
    {2829,3954,-15}, {2829,4023,-30}, {2829,4034,-30}, {2829,4037,-30}, 
    {2829,4053,-30}, {2902,736,-55}, {2902,738,-55}, {2902,760,-55}, 
    {2902,777,-55}, {2902,1032,-55}, {2902,1113,-55}, {2902,1175,-55}, 
    {2902,1194,-55}, {2902,1314,-55}, {2902,1334,-55}, {2902,1411,-95}, 
    {2902,1483,-55}, {2902,1632,-55}, {2902,1633,-55}, {2902,1647,-55}, 
    {2902,1657,-55}, {2902,1664,-55}, {2902,1789,-55}, {2902,1888,-55}, 
    {2902,1990,-55}, {2902,2134,-55}, {2902,2135,-55}, {2902,2142,-55}, 
    {2902,2198,-55}, {2902,2209,-55}, {2902,2276,-55}, {2902,2304,-55}, 
    {2902,2418,-55}, {2902,2503,-55}, {2902,2513,-55}, {2902,2650,-55}, 
    {2902,2772,-55}, {2902,2797,-55}, {2902,2798,-55}, {2902,2812,-55}, 
    {2902,2822,-55}, {2902,2829,-55}, {2902,2902,-55}, {2902,2910,-55}, 
    {2902,2924,-55}, {2902,2998,-95}, {2902,3086,-55}, {2902,3151,-55}, 
    {2902,3282,-55}, {2902,3320,-55}, {2902,3559,-55}, {2902,3819,-55}, 
    {2902,3820,-55}, {2902,3827,-55}, {2902,3834,-55}, {2902,3842,-55}, 
    {2902,3905,-70}, {2902,3954,-70}, {2902,4023,-85}, {2902,4034,-70}, 
    {2902,4037,-70}, {2902,4053,-70}, {2902,4135,-55}, {2902,4156,-55}, 
    {2910,1411,-40}, {2910,2998,-40}, {2910,3905,-15}, {2910,3954,-15}, 
    {2910,4023,-30}, {2910,4034,-30}, {2910,4037,-30}, {2910,4053,-30}, 
    {2924,1411,-35}, {2924,2998,-35}, {2924,4034,-30}, {2924,4037,-30}, 
    {2924,4053,-30}, {2998,3142,-100}, {2998,3146,-100}, 
    {2998,3501,-60}, {3142,3501,-40}, {3143,3143,-57}, {3146,1483,-50}, 
    {3146,3146,-57}, {3146,3151,-50}, {3146,3282,-50}, {3146,3320,-50}, 
    {3146,3501,-70}, {3151,736,-10}, {3151,738,-10}, {3151,760,-10}, 
    {3151,777,-10}, {3151,1032,-10}, {3151,1113,-10}, {3151,1175,-10}, 
    {3151,1404,30}, {3151,1411,-50}, {3151,2134,15}, {3151,2135,15}, 
    {3151,2142,15}, {3151,2198,15}, {3151,2209,15}, {3151,2304,15}, 
    {3151,2418,15}, {3151,2503,15}, {3151,2513,25}, {3151,2650,25}, 
    {3151,2772,25}, {3151,2924,30}, {3151,2998,-50}, {3151,3354,30}, 
    {3151,3559,40}, {3151,3819,15}, {3151,3820,15}, {3151,3827,15}, 
    {3151,3834,15}, {3151,3842,15}, {3151,3905,30}, {3151,4034,30}, 
    {3151,4037,30}, {3151,4053,30}, {3282,1411,-15}, {3282,2998,-15}, 
    {3282,3954,-30}, {3320,1411,-15}, {3320,2998,-15}, {3320,3954,-30}, 
    {3354,3501,-50}, {3501,572,-50}, {3501,660,-50}, {3501,670,-40}, 
    {3501,688,-90}, {3501,689,-90}, {3501,694,-90}, {3501,3138,-30}, 
    {3501,3143,-60}, {3905,736,-25}, {3905,738,-25}, {3905,760,-25}, 
    {3905,777,-25}, {3905,1032,-25}, {3905,1113,-25}, {3905,1175,-25}, 
    {3905,1411,-80}, {3905,1632,-25}, {3905,1633,-25}, {3905,1647,-25}, 
    {3905,1657,-25}, {3905,1664,-25}, {3905,2797,-25}, {3905,2798,-25}, 
    {3905,2812,-25}, {3905,2822,-25}, {3905,2829,-25}, {3905,2902,-25}, 
    {3905,2910,-25}, {3905,2998,-80}, {3954,736,-15}, {3954,738,-15}, 
    {3954,760,-15}, {3954,777,-15}, {3954,1032,-15}, {3954,1113,-15}, 
    {3954,1175,-15}, {3954,1411,-60}, {3954,1632,-10}, {3954,1633,-10}, 
    {3954,1647,-10}, {3954,1657,-10}, {3954,1664,-10}, {3954,2797,-10}, 
    {3954,2798,-10}, {3954,2812,-10}, {3954,2822,-10}, {3954,2829,-10}, 
    {3954,2902,-10}, {3954,2910,-10}, {3954,2998,-60}, {4023,1632,-30}, 
    {4023,1633,-30}, {4023,1647,-30}, {4023,1657,-30}, {4023,1664,-30}, 
    {4034,736,-20}, {4034,738,-20}, {4034,760,-20}, {4034,777,-20}, 
    {4034,1032,-20}, {4034,1113,-20}, {4034,1175,-20}, 
    {4034,1411,-100}, {4034,1632,-20}, {4034,1633,-20}, 
    {4034,1647,-20}, {4034,1657,-20}, {4034,1664,-20}, {4034,2797,-20}, 
    {4034,2798,-20}, {4034,2812,-20}, {4034,2822,-20}, {4034,2829,-20}, 
    {4034,2902,-20}, {4034,2910,-20}, {4034,2998,-100}, {4037,736,-20}, 
    {4037,738,-20}, {4037,760,-20}, {4037,777,-20}, {4037,1032,-20}, 
    {4037,1113,-20}, {4037,1175,-20}, {4037,1411,-100}, 
    {4037,1632,-20}, {4037,1633,-20}, {4037,1647,-20}, {4037,1657,-20}, 
    {4037,1664,-20}, {4037,2797,-20}, {4037,2798,-20}, {4037,2812,-20}, 
    {4037,2822,-20}, {4037,2829,-20}, {4037,2902,-20}, {4037,2910,-20}, 
    {4037,2998,-100}, {4053,736,-20}, {4053,738,-20}, {4053,760,-20}, 
    {4053,777,-20}, {4053,1032,-20}, {4053,1113,-20}, {4053,1175,-20}, 
    {4053,1411,-100}, {4053,1632,-20}, {4053,1633,-20}, 
    {4053,1647,-20}, {4053,1657,-20}, {4053,1664,-20}, {4053,2797,-20}, 
    {4053,2798,-20}, {4053,2812,-20}, {4053,2822,-20}, {4053,2829,-20}, 
    {4053,2902,-20}, {4053,2910,-20}, {4053,2998,-100}, 
    {4135,1632,-15}, {4135,1633,-15}, {4135,1647,-15}, {4135,1657,-15}, 
    {4135,1664,-15}, {4135,2797,-15}, {4135,2798,-15}, {4135,2812,-15}, 
    {4135,2822,-15}, {4135,2829,-15}, {4135,2902,-15}, {4135,2910,-15}, 
    {4156,1632,-15}, {4156,1633,-15}, {4156,1647,-15}, {4156,1657,-15}, 
    {4156,1664,-15}, {4156,2797,-15}, {4156,2798,-15}, {4156,2812,-15}, 
    {4156,2822,-15}, {4156,2829,-15}, {4156,2902,-15}, {4156,2910,-15}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature helvetica_ligs[] = {
    {1789,2134,1809}, {1789,2418,1852}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair helvetica_oblique_kerns[] = {
    {2,66,-30}, {2,72,-30}, {2,205,-30}, {2,395,-30}, {2,398,-30}, 
    {2,406,-30}, {2,416,-30}, {2,421,-30}, {2,451,-30}, {2,457,-30}, 
    {2,476,-30}, {2,572,-120}, {2,605,-50}, {2,606,-50}, {2,611,-50}, 
    {2,617,-50}, {2,626,-50}, {2,660,-70}, {2,670,-50}, {2,688,-100}, 
    {2,689,-100}, {2,694,-100}, {2,3819,-30}, {2,3820,-30}, 
    {2,3827,-30}, {2,3834,-30}, {2,3842,-30}, {2,3905,-40}, 
    {2,3954,-40}, {2,4034,-40}, {2,4037,-40}, {2,4053,-40}, {7,66,-30}, 
    {7,72,-30}, {7,205,-30}, {7,395,-30}, {7,398,-30}, {7,406,-30}, 
    {7,416,-30}, {7,421,-30}, {7,451,-30}, {7,457,-30}, {7,476,-30}, 
    {7,572,-120}, {7,605,-50}, {7,606,-50}, {7,611,-50}, {7,617,-50}, 
    {7,626,-50}, {7,660,-70}, {7,670,-50}, {7,688,-100}, {7,689,-100}, 
    {7,694,-100}, {7,3819,-30}, {7,3820,-30}, {7,3827,-30}, 
    {7,3834,-30}, {7,3842,-30}, {7,3905,-40}, {7,3954,-40}, 
    {7,4034,-40}, {7,4037,-40}, {7,4053,-40}, {18,66,-30}, {18,72,-30}, 
    {18,205,-30}, {18,395,-30}, {18,398,-30}, {18,406,-30}, 
    {18,416,-30}, {18,421,-30}, {18,451,-30}, {18,457,-30}, 
    {18,476,-30}, {18,572,-120}, {18,605,-50}, {18,606,-50}, 
    {18,611,-50}, {18,617,-50}, {18,626,-50}, {18,660,-70}, 
    {18,670,-50}, {18,688,-100}, {18,689,-100}, {18,694,-100}, 
    {18,3819,-30}, {18,3820,-30}, {18,3827,-30}, {18,3834,-30}, 
    {18,3842,-30}, {18,3905,-40}, {18,3954,-40}, {18,4034,-40}, 
    {18,4037,-40}, {18,4053,-40}, {29,66,-30}, {29,72,-30}, 
    {29,205,-30}, {29,395,-30}, {29,398,-30}, {29,406,-30}, 
    {29,416,-30}, {29,421,-30}, {29,451,-30}, {29,457,-30}, 
    {29,476,-30}, {29,572,-120}, {29,605,-50}, {29,606,-50}, 
    {29,611,-50}, {29,617,-50}, {29,626,-50}, {29,660,-70}, 
    {29,670,-50}, {29,688,-100}, {29,689,-100}, {29,694,-100}, 
    {29,3819,-30}, {29,3820,-30}, {29,3827,-30}, {29,3834,-30}, 
    {29,3842,-30}, {29,3905,-40}, {29,3954,-40}, {29,4034,-40}, 
    {29,4037,-40}, {29,4053,-40}, {35,66,-30}, {35,72,-30}, 
    {35,205,-30}, {35,395,-30}, {35,398,-30}, {35,406,-30}, 
    {35,416,-30}, {35,421,-30}, {35,451,-30}, {35,457,-30}, 
    {35,476,-30}, {35,572,-120}, {35,605,-50}, {35,606,-50}, 
    {35,611,-50}, {35,617,-50}, {35,626,-50}, {35,660,-70}, 
    {35,670,-50}, {35,688,-100}, {35,689,-100}, {35,694,-100}, 
    {35,3819,-30}, {35,3820,-30}, {35,3827,-30}, {35,3834,-30}, 
    {35,3842,-30}, {35,3905,-40}, {35,3954,-40}, {35,4034,-40}, 
    {35,4037,-40}, {35,4053,-40}, {45,66,-30}, {45,72,-30}, 
    {45,205,-30}, {45,395,-30}, {45,398,-30}, {45,406,-30}, 
    {45,416,-30}, {45,421,-30}, {45,451,-30}, {45,457,-30}, 
    {45,476,-30}, {45,572,-120}, {45,605,-50}, {45,606,-50}, 
    {45,611,-50}, {45,617,-50}, {45,626,-50}, {45,660,-70}, 
    {45,670,-50}, {45,688,-100}, {45,689,-100}, {45,694,-100}, 
    {45,3819,-30}, {45,3820,-30}, {45,3827,-30}, {45,3834,-30}, 
    {45,3842,-30}, {45,3905,-40}, {45,3954,-40}, {45,4034,-40}, 
    {45,4037,-40}, {45,4053,-40}, {50,66,-30}, {50,72,-30}, 
    {50,205,-30}, {50,395,-30}, {50,398,-30}, {50,406,-30}, 
    {50,416,-30}, {50,421,-30}, {50,451,-30}, {50,457,-30}, 
    {50,476,-30}, {50,572,-120}, {50,605,-50}, {50,606,-50}, 
    {50,611,-50}, {50,617,-50}, {50,626,-50}, {50,660,-70}, 
    {50,670,-50}, {50,688,-100}, {50,689,-100}, {50,694,-100}, 
    {50,3819,-30}, {50,3820,-30}, {50,3827,-30}, {50,3834,-30}, 
    {50,3842,-30}, {50,3905,-40}, {50,3954,-40}, {50,4034,-40}, 
    {50,4037,-40}, {50,4053,-40}, {53,605,-10}, {53,606,-10}, 
    {53,611,-10}, {53,617,-10}, {53,626,-10}, {53,1411,-20}, 
    {53,2998,-20}, {66,1411,-30}, {66,2998,-30}, {72,1411,-30}, 
    {72,2998,-30}, {95,2,-40}, {95,7,-40}, {95,18,-40}, {95,29,-40}, 
    {95,35,-40}, {95,45,-40}, {95,50,-40}, {95,660,-70}, {95,670,-40}, 
    {95,688,-90}, {95,689,-90}, {95,694,-90}, {95,1411,-70}, 
    {95,2998,-70}, {194,2,-80}, {194,7,-80}, {194,18,-80}, 
    {194,29,-80}, {194,35,-80}, {194,45,-80}, {194,50,-80}, 
    {194,736,-50}, {194,738,-50}, {194,760,-50}, {194,777,-50}, 
    {194,1032,-50}, {194,1113,-50}, {194,1175,-50}, {194,1411,-150}, 
    {194,1632,-30}, {194,1633,-30}, {194,1647,-30}, {194,1657,-30}, 
    {194,1664,-30}, {194,2797,-30}, {194,2798,-30}, {194,2812,-30}, 
    {194,2822,-30}, {194,2829,-30}, {194,2902,-30}, {194,2910,-30}, 
    {194,2998,-150}, {194,3151,-45}, {303,2,-20}, {303,7,-20}, 
    {303,18,-20}, {303,29,-20}, {303,35,-20}, {303,45,-20}, 
    {303,50,-20}, {303,736,-20}, {303,738,-20}, {303,760,-20}, 
    {303,777,-20}, {303,1032,-20}, {303,1113,-20}, {303,1175,-20}, 
    {303,1411,-30}, {303,2998,-30}, {303,3819,-20}, {303,3820,-20}, 
    {303,3827,-20}, {303,3834,-20}, {303,3842,-20}, {311,395,-50}, 
    {311,398,-50}, {311,406,-50}, {311,416,-50}, {311,421,-50}, 
    {311,451,-50}, {311,457,-50}, {311,1632,-40}, {311,1633,-40}, 
    {311,1647,-40}, {311,1657,-40}, {311,1664,-40}, {311,2797,-40}, 
    {311,2798,-40}, {311,2812,-40}, {311,2822,-40}, {311,2829,-40}, 
    {311,2902,-40}, {311,2910,-40}, {311,3819,-30}, {311,3820,-30}, 
    {311,3827,-30}, {311,3834,-30}, {311,3842,-30}, {311,4034,-50}, 
    {311,4037,-50}, {311,4053,-50}, {339,572,-110}, {339,660,-110}, 
    {339,670,-70}, {339,688,-140}, {339,689,-140}, {339,694,-140}, 
    {339,3142,-140}, {339,3146,-160}, {339,4034,-30}, {339,4037,-30}, 
    {339,4053,-30}, {358,572,-110}, {358,660,-110}, {358,670,-70}, 
    {358,688,-140}, {358,689,-140}, {358,694,-140}, {358,3142,-140}, 
    {358,3146,-160}, {358,4034,-30}, {358,4037,-30}, {358,4053,-30}, 
    {395,2,-20}, {395,7,-20}, {395,18,-20}, {395,29,-20}, {395,35,-20}, 
    {395,45,-20}, {395,50,-20}, {395,572,-40}, {395,660,-50}, 
    {395,670,-30}, {395,680,-60}, {395,688,-70}, {395,689,-70}, 
    {395,694,-70}, {395,1411,-40}, {395,2998,-40}, {398,2,-20}, 
    {398,7,-20}, {398,18,-20}, {398,29,-20}, {398,35,-20}, 
    {398,45,-20}, {398,50,-20}, {398,572,-40}, {398,660,-50}, 
    {398,670,-30}, {398,680,-60}, {398,688,-70}, {398,689,-70}, 
    {398,694,-70}, {398,1411,-40}, {398,2998,-40}, {406,2,-20}, 
    {406,7,-20}, {406,18,-20}, {406,29,-20}, {406,35,-20}, 
    {406,45,-20}, {406,50,-20}, {406,572,-40}, {406,660,-50}, 
    {406,670,-30}, {406,680,-60}, {406,688,-70}, {406,689,-70}, 
    {406,694,-70}, {406,1411,-40}, {406,2998,-40}, {416,2,-20}, 
    {416,7,-20}, {416,18,-20}, {416,29,-20}, {416,35,-20}, 
    {416,45,-20}, {416,50,-20}, {416,572,-40}, {416,660,-50}, 
    {416,670,-30}, {416,680,-60}, {416,688,-70}, {416,689,-70}, 
    {416,694,-70}, {416,1411,-40}, {416,2998,-40}, {421,2,-20}, 
    {421,7,-20}, {421,18,-20}, {421,29,-20}, {421,35,-20}, 
    {421,45,-20}, {421,50,-20}, {421,572,-40}, {421,660,-50}, 
    {421,670,-30}, {421,680,-60}, {421,688,-70}, {421,689,-70}, 
    {421,694,-70}, {421,1411,-40}, {421,2998,-40}, {451,2,-20}, 
    {451,7,-20}, {451,18,-20}, {451,29,-20}, {451,35,-20}, 
    {451,45,-20}, {451,50,-20}, {451,572,-40}, {451,660,-50}, 
    {451,670,-30}, {451,680,-60}, {451,688,-70}, {451,689,-70}, 
    {451,694,-70}, {451,1411,-40}, {451,2998,-40}, {457,2,-20}, 
    {457,7,-20}, {457,18,-20}, {457,29,-20}, {457,35,-20}, 
    {457,45,-20}, {457,50,-20}, {457,572,-40}, {457,660,-50}, 
    {457,670,-30}, {457,680,-60}, {457,688,-70}, {457,689,-70}, 
    {457,694,-70}, {457,1411,-40}, {457,2998,-40}, {461,2,-120}, 
    {461,7,-120}, {461,18,-120}, {461,29,-120}, {461,35,-120}, 
    {461,45,-120}, {461,50,-120}, {461,736,-40}, {461,738,-40}, 
    {461,760,-40}, {461,777,-40}, {461,1032,-40}, {461,1113,-40}, 
    {461,1175,-40}, {461,1411,-180}, {461,1632,-50}, {461,1633,-50}, 
    {461,1647,-50}, {461,1657,-50}, {461,1664,-50}, {461,2797,-50}, 
    {461,2798,-50}, {461,2812,-50}, {461,2822,-50}, {461,2829,-50}, 
    {461,2902,-50}, {461,2910,-50}, {461,2998,-180}, {476,605,-10}, 
    {476,606,-10}, {476,611,-10}, {476,617,-10}, {476,626,-10}, 
    {480,395,-20}, {480,398,-20}, {480,406,-20}, {480,416,-20}, 
    {480,421,-20}, {480,451,-20}, {480,457,-20}, {480,572,-30}, 
    {480,605,-40}, {480,606,-40}, {480,611,-40}, {480,617,-40}, 
    {480,626,-40}, {480,660,-50}, {480,670,-30}, {480,688,-50}, 
    {480,689,-50}, {480,694,-50}, {501,1411,-20}, {501,2998,-20}, 
    {545,1411,-20}, {545,2998,-20}, {572,2,-120}, {572,7,-120}, 
    {572,18,-120}, {572,29,-120}, {572,35,-120}, {572,45,-120}, 
    {572,50,-120}, {572,395,-40}, {572,398,-40}, {572,406,-40}, 
    {572,416,-40}, {572,421,-40}, {572,451,-40}, {572,457,-40}, 
    {572,736,-120}, {572,738,-120}, {572,760,-120}, {572,777,-120}, 
    {572,1032,-120}, {572,1113,-120}, {572,1175,-60}, {572,1404,-20}, 
    {572,1411,-120}, {572,1632,-120}, {572,1633,-120}, {572,1647,-120}, 
    {572,1657,-120}, {572,1664,-60}, {572,2128,-140}, {572,2797,-120}, 
    {572,2798,-120}, {572,2812,-120}, {572,2822,-120}, {572,2829,-120}, 
    {572,2902,-120}, {572,2910,-60}, {572,2998,-120}, {572,3151,-120}, 
    {572,3354,-20}, {572,3819,-120}, {572,3820,-120}, {572,3827,-120}, 
    {572,3834,-120}, {572,3842,-120}, {572,3954,-120}, {572,4034,-120}, 
    {572,4037,-120}, {572,4053,-60}, {605,2,-40}, {605,7,-40}, 
    {605,18,-40}, {605,29,-40}, {605,35,-40}, {605,45,-40}, 
    {605,50,-40}, {605,1411,-40}, {605,2998,-40}, {606,2,-40}, 
    {606,7,-40}, {606,18,-40}, {606,29,-40}, {606,35,-40}, 
    {606,45,-40}, {606,50,-40}, {606,1411,-40}, {606,2998,-40}, 
    {611,2,-40}, {611,7,-40}, {611,18,-40}, {611,29,-40}, {611,35,-40}, 
    {611,45,-40}, {611,50,-40}, {611,1411,-40}, {611,2998,-40}, 
    {617,2,-40}, {617,7,-40}, {617,18,-40}, {617,29,-40}, {617,35,-40}, 
    {617,45,-40}, {617,50,-40}, {617,1411,-40}, {617,2998,-40}, 
    {626,2,-40}, {626,7,-40}, {626,18,-40}, {626,29,-40}, {626,35,-40}, 
    {626,45,-40}, {626,50,-40}, {626,1411,-40}, {626,2998,-40}, 
    {660,2,-80}, {660,7,-80}, {660,18,-80}, {660,29,-80}, {660,35,-80}, 
    {660,45,-80}, {660,50,-80}, {660,205,-40}, {660,395,-40}, 
    {660,398,-40}, {660,406,-40}, {660,416,-40}, {660,421,-40}, 
    {660,451,-40}, {660,457,-40}, {660,736,-70}, {660,738,-70}, 
    {660,760,-70}, {660,777,-70}, {660,1032,-70}, {660,1113,-70}, 
    {660,1175,-70}, {660,1404,-40}, {660,1411,-125}, {660,1632,-80}, 
    {660,1633,-80}, {660,1647,-80}, {660,1657,-80}, {660,1664,-80}, 
    {660,2128,-80}, {660,2797,-80}, {660,2798,-80}, {660,2812,-80}, 
    {660,2822,-80}, {660,2829,-80}, {660,2902,-80}, {660,2910,-80}, 
    {660,2998,-125}, {660,3354,-40}, {660,3819,-70}, {660,3820,-70}, 
    {660,3827,-70}, {660,3834,-70}, {660,3842,-70}, {670,2,-50}, 
    {670,7,-50}, {670,18,-50}, {670,29,-50}, {670,35,-50}, 
    {670,45,-50}, {670,50,-50}, {670,395,-20}, {670,398,-20}, 
    {670,406,-20}, {670,416,-20}, {670,421,-20}, {670,451,-20}, 
    {670,457,-20}, {670,736,-40}, {670,738,-40}, {670,760,-40}, 
    {670,777,-40}, {670,1032,-40}, {670,1113,-40}, {670,1175,-40}, 
    {670,1411,-80}, {670,1632,-30}, {670,1633,-30}, {670,1647,-30}, 
    {670,1657,-30}, {670,1664,-30}, {670,2128,-40}, {670,2797,-30}, 
    {670,2798,-30}, {670,2812,-30}, {670,2822,-30}, {670,2829,-30}, 
    {670,2902,-30}, {670,2910,-30}, {670,2998,-80}, {670,3819,-30}, 
    {670,3820,-30}, {670,3827,-30}, {670,3834,-30}, {670,3842,-30}, 
    {670,4034,-20}, {670,4037,-20}, {670,4053,-20}, {688,2,-110}, 
    {688,7,-110}, {688,18,-110}, {688,29,-110}, {688,35,-110}, 
    {688,45,-110}, {688,50,-110}, {688,395,-85}, {688,398,-85}, 
    {688,406,-85}, {688,416,-85}, {688,421,-85}, {688,451,-85}, 
    {688,457,-85}, {688,736,-140}, {688,738,-140}, {688,760,-140}, 
    {688,777,-140}, {688,1032,-140}, {688,1113,-140}, {688,1175,-140}, 
    {688,1404,-60}, {688,1411,-140}, {688,1632,-140}, {688,1633,-140}, 
    {688,1647,-140}, {688,1657,-140}, {688,1664,-140}, {688,2128,-140}, 
    {688,2134,-20}, {688,2135,-20}, {688,2797,-140}, {688,2798,-140}, 
    {688,2812,-140}, {688,2822,-140}, {688,2829,-140}, {688,2902,-140}, 
    {688,2910,-140}, {688,2998,-140}, {688,3354,-60}, {688,3819,-110}, 
    {688,3820,-110}, {688,3827,-110}, {688,3834,-110}, {688,3842,-110}, 
    {689,2,-110}, {689,7,-110}, {689,18,-110}, {689,29,-110}, 
    {689,35,-110}, {689,45,-110}, {689,50,-110}, {689,395,-85}, 
    {689,398,-85}, {689,406,-85}, {689,416,-85}, {689,421,-85}, 
    {689,451,-85}, {689,457,-85}, {689,736,-140}, {689,738,-140}, 
    {689,760,-140}, {689,777,-140}, {689,1032,-140}, {689,1113,-140}, 
    {689,1175,-70}, {689,1404,-60}, {689,1411,-140}, {689,1632,-140}, 
    {689,1633,-140}, {689,1647,-140}, {689,1657,-140}, {689,1664,-140}, 
    {689,2128,-140}, {689,2134,-20}, {689,2135,-20}, {689,2797,-140}, 
    {689,2798,-140}, {689,2812,-140}, {689,2822,-140}, {689,2829,-140}, 
    {689,2902,-140}, {689,2910,-140}, {689,2998,-140}, {689,3354,-60}, 
    {689,3819,-110}, {689,3820,-110}, {689,3827,-110}, {689,3834,-110}, 
    {689,3842,-110}, {694,2,-110}, {694,7,-110}, {694,18,-110}, 
    {694,29,-110}, {694,35,-110}, {694,45,-110}, {694,50,-110}, 
    {694,395,-85}, {694,398,-85}, {694,406,-85}, {694,416,-85}, 
    {694,421,-85}, {694,451,-85}, {694,457,-85}, {694,736,-140}, 
    {694,738,-140}, {694,760,-140}, {694,777,-140}, {694,1032,-140}, 
    {694,1113,-140}, {694,1175,-70}, {694,1404,-60}, {694,1411,-140}, 
    {694,1632,-140}, {694,1633,-140}, {694,1647,-140}, {694,1657,-140}, 
    {694,1664,-140}, {694,2128,-140}, {694,2134,-20}, {694,2135,-20}, 
    {694,2797,-140}, {694,2798,-140}, {694,2812,-140}, {694,2822,-140}, 
    {694,2829,-140}, {694,2902,-140}, {694,2910,-140}, {694,2998,-140}, 
    {694,3354,-60}, {694,3819,-110}, {694,3820,-110}, {694,3827,-110}, 
    {694,3834,-110}, {694,3842,-110}, {736,3905,-20}, {736,3954,-20}, 
    {736,4034,-30}, {736,4037,-30}, {736,4053,-30}, {738,3905,-20}, 
    {738,3954,-20}, {738,4034,-30}, {738,4037,-30}, {738,4053,-30}, 
    {760,3905,-20}, {760,3954,-20}, {760,4034,-30}, {760,4037,-30}, 
    {760,4053,-30}, {777,3905,-20}, {777,3954,-20}, {777,4034,-30}, 
    {777,4037,-30}, {777,4053,-30}, {1032,3905,-20}, {1032,3954,-20}, 
    {1032,4034,-30}, {1032,4037,-30}, {1032,4053,-30}, {1113,3905,-20}, 
    {1113,3954,-20}, {1113,4034,-30}, {1113,4037,-30}, {1113,4053,-30}, 
    {1175,3905,-20}, {1175,3954,-20}, {1175,4034,-30}, {1175,4037,-30}, 
    {1175,4053,-30}, {1194,1194,-10}, {1194,1411,-40}, {1194,2418,-20}, 
    {1194,2503,-20}, {1194,2998,-40}, {1194,3819,-20}, {1194,3820,-20}, 
    {1194,3827,-20}, {1194,3834,-20}, {1194,3842,-20}, {1194,3905,-20}, 
    {1194,4034,-20}, {1194,4037,-20}, {1194,4053,-20}, {1314,1411,-15}, 
    {1314,2304,-20}, {1334,1411,-15}, {1334,2304,-20}, {1404,3501,-50}, 
    {1411,3142,-100}, {1411,3146,-100}, {1632,1411,-15}, 
    {1632,2998,-15}, {1632,3905,-30}, {1632,3954,-20}, {1632,4023,-30}, 
    {1632,4034,-20}, {1632,4037,-20}, {1632,4053,-20}, {1633,1411,-15}, 
    {1633,2998,-15}, {1633,3905,-30}, {1633,3954,-20}, {1633,4023,-30}, 
    {1633,4034,-20}, {1633,4037,-20}, {1633,4053,-20}, {1647,1411,-15}, 
    {1647,2998,-15}, {1647,3905,-30}, {1647,3954,-20}, {1647,4023,-30}, 
    {1647,4034,-20}, {1647,4037,-20}, {1647,4053,-20}, {1657,1411,-15}, 
    {1657,2998,-15}, {1657,3905,-30}, {1657,3954,-20}, {1657,4023,-30}, 
    {1657,4034,-20}, {1657,4037,-20}, {1657,4053,-20}, {1664,1411,-15}, 
    {1664,2998,-15}, {1664,3905,-30}, {1664,3954,-20}, {1664,4023,-30}, 
    {1664,4034,-20}, {1664,4037,-20}, {1664,4053,-20}, {1789,736,-30}, 
    {1789,738,-30}, {1789,760,-30}, {1789,777,-30}, {1789,1032,-30}, 
    {1789,1113,-30}, {1789,1175,-30}, {1789,1411,-30}, {1789,1610,-28}, 
    {1789,1632,-30}, {1789,1633,-30}, {1789,1647,-30}, {1789,1657,-30}, 
    {1789,1664,-30}, {1789,2797,-30}, {1789,2798,-30}, {1789,2812,-30}, 
    {1789,2822,-30}, {1789,2829,-30}, {1789,2902,-30}, {1789,2910,-30}, 
    {1789,2998,-30}, {1789,3142,60}, {1789,3146,50}, {1888,3151,-10}, 
    {1990,4034,-30}, {1990,4037,-30}, {1990,4053,-30}, {2304,1632,-20}, 
    {2304,1633,-20}, {2304,1647,-20}, {2304,1657,-20}, {2304,1664,-20}, 
    {2304,2797,-20}, {2304,2798,-20}, {2304,2812,-20}, {2304,2822,-20}, 
    {2304,2829,-20}, {2304,2902,-20}, {2304,2910,-20}, {2513,3819,-10}, 
    {2513,3820,-10}, {2513,3827,-10}, {2513,3834,-10}, {2513,3842,-10}, 
    {2513,4034,-15}, {2513,4037,-15}, {2513,4053,-15}, {2650,3819,-10}, 
    {2650,3820,-10}, {2650,3827,-10}, {2650,3834,-10}, {2650,3842,-10}, 
    {2650,3905,-20}, {2650,4034,-15}, {2650,4037,-15}, {2650,4053,-15}, 
    {2772,3819,-10}, {2772,3820,-10}, {2772,3827,-10}, {2772,3834,-10}, 
    {2772,3842,-10}, {2772,3905,-20}, {2772,4034,-15}, {2772,4037,-15}, 
    {2772,4053,-15}, {2797,1411,-40}, {2797,2998,-40}, {2797,3905,-15}, 
    {2797,3954,-15}, {2797,4023,-30}, {2797,4034,-30}, {2797,4037,-30}, 
    {2797,4053,-30}, {2798,1411,-40}, {2798,2998,-40}, {2798,3905,-15}, 
    {2798,3954,-15}, {2798,4023,-30}, {2798,4034,-30}, {2798,4037,-30}, 
    {2798,4053,-30}, {2812,1411,-40}, {2812,2998,-40}, {2812,3905,-15}, 
    {2812,3954,-15}, {2812,4023,-30}, {2812,4034,-30}, {2812,4037,-30}, 
    {2812,4053,-30}, {2822,1411,-40}, {2822,2998,-40}, {2822,3905,-15}, 
    {2822,3954,-15}, {2822,4023,-30}, {2822,4034,-30}, {2822,4037,-30}, 
    {2822,4053,-30}, {2829,1411,-40}, {2829,2998,-40}, {2829,3905,-15}, 
    {2829,3954,-15}, {2829,4023,-30}, {2829,4034,-30}, {2829,4037,-30}, 
    {2829,4053,-30}, {2902,736,-55}, {2902,738,-55}, {2902,760,-55}, 
    {2902,777,-55}, {2902,1032,-55}, {2902,1113,-55}, {2902,1175,-55}, 
    {2902,1194,-55}, {2902,1314,-55}, {2902,1334,-55}, {2902,1411,-95}, 
    {2902,1483,-55}, {2902,1632,-55}, {2902,1633,-55}, {2902,1647,-55}, 
    {2902,1657,-55}, {2902,1664,-55}, {2902,1789,-55}, {2902,1888,-55}, 
    {2902,1990,-55}, {2902,2134,-55}, {2902,2135,-55}, {2902,2142,-55}, 
    {2902,2198,-55}, {2902,2209,-55}, {2902,2276,-55}, {2902,2304,-55}, 
    {2902,2418,-55}, {2902,2503,-55}, {2902,2513,-55}, {2902,2650,-55}, 
    {2902,2772,-55}, {2902,2797,-55}, {2902,2798,-55}, {2902,2812,-55}, 
    {2902,2822,-55}, {2902,2829,-55}, {2902,2902,-55}, {2902,2910,-55}, 
    {2902,2924,-55}, {2902,2998,-95}, {2902,3086,-55}, {2902,3151,-55}, 
    {2902,3282,-55}, {2902,3320,-55}, {2902,3559,-55}, {2902,3819,-55}, 
    {2902,3820,-55}, {2902,3827,-55}, {2902,3834,-55}, {2902,3842,-55}, 
    {2902,3905,-70}, {2902,3954,-70}, {2902,4023,-85}, {2902,4034,-70}, 
    {2902,4037,-70}, {2902,4053,-70}, {2902,4135,-55}, {2902,4156,-55}, 
    {2910,1411,-40}, {2910,2998,-40}, {2910,3905,-15}, {2910,3954,-15}, 
    {2910,4023,-30}, {2910,4034,-30}, {2910,4037,-30}, {2910,4053,-30}, 
    {2924,1411,-35}, {2924,2998,-35}, {2924,4034,-30}, {2924,4037,-30}, 
    {2924,4053,-30}, {2998,3142,-100}, {2998,3146,-100}, 
    {2998,3501,-60}, {3142,3501,-40}, {3143,3143,-57}, {3146,1483,-50}, 
    {3146,3146,-57}, {3146,3151,-50}, {3146,3282,-50}, {3146,3320,-50}, 
    {3146,3501,-70}, {3151,736,-10}, {3151,738,-10}, {3151,760,-10}, 
    {3151,777,-10}, {3151,1032,-10}, {3151,1113,-10}, {3151,1175,-10}, 
    {3151,1404,30}, {3151,1411,-50}, {3151,2134,15}, {3151,2135,15}, 
    {3151,2142,15}, {3151,2198,15}, {3151,2209,15}, {3151,2304,15}, 
    {3151,2418,15}, {3151,2503,15}, {3151,2513,25}, {3151,2650,25}, 
    {3151,2772,25}, {3151,2924,30}, {3151,2998,-50}, {3151,3354,30}, 
    {3151,3559,40}, {3151,3819,15}, {3151,3820,15}, {3151,3827,15}, 
    {3151,3834,15}, {3151,3842,15}, {3151,3905,30}, {3151,4034,30}, 
    {3151,4037,30}, {3151,4053,30}, {3282,1411,-15}, {3282,2998,-15}, 
    {3282,3954,-30}, {3320,1411,-15}, {3320,2998,-15}, {3320,3954,-30}, 
    {3354,3501,-50}, {3501,572,-50}, {3501,660,-50}, {3501,670,-40}, 
    {3501,688,-90}, {3501,689,-90}, {3501,694,-90}, {3501,3138,-30}, 
    {3501,3143,-60}, {3905,736,-25}, {3905,738,-25}, {3905,760,-25}, 
    {3905,777,-25}, {3905,1032,-25}, {3905,1113,-25}, {3905,1175,-25}, 
    {3905,1411,-80}, {3905,1632,-25}, {3905,1633,-25}, {3905,1647,-25}, 
    {3905,1657,-25}, {3905,1664,-25}, {3905,2797,-25}, {3905,2798,-25}, 
    {3905,2812,-25}, {3905,2822,-25}, {3905,2829,-25}, {3905,2902,-25}, 
    {3905,2910,-25}, {3905,2998,-80}, {3954,736,-15}, {3954,738,-15}, 
    {3954,760,-15}, {3954,777,-15}, {3954,1032,-15}, {3954,1113,-15}, 
    {3954,1175,-15}, {3954,1411,-60}, {3954,1632,-10}, {3954,1633,-10}, 
    {3954,1647,-10}, {3954,1657,-10}, {3954,1664,-10}, {3954,2797,-10}, 
    {3954,2798,-10}, {3954,2812,-10}, {3954,2822,-10}, {3954,2829,-10}, 
    {3954,2902,-10}, {3954,2910,-10}, {3954,2998,-60}, {4023,1632,-30}, 
    {4023,1633,-30}, {4023,1647,-30}, {4023,1657,-30}, {4023,1664,-30}, 
    {4034,736,-20}, {4034,738,-20}, {4034,760,-20}, {4034,777,-20}, 
    {4034,1032,-20}, {4034,1113,-20}, {4034,1175,-20}, 
    {4034,1411,-100}, {4034,1632,-20}, {4034,1633,-20}, 
    {4034,1647,-20}, {4034,1657,-20}, {4034,1664,-20}, {4034,2797,-20}, 
    {4034,2798,-20}, {4034,2812,-20}, {4034,2822,-20}, {4034,2829,-20}, 
    {4034,2902,-20}, {4034,2910,-20}, {4034,2998,-100}, {4037,736,-20}, 
    {4037,738,-20}, {4037,760,-20}, {4037,777,-20}, {4037,1032,-20}, 
    {4037,1113,-20}, {4037,1175,-20}, {4037,1411,-100}, 
    {4037,1632,-20}, {4037,1633,-20}, {4037,1647,-20}, {4037,1657,-20}, 
    {4037,1664,-20}, {4037,2797,-20}, {4037,2798,-20}, {4037,2812,-20}, 
    {4037,2822,-20}, {4037,2829,-20}, {4037,2902,-20}, {4037,2910,-20}, 
    {4037,2998,-100}, {4053,736,-20}, {4053,738,-20}, {4053,760,-20}, 
    {4053,777,-20}, {4053,1032,-20}, {4053,1113,-20}, {4053,1175,-20}, 
    {4053,1411,-100}, {4053,1632,-20}, {4053,1633,-20}, 
    {4053,1647,-20}, {4053,1657,-20}, {4053,1664,-20}, {4053,2797,-20}, 
    {4053,2798,-20}, {4053,2812,-20}, {4053,2822,-20}, {4053,2829,-20}, 
    {4053,2902,-20}, {4053,2910,-20}, {4053,2998,-100}, 
    {4135,1632,-15}, {4135,1633,-15}, {4135,1647,-15}, {4135,1657,-15}, 
    {4135,1664,-15}, {4135,2797,-15}, {4135,2798,-15}, {4135,2812,-15}, 
    {4135,2822,-15}, {4135,2829,-15}, {4135,2902,-15}, {4135,2910,-15}, 
    {4156,1632,-15}, {4156,1633,-15}, {4156,1647,-15}, {4156,1657,-15}, 
    {4156,1664,-15}, {4156,2797,-15}, {4156,2798,-15}, {4156,2812,-15}, 
    {4156,2822,-15}, {4156,2829,-15}, {4156,2902,-15}, {4156,2910,-15}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature helvetica_oblique_ligs[] = {
    {1789,2134,1809}, {1789,2418,1852}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair helvetica_bold_kerns[] = {
    {2,66,-40}, {2,72,-40}, {2,205,-50}, {2,395,-40}, {2,398,-40}, 
    {2,406,-40}, {2,416,-40}, {2,421,-40}, {2,451,-40}, {2,457,-40}, 
    {2,476,-40}, {2,572,-90}, {2,605,-50}, {2,606,-50}, {2,611,-50}, 
    {2,617,-50}, {2,626,-50}, {2,660,-80}, {2,670,-60}, {2,688,-110}, 
    {2,689,-110}, {2,694,-110}, {2,3819,-30}, {2,3820,-30}, 
    {2,3827,-30}, {2,3834,-30}, {2,3842,-30}, {2,3905,-40}, 
    {2,3954,-30}, {2,4034,-30}, {2,4037,-30}, {2,4053,-30}, {7,66,-40}, 
    {7,72,-40}, {7,205,-50}, {7,395,-40}, {7,398,-40}, {7,406,-40}, 
    {7,416,-40}, {7,421,-40}, {7,451,-40}, {7,457,-40}, {7,476,-40}, 
    {7,572,-90}, {7,605,-50}, {7,606,-50}, {7,611,-50}, {7,617,-50}, 
    {7,626,-50}, {7,660,-80}, {7,670,-60}, {7,688,-110}, {7,689,-110}, 
    {7,694,-110}, {7,3819,-30}, {7,3820,-30}, {7,3827,-30}, 
    {7,3834,-30}, {7,3842,-30}, {7,3905,-40}, {7,3954,-30}, 
    {7,4034,-30}, {7,4037,-30}, {7,4053,-30}, {18,66,-40}, {18,72,-40}, 
    {18,205,-50}, {18,395,-40}, {18,398,-40}, {18,406,-40}, 
    {18,416,-40}, {18,421,-40}, {18,451,-40}, {18,457,-40}, 
    {18,476,-40}, {18,572,-90}, {18,605,-50}, {18,606,-50}, 
    {18,611,-50}, {18,617,-50}, {18,626,-50}, {18,660,-80}, 
    {18,670,-60}, {18,688,-110}, {18,689,-110}, {18,694,-110}, 
    {18,3819,-30}, {18,3820,-30}, {18,3827,-30}, {18,3834,-30}, 
    {18,3842,-30}, {18,3905,-40}, {18,3954,-30}, {18,4034,-30}, 
    {18,4037,-30}, {18,4053,-30}, {29,66,-40}, {29,72,-40}, 
    {29,205,-50}, {29,395,-40}, {29,398,-40}, {29,406,-40}, 
    {29,416,-40}, {29,421,-40}, {29,451,-40}, {29,457,-40}, 
    {29,476,-40}, {29,572,-90}, {29,605,-50}, {29,606,-50}, 
    {29,611,-50}, {29,617,-50}, {29,626,-50}, {29,660,-80}, 
    {29,670,-60}, {29,688,-110}, {29,689,-110}, {29,694,-110}, 
    {29,3819,-30}, {29,3820,-30}, {29,3827,-30}, {29,3834,-30}, 
    {29,3842,-30}, {29,3905,-40}, {29,3954,-30}, {29,4034,-30}, 
    {29,4037,-30}, {29,4053,-30}, {35,66,-40}, {35,72,-40}, 
    {35,205,-50}, {35,395,-40}, {35,398,-40}, {35,406,-40}, 
    {35,416,-40}, {35,421,-40}, {35,451,-40}, {35,457,-40}, 
    {35,476,-40}, {35,572,-90}, {35,605,-50}, {35,606,-50}, 
    {35,611,-50}, {35,617,-50}, {35,626,-50}, {35,660,-80}, 
    {35,670,-60}, {35,688,-110}, {35,689,-110}, {35,694,-110}, 
    {35,3819,-30}, {35,3820,-30}, {35,3827,-30}, {35,3834,-30}, 
    {35,3842,-30}, {35,3905,-40}, {35,3954,-30}, {35,4034,-30}, 
    {35,4037,-30}, {35,4053,-30}, {45,66,-40}, {45,72,-40}, 
    {45,205,-50}, {45,395,-40}, {45,398,-40}, {45,406,-40}, 
    {45,416,-40}, {45,421,-40}, {45,451,-40}, {45,457,-40}, 
    {45,476,-40}, {45,572,-90}, {45,605,-50}, {45,606,-50}, 
    {45,611,-50}, {45,617,-50}, {45,626,-50}, {45,660,-80}, 
    {45,670,-60}, {45,688,-110}, {45,689,-110}, {45,694,-110}, 
    {45,3819,-30}, {45,3820,-30}, {45,3827,-30}, {45,3834,-30}, 
    {45,3842,-30}, {45,3905,-40}, {45,3954,-30}, {45,4034,-30}, 
    {45,4037,-30}, {45,4053,-30}, {50,66,-40}, {50,72,-40}, 
    {50,205,-50}, {50,395,-40}, {50,398,-40}, {50,406,-40}, 
    {50,416,-40}, {50,421,-40}, {50,451,-40}, {50,457,-40}, 
    {50,476,-40}, {50,572,-90}, {50,605,-50}, {50,606,-50}, 
    {50,611,-50}, {50,617,-50}, {50,626,-50}, {50,660,-80}, 
    {50,670,-60}, {50,688,-110}, {50,689,-110}, {50,694,-110}, 
    {50,3819,-30}, {50,3820,-30}, {50,3827,-30}, {50,3834,-30}, 
    {50,3842,-30}, {50,3905,-40}, {50,3954,-30}, {50,4034,-30}, 
    {50,4037,-30}, {50,4053,-30}, {53,2,-30}, {53,7,-30}, {53,18,-30}, 
    {53,29,-30}, {53,35,-30}, {53,45,-30}, {53,50,-30}, {53,605,-10}, 
    {53,606,-10}, {53,611,-10}, {53,617,-10}, {53,626,-10}, {95,2,-40}, 
    {95,7,-40}, {95,18,-40}, {95,29,-40}, {95,35,-40}, {95,45,-40}, 
    {95,50,-40}, {95,660,-40}, {95,670,-40}, {95,688,-70}, 
    {95,689,-70}, {95,694,-70}, {95,1411,-30}, {95,2998,-30}, 
    {194,2,-80}, {194,7,-80}, {194,18,-80}, {194,29,-80}, {194,35,-80}, 
    {194,45,-80}, {194,50,-80}, {194,736,-20}, {194,738,-20}, 
    {194,760,-20}, {194,777,-20}, {194,1032,-20}, {194,1113,-20}, 
    {194,1175,-20}, {194,1411,-100}, {194,2998,-100}, {303,2,-20}, 
    {303,7,-20}, {303,18,-20}, {303,29,-20}, {303,35,-20}, 
    {303,45,-20}, {303,50,-20}, {303,1411,-20}, {303,2998,-20}, 
    {303,3819,-20}, {303,3820,-20}, {303,3827,-20}, {303,3834,-20}, 
    {303,3842,-20}, {311,395,-30}, {311,398,-30}, {311,406,-30}, 
    {311,416,-30}, {311,421,-30}, {311,451,-30}, {311,457,-30}, 
    {311,1632,-15}, {311,1633,-15}, {311,1647,-15}, {311,1657,-15}, 
    {311,1664,-15}, {311,2797,-35}, {311,2798,-35}, {311,2812,-35}, 
    {311,2822,-35}, {311,2829,-35}, {311,2902,-35}, {311,2910,-35}, 
    {311,3819,-30}, {311,3820,-30}, {311,3827,-30}, {311,3834,-30}, 
    {311,3842,-30}, {311,4034,-40}, {311,4037,-40}, {311,4053,-40}, 
    {339,572,-90}, {339,660,-110}, {339,670,-80}, {339,688,-120}, 
    {339,689,-120}, {339,694,-120}, {339,3142,-140}, {339,3146,-140}, 
    {339,4034,-30}, {339,4037,-30}, {339,4053,-30}, {358,572,-90}, 
    {358,660,-110}, {358,670,-80}, {358,688,-120}, {358,689,-120}, 
    {358,694,-120}, {358,3142,-140}, {358,3146,-140}, {358,4034,-30}, 
    {358,4037,-30}, {358,4053,-30}, {395,2,-50}, {395,7,-50}, 
    {395,18,-50}, {395,29,-50}, {395,35,-50}, {395,45,-50}, 
    {395,50,-50}, {395,572,-40}, {395,660,-50}, {395,670,-50}, 
    {395,680,-50}, {395,688,-70}, {395,689,-70}, {395,694,-70}, 
    {395,1411,-40}, {395,2998,-40}, {398,2,-50}, {398,7,-50}, 
    {398,18,-50}, {398,29,-50}, {398,35,-50}, {398,45,-50}, 
    {398,50,-50}, {398,572,-40}, {398,660,-50}, {398,670,-50}, 
    {398,680,-50}, {398,688,-70}, {398,689,-70}, {398,694,-70}, 
    {398,1411,-40}, {398,2998,-40}, {406,2,-50}, {406,7,-50}, 
    {406,18,-50}, {406,29,-50}, {406,35,-50}, {406,45,-50}, 
    {406,50,-50}, {406,572,-40}, {406,660,-50}, {406,670,-50}, 
    {406,680,-50}, {406,688,-70}, {406,689,-70}, {406,694,-70}, 
    {406,1411,-40}, {406,2998,-40}, {416,2,-50}, {416,7,-50}, 
    {416,18,-50}, {416,29,-50}, {416,35,-50}, {416,45,-50}, 
    {416,50,-50}, {416,572,-40}, {416,660,-50}, {416,670,-50}, 
    {416,680,-50}, {416,688,-70}, {416,689,-70}, {416,694,-70}, 
    {416,1411,-40}, {416,2998,-40}, {421,2,-50}, {421,7,-50}, 
    {421,18,-50}, {421,29,-50}, {421,35,-50}, {421,45,-50}, 
    {421,50,-50}, {421,572,-40}, {421,660,-50}, {421,670,-50}, 
    {421,680,-50}, {421,688,-70}, {421,689,-70}, {421,694,-70}, 
    {421,1411,-40}, {421,2998,-40}, {451,2,-50}, {451,7,-50}, 
    {451,18,-50}, {451,29,-50}, {451,35,-50}, {451,45,-50}, 
    {451,50,-50}, {451,572,-40}, {451,660,-50}, {451,670,-50}, 
    {451,680,-50}, {451,688,-70}, {451,689,-70}, {451,694,-70}, 
    {451,1411,-40}, {451,2998,-40}, {457,2,-50}, {457,7,-50}, 
    {457,18,-50}, {457,29,-50}, {457,35,-50}, {457,45,-50}, 
    {457,50,-50}, {457,572,-40}, {457,660,-50}, {457,670,-50}, 
    {457,680,-50}, {457,688,-70}, {457,689,-70}, {457,694,-70}, 
    {457,1411,-40}, {457,2998,-40}, {461,2,-100}, {461,7,-100}, 
    {461,18,-100}, {461,29,-100}, {461,35,-100}, {461,45,-100}, 
    {461,50,-100}, {461,736,-30}, {461,738,-30}, {461,760,-30}, 
    {461,777,-30}, {461,1032,-30}, {461,1113,-30}, {461,1175,-30}, 
    {461,1411,-120}, {461,1632,-30}, {461,1633,-30}, {461,1647,-30}, 
    {461,1657,-30}, {461,1664,-30}, {461,2797,-40}, {461,2798,-40}, 
    {461,2812,-40}, {461,2822,-40}, {461,2829,-40}, {461,2902,-40}, 
    {461,2910,-40}, {461,2998,-120}, {476,605,-10}, {476,606,-10}, 
    {476,611,-10}, {476,617,-10}, {476,626,-10}, {476,1411,20}, 
    {476,2998,20}, {480,395,-20}, {480,398,-20}, {480,406,-20}, 
    {480,416,-20}, {480,421,-20}, {480,451,-20}, {480,457,-20}, 
    {480,572,-20}, {480,605,-20}, {480,606,-20}, {480,611,-20}, 
    {480,617,-20}, {480,626,-20}, {480,660,-50}, {480,670,-40}, 
    {480,688,-50}, {480,689,-50}, {480,694,-50}, {572,2,-90}, 
    {572,7,-90}, {572,18,-90}, {572,29,-90}, {572,35,-90}, 
    {572,45,-90}, {572,50,-90}, {572,395,-40}, {572,398,-40}, 
    {572,406,-40}, {572,416,-40}, {572,421,-40}, {572,451,-40}, 
    {572,457,-40}, {572,736,-80}, {572,738,-80}, {572,760,-80}, 
    {572,777,-80}, {572,1032,-80}, {572,1113,-80}, {572,1175,-80}, 
    {572,1404,-40}, {572,1411,-80}, {572,1632,-60}, {572,1633,-60}, 
    {572,1647,-60}, {572,1657,-60}, {572,1664,-60}, {572,2128,-120}, 
    {572,2797,-80}, {572,2798,-80}, {572,2812,-80}, {572,2822,-80}, 
    {572,2829,-80}, {572,2902,-80}, {572,2910,-80}, {572,2998,-80}, 
    {572,3151,-80}, {572,3354,-40}, {572,3819,-90}, {572,3820,-90}, 
    {572,3827,-90}, {572,3834,-90}, {572,3842,-90}, {572,3954,-60}, 
    {572,4034,-60}, {572,4037,-60}, {572,4053,-60}, {605,2,-50}, 
    {605,7,-50}, {605,18,-50}, {605,29,-50}, {605,35,-50}, 
    {605,45,-50}, {605,50,-50}, {605,1411,-30}, {605,2998,-30}, 
    {606,2,-50}, {606,7,-50}, {606,18,-50}, {606,29,-50}, {606,35,-50}, 
    {606,45,-50}, {606,50,-50}, {606,1411,-30}, {606,2998,-30}, 
    {611,2,-50}, {611,7,-50}, {611,18,-50}, {611,29,-50}, {611,35,-50}, 
    {611,45,-50}, {611,50,-50}, {611,1411,-30}, {611,2998,-30}, 
    {617,2,-50}, {617,7,-50}, {617,18,-50}, {617,29,-50}, {617,35,-50}, 
    {617,45,-50}, {617,50,-50}, {617,1411,-30}, {617,2998,-30}, 
    {626,2,-50}, {626,7,-50}, {626,18,-50}, {626,29,-50}, {626,35,-50}, 
    {626,45,-50}, {626,50,-50}, {626,1411,-30}, {626,2998,-30}, 
    {660,2,-80}, {660,7,-80}, {660,18,-80}, {660,29,-80}, {660,35,-80}, 
    {660,45,-80}, {660,50,-80}, {660,205,-50}, {660,395,-50}, 
    {660,398,-50}, {660,406,-50}, {660,416,-50}, {660,421,-50}, 
    {660,451,-50}, {660,457,-50}, {660,736,-60}, {660,738,-60}, 
    {660,760,-60}, {660,777,-60}, {660,1032,-60}, {660,1113,-60}, 
    {660,1175,-60}, {660,1404,-40}, {660,1411,-120}, {660,1632,-50}, 
    {660,1633,-50}, {660,1647,-50}, {660,1657,-50}, {660,1664,-50}, 
    {660,2128,-80}, {660,2797,-90}, {660,2798,-90}, {660,2812,-90}, 
    {660,2822,-90}, {660,2829,-90}, {660,2902,-90}, {660,2910,-90}, 
    {660,2998,-120}, {660,3354,-40}, {660,3819,-60}, {660,3820,-60}, 
    {660,3827,-60}, {660,3834,-60}, {660,3842,-60}, {670,2,-60}, 
    {670,7,-60}, {670,18,-60}, {670,29,-60}, {670,35,-60}, 
    {670,45,-60}, {670,50,-60}, {670,395,-20}, {670,398,-20}, 
    {670,406,-20}, {670,416,-20}, {670,421,-20}, {670,451,-20}, 
    {670,457,-20}, {670,736,-40}, {670,738,-40}, {670,760,-40}, 
    {670,777,-40}, {670,1032,-40}, {670,1113,-40}, {670,1175,-40}, 
    {670,1404,-10}, {670,1411,-80}, {670,1632,-35}, {670,1633,-35}, 
    {670,1647,-35}, {670,1657,-35}, {670,1664,-35}, {670,2128,-40}, 
    {670,2797,-60}, {670,2798,-60}, {670,2812,-60}, {670,2822,-60}, 
    {670,2829,-60}, {670,2902,-60}, {670,2910,-60}, {670,2998,-80}, 
    {670,3354,-10}, {670,3819,-45}, {670,3820,-45}, {670,3827,-45}, 
    {670,3834,-45}, {670,3842,-45}, {670,4034,-20}, {670,4037,-20}, 
    {670,4053,-20}, {688,2,-110}, {688,7,-110}, {688,18,-110}, 
    {688,29,-110}, {688,35,-110}, {688,45,-110}, {688,50,-110}, 
    {688,395,-70}, {688,398,-70}, {688,406,-70}, {688,416,-70}, 
    {688,421,-70}, {688,451,-70}, {688,457,-70}, {688,736,-90}, 
    {688,738,-90}, {688,760,-90}, {688,777,-90}, {688,1032,-90}, 
    {688,1113,-90}, {688,1175,-90}, {688,1404,-50}, {688,1411,-100}, 
    {688,1632,-80}, {688,1633,-80}, {688,1647,-80}, {688,1657,-80}, 
    {688,1664,-80}, {688,2797,-100}, {688,2798,-100}, {688,2812,-100}, 
    {688,2822,-100}, {688,2829,-100}, {688,2902,-100}, {688,2910,-100}, 
    {688,2998,-100}, {688,3354,-50}, {688,3819,-100}, {688,3820,-100}, 
    {688,3827,-100}, {688,3834,-100}, {688,3842,-100}, {689,2,-110}, 
    {689,7,-110}, {689,18,-110}, {689,29,-110}, {689,35,-110}, 
    {689,45,-110}, {689,50,-110}, {689,395,-70}, {689,398,-70}, 
    {689,406,-70}, {689,416,-70}, {689,421,-70}, {689,451,-70}, 
    {689,457,-70}, {689,736,-90}, {689,738,-90}, {689,760,-90}, 
    {689,777,-90}, {689,1032,-90}, {689,1113,-90}, {689,1175,-90}, 
    {689,1404,-50}, {689,1411,-100}, {689,1632,-80}, {689,1633,-80}, 
    {689,1647,-80}, {689,1657,-80}, {689,1664,-80}, {689,2797,-100}, 
    {689,2798,-100}, {689,2812,-100}, {689,2822,-100}, {689,2829,-100}, 
    {689,2902,-100}, {689,2910,-100}, {689,2998,-100}, {689,3354,-50}, 
    {689,3819,-100}, {689,3820,-100}, {689,3827,-100}, {689,3834,-100}, 
    {689,3842,-100}, {694,2,-110}, {694,7,-110}, {694,18,-110}, 
    {694,29,-110}, {694,35,-110}, {694,45,-110}, {694,50,-110}, 
    {694,395,-70}, {694,398,-70}, {694,406,-70}, {694,416,-70}, 
    {694,421,-70}, {694,451,-70}, {694,457,-70}, {694,736,-90}, 
    {694,738,-90}, {694,760,-90}, {694,777,-90}, {694,1032,-90}, 
    {694,1113,-90}, {694,1175,-90}, {694,1404,-50}, {694,1411,-100}, 
    {694,1632,-80}, {694,1633,-80}, {694,1647,-80}, {694,1657,-80}, 
    {694,1664,-80}, {694,2797,-100}, {694,2798,-100}, {694,2812,-100}, 
    {694,2822,-100}, {694,2829,-100}, {694,2902,-100}, {694,2910,-100}, 
    {694,2998,-100}, {694,3354,-50}, {694,3819,-100}, {694,3820,-100}, 
    {694,3827,-100}, {694,3834,-100}, {694,3842,-100}, {736,1888,-10}, 
    {736,3905,-15}, {736,3954,-15}, {736,4034,-20}, {736,4037,-20}, 
    {736,4053,-20}, {738,1888,-10}, {738,3905,-15}, {738,3954,-15}, 
    {738,4034,-20}, {738,4037,-20}, {738,4053,-20}, {760,1888,-10}, 
    {760,3905,-15}, {760,3954,-15}, {760,4034,-20}, {760,4037,-20}, 
    {760,4053,-20}, {777,1888,-10}, {777,3905,-15}, {777,3954,-15}, 
    {777,4034,-20}, {777,4037,-20}, {777,4053,-20}, {1032,1888,-10}, 
    {1032,3905,-15}, {1032,3954,-15}, {1032,4034,-20}, {1032,4037,-20}, 
    {1032,4053,-20}, {1113,1888,-10}, {1113,3905,-15}, {1113,3954,-15}, 
    {1113,4034,-20}, {1113,4037,-20}, {1113,4053,-20}, {1175,1888,-10}, 
    {1175,3905,-15}, {1175,3954,-15}, {1175,4034,-20}, {1175,4037,-20}, 
    {1175,4053,-20}, {1194,2418,-10}, {1194,2503,-10}, {1194,3819,-20}, 
    {1194,3820,-20}, {1194,3827,-20}, {1194,3834,-20}, {1194,3842,-20}, 
    {1194,3905,-20}, {1194,4034,-20}, {1194,4037,-20}, {1194,4053,-20}, 
    {1314,1990,-10}, {1314,2304,-20}, {1314,2418,-20}, {1314,2503,-20}, 
    {1314,4034,-10}, {1314,4037,-10}, {1314,4053,-10}, {1334,1990,-10}, 
    {1334,2304,-20}, {1334,2418,-20}, {1334,2503,-20}, {1334,4034,-10}, 
    {1334,4037,-10}, {1334,4053,-10}, {1404,3501,-40}, 
    {1411,3142,-120}, {1411,3146,-120}, {1411,3501,-40}, 
    {1483,1483,-10}, {1483,3905,-15}, {1483,3954,-15}, {1483,4034,-15}, 
    {1483,4037,-15}, {1483,4053,-15}, {1632,1411,10}, {1632,2998,20}, 
    {1632,3905,-15}, {1632,3954,-15}, {1632,4023,-15}, {1632,4034,-15}, 
    {1632,4037,-15}, {1632,4053,-15}, {1633,1411,10}, {1633,2998,20}, 
    {1633,3905,-15}, {1633,3954,-15}, {1633,4023,-15}, {1633,4034,-15}, 
    {1633,4037,-15}, {1633,4053,-15}, {1647,1411,10}, {1647,2998,20}, 
    {1647,3905,-15}, {1647,3954,-15}, {1647,4023,-15}, {1647,4034,-15}, 
    {1647,4037,-15}, {1647,4053,-15}, {1657,1411,10}, {1657,2998,20}, 
    {1657,3905,-15}, {1657,3954,-15}, {1657,4023,-15}, {1657,4034,-15}, 
    {1657,4037,-15}, {1657,4053,-15}, {1664,1411,10}, {1664,2998,20}, 
    {1664,3905,-15}, {1664,3954,-15}, {1664,4023,-15}, {1664,4034,-15}, 
    {1664,4037,-15}, {1664,4053,-15}, {1789,1411,-10}, {1789,1632,-10}, 
    {1789,1633,-10}, {1789,1647,-10}, {1789,1657,-10}, {1789,1664,-10}, 
    {1789,2797,-20}, {1789,2798,-20}, {1789,2812,-20}, {1789,2822,-20}, 
    {1789,2829,-20}, {1789,2902,-20}, {1789,2910,-20}, {1789,2998,-10}, 
    {1789,3142,30}, {1789,3146,30}, {1888,1632,10}, {1888,1633,10}, 
    {1888,1647,10}, {1888,1657,10}, {1888,1664,10}, {1888,1888,-10}, 
    {1990,4034,-20}, {1990,4037,-20}, {1990,4053,-20}, {2304,2797,-15}, 
    {2304,2798,-15}, {2304,2812,-15}, {2304,2822,-15}, {2304,2829,-15}, 
    {2304,2902,-15}, {2304,2910,-15}, {2418,3954,-15}, {2418,4034,-15}, 
    {2418,4037,-15}, {2418,4053,-15}, {2503,3954,-15}, {2503,4034,-15}, 
    {2503,4037,-15}, {2503,4053,-15}, {2513,3819,-20}, {2513,3820,-20}, 
    {2513,3827,-20}, {2513,3834,-20}, {2513,3842,-20}, {2513,4034,-30}, 
    {2513,4037,-30}, {2513,4053,-30}, {2650,3819,-10}, {2650,3820,-10}, 
    {2650,3827,-10}, {2650,3834,-10}, {2650,3842,-10}, {2650,3905,-40}, 
    {2650,4034,-20}, {2650,4037,-20}, {2650,4053,-20}, {2772,3819,-10}, 
    {2772,3820,-10}, {2772,3827,-10}, {2772,3834,-10}, {2772,3842,-10}, 
    {2772,3905,-40}, {2772,4034,-20}, {2772,4037,-20}, {2772,4053,-20}, 
    {2797,3905,-20}, {2797,3954,-15}, {2797,4023,-30}, {2797,4034,-20}, 
    {2797,4037,-20}, {2797,4053,-20}, {2798,3905,-20}, {2798,3954,-15}, 
    {2798,4023,-30}, {2798,4034,-20}, {2798,4037,-20}, {2798,4053,-20}, 
    {2812,3905,-20}, {2812,3954,-15}, {2812,4023,-30}, {2812,4034,-20}, 
    {2812,4037,-20}, {2812,4053,-20}, {2822,3905,-20}, {2822,3954,-15}, 
    {2822,4023,-30}, {2822,4034,-20}, {2822,4037,-20}, {2822,4053,-20}, 
    {2829,3905,-20}, {2829,3954,-15}, {2829,4023,-30}, {2829,4034,-20}, 
    {2829,4037,-20}, {2829,4053,-20}, {2902,3905,-20}, {2902,3954,-15}, 
    {2902,4023,-30}, {2902,4034,-20}, {2902,4037,-20}, {2902,4053,-20}, 
    {2910,3905,-20}, {2910,3954,-15}, {2910,4023,-30}, {2910,4034,-20}, 
    {2910,4037,-20}, {2910,4053,-20}, {2924,4034,-15}, {2924,4037,-15}, 
    {2924,4053,-15}, {2998,3142,-120}, {2998,3146,-120}, 
    {2998,3501,-40}, {3142,3501,-80}, {3143,3143,-46}, {3146,1483,-80}, 
    {3146,2418,-20}, {3146,2503,-20}, {3146,3146,-46}, {3146,3151,-40}, 
    {3146,3282,-60}, {3146,3320,-60}, {3146,3501,-80}, {3146,3905,-20}, 
    {3151,1314,-20}, {3151,1334,-20}, {3151,1411,-60}, {3151,1483,-20}, 
    {3151,1888,-15}, {3151,2128,-20}, {3151,2797,-20}, {3151,2798,-20}, 
    {3151,2812,-20}, {3151,2822,-20}, {3151,2829,-20}, {3151,2902,-20}, 
    {3151,2910,-20}, {3151,2998,-60}, {3151,3086,-20}, {3151,3282,-15}, 
    {3151,3320,-15}, {3151,3559,20}, {3151,3905,10}, {3151,4034,10}, 
    {3151,4037,10}, {3151,4053,10}, {3282,3954,-15}, {3320,3954,-15}, 
    {3354,3501,-40}, {3501,572,-100}, {3501,660,-80}, {3501,670,-80}, 
    {3501,688,-120}, {3501,689,-120}, {3501,694,-120}, {3501,3138,-80}, 
    {3501,3143,-60}, {3905,736,-20}, {3905,738,-20}, {3905,760,-20}, 
    {3905,777,-20}, {3905,1032,-20}, {3905,1113,-20}, {3905,1175,-20}, 
    {3905,1411,-80}, {3905,2797,-30}, {3905,2798,-30}, {3905,2812,-30}, 
    {3905,2822,-30}, {3905,2829,-30}, {3905,2902,-30}, {3905,2910,-30}, 
    {3905,2998,-80}, {3954,1411,-40}, {3954,2797,-20}, {3954,2798,-20}, 
    {3954,2812,-20}, {3954,2822,-20}, {3954,2829,-20}, {3954,2902,-20}, 
    {3954,2910,-20}, {3954,2998,-40}, {4023,1632,-10}, {4023,1633,-10}, 
    {4023,1647,-10}, {4023,1657,-10}, {4023,1664,-10}, {4034,736,-30}, 
    {4034,738,-30}, {4034,760,-30}, {4034,777,-30}, {4034,1032,-30}, 
    {4034,1113,-30}, {4034,1175,-30}, {4034,1411,-80}, {4034,1632,-10}, 
    {4034,1633,-10}, {4034,1647,-10}, {4034,1657,-10}, {4034,1664,-10}, 
    {4034,2797,-25}, {4034,2798,-25}, {4034,2812,-25}, {4034,2822,-25}, 
    {4034,2829,-25}, {4034,2902,-25}, {4034,2910,-25}, {4034,2998,-80}, 
    {4037,736,-30}, {4037,738,-30}, {4037,760,-30}, {4037,777,-30}, 
    {4037,1032,-30}, {4037,1113,-30}, {4037,1175,-30}, {4037,1411,-80}, 
    {4037,1632,-10}, {4037,1633,-10}, {4037,1647,-10}, {4037,1657,-10}, 
    {4037,1664,-10}, {4037,2797,-25}, {4037,2798,-25}, {4037,2812,-25}, 
    {4037,2822,-25}, {4037,2829,-25}, {4037,2902,-25}, {4037,2910,-25}, 
    {4037,2998,-80}, {4053,736,-30}, {4053,738,-30}, {4053,760,-30}, 
    {4053,777,-30}, {4053,1032,-30}, {4053,1113,-30}, {4053,1175,-30}, 
    {4053,1411,-80}, {4053,1632,-10}, {4053,1633,-10}, {4053,1647,-10}, 
    {4053,1657,-10}, {4053,1664,-10}, {4053,2797,-25}, {4053,2798,-25}, 
    {4053,2812,-25}, {4053,2822,-25}, {4053,2829,-25}, {4053,2902,-25}, 
    {4053,2910,-25}, {4053,2998,-80}, {4135,1632,10}, {4135,1633,10}, 
    {4135,1647,10}, {4135,1657,10}, {4135,1664,10}, {4156,1632,10}, 
    {4156,1633,10}, {4156,1647,10}, {4156,1657,10}, {4156,1664,10}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature helvetica_bold_ligs[] = {
    {1789,2134,1809}, {1789,2418,1852}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair helvetica_boldoblique_kerns[] = {
    {2,66,-40}, {2,72,-40}, {2,205,-50}, {2,395,-40}, {2,398,-40}, 
    {2,406,-40}, {2,416,-40}, {2,421,-40}, {2,451,-40}, {2,457,-40}, 
    {2,476,-40}, {2,572,-90}, {2,605,-50}, {2,606,-50}, {2,611,-50}, 
    {2,617,-50}, {2,626,-50}, {2,660,-80}, {2,670,-60}, {2,688,-110}, 
    {2,689,-110}, {2,694,-110}, {2,3819,-30}, {2,3820,-30}, 
    {2,3827,-30}, {2,3834,-30}, {2,3842,-30}, {2,3905,-40}, 
    {2,3954,-30}, {2,4034,-30}, {2,4037,-30}, {2,4053,-30}, {7,66,-40}, 
    {7,72,-40}, {7,205,-50}, {7,395,-40}, {7,398,-40}, {7,406,-40}, 
    {7,416,-40}, {7,421,-40}, {7,451,-40}, {7,457,-40}, {7,476,-40}, 
    {7,572,-90}, {7,605,-50}, {7,606,-50}, {7,611,-50}, {7,617,-50}, 
    {7,626,-50}, {7,660,-80}, {7,670,-60}, {7,688,-110}, {7,689,-110}, 
    {7,694,-110}, {7,3819,-30}, {7,3820,-30}, {7,3827,-30}, 
    {7,3834,-30}, {7,3842,-30}, {7,3905,-40}, {7,3954,-30}, 
    {7,4034,-30}, {7,4037,-30}, {7,4053,-30}, {18,66,-40}, {18,72,-40}, 
    {18,205,-50}, {18,395,-40}, {18,398,-40}, {18,406,-40}, 
    {18,416,-40}, {18,421,-40}, {18,451,-40}, {18,457,-40}, 
    {18,476,-40}, {18,572,-90}, {18,605,-50}, {18,606,-50}, 
    {18,611,-50}, {18,617,-50}, {18,626,-50}, {18,660,-80}, 
    {18,670,-60}, {18,688,-110}, {18,689,-110}, {18,694,-110}, 
    {18,3819,-30}, {18,3820,-30}, {18,3827,-30}, {18,3834,-30}, 
    {18,3842,-30}, {18,3905,-40}, {18,3954,-30}, {18,4034,-30}, 
    {18,4037,-30}, {18,4053,-30}, {29,66,-40}, {29,72,-40}, 
    {29,205,-50}, {29,395,-40}, {29,398,-40}, {29,406,-40}, 
    {29,416,-40}, {29,421,-40}, {29,451,-40}, {29,457,-40}, 
    {29,476,-40}, {29,572,-90}, {29,605,-50}, {29,606,-50}, 
    {29,611,-50}, {29,617,-50}, {29,626,-50}, {29,660,-80}, 
    {29,670,-60}, {29,688,-110}, {29,689,-110}, {29,694,-110}, 
    {29,3819,-30}, {29,3820,-30}, {29,3827,-30}, {29,3834,-30}, 
    {29,3842,-30}, {29,3905,-40}, {29,3954,-30}, {29,4034,-30}, 
    {29,4037,-30}, {29,4053,-30}, {35,66,-40}, {35,72,-40}, 
    {35,205,-50}, {35,395,-40}, {35,398,-40}, {35,406,-40}, 
    {35,416,-40}, {35,421,-40}, {35,451,-40}, {35,457,-40}, 
    {35,476,-40}, {35,572,-90}, {35,605,-50}, {35,606,-50}, 
    {35,611,-50}, {35,617,-50}, {35,626,-50}, {35,660,-80}, 
    {35,670,-60}, {35,688,-110}, {35,689,-110}, {35,694,-110}, 
    {35,3819,-30}, {35,3820,-30}, {35,3827,-30}, {35,3834,-30}, 
    {35,3842,-30}, {35,3905,-40}, {35,3954,-30}, {35,4034,-30}, 
    {35,4037,-30}, {35,4053,-30}, {45,66,-40}, {45,72,-40}, 
    {45,205,-50}, {45,395,-40}, {45,398,-40}, {45,406,-40}, 
    {45,416,-40}, {45,421,-40}, {45,451,-40}, {45,457,-40}, 
    {45,476,-40}, {45,572,-90}, {45,605,-50}, {45,606,-50}, 
    {45,611,-50}, {45,617,-50}, {45,626,-50}, {45,660,-80}, 
    {45,670,-60}, {45,688,-110}, {45,689,-110}, {45,694,-110}, 
    {45,3819,-30}, {45,3820,-30}, {45,3827,-30}, {45,3834,-30}, 
    {45,3842,-30}, {45,3905,-40}, {45,3954,-30}, {45,4034,-30}, 
    {45,4037,-30}, {45,4053,-30}, {50,66,-40}, {50,72,-40}, 
    {50,205,-50}, {50,395,-40}, {50,398,-40}, {50,406,-40}, 
    {50,416,-40}, {50,421,-40}, {50,451,-40}, {50,457,-40}, 
    {50,476,-40}, {50,572,-90}, {50,605,-50}, {50,606,-50}, 
    {50,611,-50}, {50,617,-50}, {50,626,-50}, {50,660,-80}, 
    {50,670,-60}, {50,688,-110}, {50,689,-110}, {50,694,-110}, 
    {50,3819,-30}, {50,3820,-30}, {50,3827,-30}, {50,3834,-30}, 
    {50,3842,-30}, {50,3905,-40}, {50,3954,-30}, {50,4034,-30}, 
    {50,4037,-30}, {50,4053,-30}, {53,2,-30}, {53,7,-30}, {53,18,-30}, 
    {53,29,-30}, {53,35,-30}, {53,45,-30}, {53,50,-30}, {53,605,-10}, 
    {53,606,-10}, {53,611,-10}, {53,617,-10}, {53,626,-10}, {95,2,-40}, 
    {95,7,-40}, {95,18,-40}, {95,29,-40}, {95,35,-40}, {95,45,-40}, 
    {95,50,-40}, {95,660,-40}, {95,670,-40}, {95,688,-70}, 
    {95,689,-70}, {95,694,-70}, {95,1411,-30}, {95,2998,-30}, 
    {194,2,-80}, {194,7,-80}, {194,18,-80}, {194,29,-80}, {194,35,-80}, 
    {194,45,-80}, {194,50,-80}, {194,736,-20}, {194,738,-20}, 
    {194,760,-20}, {194,777,-20}, {194,1032,-20}, {194,1113,-20}, 
    {194,1175,-20}, {194,1411,-100}, {194,2998,-100}, {303,2,-20}, 
    {303,7,-20}, {303,18,-20}, {303,29,-20}, {303,35,-20}, 
    {303,45,-20}, {303,50,-20}, {303,1411,-20}, {303,2998,-20}, 
    {303,3819,-20}, {303,3820,-20}, {303,3827,-20}, {303,3834,-20}, 
    {303,3842,-20}, {311,395,-30}, {311,398,-30}, {311,406,-30}, 
    {311,416,-30}, {311,421,-30}, {311,451,-30}, {311,457,-30}, 
    {311,1632,-15}, {311,1633,-15}, {311,1647,-15}, {311,1657,-15}, 
    {311,1664,-15}, {311,2797,-35}, {311,2798,-35}, {311,2812,-35}, 
    {311,2822,-35}, {311,2829,-35}, {311,2902,-35}, {311,2910,-35}, 
    {311,3819,-30}, {311,3820,-30}, {311,3827,-30}, {311,3834,-30}, 
    {311,3842,-30}, {311,4034,-40}, {311,4037,-40}, {311,4053,-40}, 
    {339,572,-90}, {339,660,-110}, {339,670,-80}, {339,688,-120}, 
    {339,689,-120}, {339,694,-120}, {339,3142,-140}, {339,3146,-140}, 
    {339,4034,-30}, {339,4037,-30}, {339,4053,-30}, {358,572,-90}, 
    {358,660,-110}, {358,670,-80}, {358,688,-120}, {358,689,-120}, 
    {358,694,-120}, {358,3142,-140}, {358,3146,-140}, {358,4034,-30}, 
    {358,4037,-30}, {358,4053,-30}, {395,2,-50}, {395,7,-50}, 
    {395,18,-50}, {395,29,-50}, {395,35,-50}, {395,45,-50}, 
    {395,50,-50}, {395,572,-40}, {395,660,-50}, {395,670,-50}, 
    {395,680,-50}, {395,688,-70}, {395,689,-70}, {395,694,-70}, 
    {395,1411,-40}, {395,2998,-40}, {398,2,-50}, {398,7,-50}, 
    {398,18,-50}, {398,29,-50}, {398,35,-50}, {398,45,-50}, 
    {398,50,-50}, {398,572,-40}, {398,660,-50}, {398,670,-50}, 
    {398,680,-50}, {398,688,-70}, {398,689,-70}, {398,694,-70}, 
    {398,1411,-40}, {398,2998,-40}, {406,2,-50}, {406,7,-50}, 
    {406,18,-50}, {406,29,-50}, {406,35,-50}, {406,45,-50}, 
    {406,50,-50}, {406,572,-40}, {406,660,-50}, {406,670,-50}, 
    {406,680,-50}, {406,688,-70}, {406,689,-70}, {406,694,-70}, 
    {406,1411,-40}, {406,2998,-40}, {416,2,-50}, {416,7,-50}, 
    {416,18,-50}, {416,29,-50}, {416,35,-50}, {416,45,-50}, 
    {416,50,-50}, {416,572,-40}, {416,660,-50}, {416,670,-50}, 
    {416,680,-50}, {416,688,-70}, {416,689,-70}, {416,694,-70}, 
    {416,1411,-40}, {416,2998,-40}, {421,2,-50}, {421,7,-50}, 
    {421,18,-50}, {421,29,-50}, {421,35,-50}, {421,45,-50}, 
    {421,50,-50}, {421,572,-40}, {421,660,-50}, {421,670,-50}, 
    {421,680,-50}, {421,688,-70}, {421,689,-70}, {421,694,-70}, 
    {421,1411,-40}, {421,2998,-40}, {451,2,-50}, {451,7,-50}, 
    {451,18,-50}, {451,29,-50}, {451,35,-50}, {451,45,-50}, 
    {451,50,-50}, {451,572,-40}, {451,660,-50}, {451,670,-50}, 
    {451,680,-50}, {451,688,-70}, {451,689,-70}, {451,694,-70}, 
    {451,1411,-40}, {451,2998,-40}, {457,2,-50}, {457,7,-50}, 
    {457,18,-50}, {457,29,-50}, {457,35,-50}, {457,45,-50}, 
    {457,50,-50}, {457,572,-40}, {457,660,-50}, {457,670,-50}, 
    {457,680,-50}, {457,688,-70}, {457,689,-70}, {457,694,-70}, 
    {457,1411,-40}, {457,2998,-40}, {461,2,-100}, {461,7,-100}, 
    {461,18,-100}, {461,29,-100}, {461,35,-100}, {461,45,-100}, 
    {461,50,-100}, {461,736,-30}, {461,738,-30}, {461,760,-30}, 
    {461,777,-30}, {461,1032,-30}, {461,1113,-30}, {461,1175,-30}, 
    {461,1411,-120}, {461,1632,-30}, {461,1633,-30}, {461,1647,-30}, 
    {461,1657,-30}, {461,1664,-30}, {461,2797,-40}, {461,2798,-40}, 
    {461,2812,-40}, {461,2822,-40}, {461,2829,-40}, {461,2902,-40}, 
    {461,2910,-40}, {461,2998,-120}, {476,605,-10}, {476,606,-10}, 
    {476,611,-10}, {476,617,-10}, {476,626,-10}, {476,1411,20}, 
    {476,2998,20}, {480,395,-20}, {480,398,-20}, {480,406,-20}, 
    {480,416,-20}, {480,421,-20}, {480,451,-20}, {480,457,-20}, 
    {480,572,-20}, {480,605,-20}, {480,606,-20}, {480,611,-20}, 
    {480,617,-20}, {480,626,-20}, {480,660,-50}, {480,670,-40}, 
    {480,688,-50}, {480,689,-50}, {480,694,-50}, {572,2,-90}, 
    {572,7,-90}, {572,18,-90}, {572,29,-90}, {572,35,-90}, 
    {572,45,-90}, {572,50,-90}, {572,395,-40}, {572,398,-40}, 
    {572,406,-40}, {572,416,-40}, {572,421,-40}, {572,451,-40}, 
    {572,457,-40}, {572,736,-80}, {572,738,-80}, {572,760,-80}, 
    {572,777,-80}, {572,1032,-80}, {572,1113,-80}, {572,1175,-80}, 
    {572,1404,-40}, {572,1411,-80}, {572,1632,-60}, {572,1633,-60}, 
    {572,1647,-60}, {572,1657,-60}, {572,1664,-60}, {572,2128,-120}, 
    {572,2797,-80}, {572,2798,-80}, {572,2812,-80}, {572,2822,-80}, 
    {572,2829,-80}, {572,2902,-80}, {572,2910,-80}, {572,2998,-80}, 
    {572,3151,-80}, {572,3354,-40}, {572,3819,-90}, {572,3820,-90}, 
    {572,3827,-90}, {572,3834,-90}, {572,3842,-90}, {572,3954,-60}, 
    {572,4034,-60}, {572,4037,-60}, {572,4053,-60}, {605,2,-50}, 
    {605,7,-50}, {605,18,-50}, {605,29,-50}, {605,35,-50}, 
    {605,45,-50}, {605,50,-50}, {605,1411,-30}, {605,2998,-30}, 
    {606,2,-50}, {606,7,-50}, {606,18,-50}, {606,29,-50}, {606,35,-50}, 
    {606,45,-50}, {606,50,-50}, {606,1411,-30}, {606,2998,-30}, 
    {611,2,-50}, {611,7,-50}, {611,18,-50}, {611,29,-50}, {611,35,-50}, 
    {611,45,-50}, {611,50,-50}, {611,1411,-30}, {611,2998,-30}, 
    {617,2,-50}, {617,7,-50}, {617,18,-50}, {617,29,-50}, {617,35,-50}, 
    {617,45,-50}, {617,50,-50}, {617,1411,-30}, {617,2998,-30}, 
    {626,2,-50}, {626,7,-50}, {626,18,-50}, {626,29,-50}, {626,35,-50}, 
    {626,45,-50}, {626,50,-50}, {626,1411,-30}, {626,2998,-30}, 
    {660,2,-80}, {660,7,-80}, {660,18,-80}, {660,29,-80}, {660,35,-80}, 
    {660,45,-80}, {660,50,-80}, {660,205,-50}, {660,395,-50}, 
    {660,398,-50}, {660,406,-50}, {660,416,-50}, {660,421,-50}, 
    {660,451,-50}, {660,457,-50}, {660,736,-60}, {660,738,-60}, 
    {660,760,-60}, {660,777,-60}, {660,1032,-60}, {660,1113,-60}, 
    {660,1175,-60}, {660,1404,-40}, {660,1411,-120}, {660,1632,-50}, 
    {660,1633,-50}, {660,1647,-50}, {660,1657,-50}, {660,1664,-50}, 
    {660,2128,-80}, {660,2797,-90}, {660,2798,-90}, {660,2812,-90}, 
    {660,2822,-90}, {660,2829,-90}, {660,2902,-90}, {660,2910,-90}, 
    {660,2998,-120}, {660,3354,-40}, {660,3819,-60}, {660,3820,-60}, 
    {660,3827,-60}, {660,3834,-60}, {660,3842,-60}, {670,2,-60}, 
    {670,7,-60}, {670,18,-60}, {670,29,-60}, {670,35,-60}, 
    {670,45,-60}, {670,50,-60}, {670,395,-20}, {670,398,-20}, 
    {670,406,-20}, {670,416,-20}, {670,421,-20}, {670,451,-20}, 
    {670,457,-20}, {670,736,-40}, {670,738,-40}, {670,760,-40}, 
    {670,777,-40}, {670,1032,-40}, {670,1113,-40}, {670,1175,-40}, 
    {670,1404,-10}, {670,1411,-80}, {670,1632,-35}, {670,1633,-35}, 
    {670,1647,-35}, {670,1657,-35}, {670,1664,-35}, {670,2128,-40}, 
    {670,2797,-60}, {670,2798,-60}, {670,2812,-60}, {670,2822,-60}, 
    {670,2829,-60}, {670,2902,-60}, {670,2910,-60}, {670,2998,-80}, 
    {670,3354,-10}, {670,3819,-45}, {670,3820,-45}, {670,3827,-45}, 
    {670,3834,-45}, {670,3842,-45}, {670,4034,-20}, {670,4037,-20}, 
    {670,4053,-20}, {688,2,-110}, {688,7,-110}, {688,18,-110}, 
    {688,29,-110}, {688,35,-110}, {688,45,-110}, {688,50,-110}, 
    {688,395,-70}, {688,398,-70}, {688,406,-70}, {688,416,-70}, 
    {688,421,-70}, {688,451,-70}, {688,457,-70}, {688,736,-90}, 
    {688,738,-90}, {688,760,-90}, {688,777,-90}, {688,1032,-90}, 
    {688,1113,-90}, {688,1175,-90}, {688,1404,-50}, {688,1411,-100}, 
    {688,1632,-80}, {688,1633,-80}, {688,1647,-80}, {688,1657,-80}, 
    {688,1664,-80}, {688,2797,-100}, {688,2798,-100}, {688,2812,-100}, 
    {688,2822,-100}, {688,2829,-100}, {688,2902,-100}, {688,2910,-100}, 
    {688,2998,-100}, {688,3354,-50}, {688,3819,-100}, {688,3820,-100}, 
    {688,3827,-100}, {688,3834,-100}, {688,3842,-100}, {689,2,-110}, 
    {689,7,-110}, {689,18,-110}, {689,29,-110}, {689,35,-110}, 
    {689,45,-110}, {689,50,-110}, {689,395,-70}, {689,398,-70}, 
    {689,406,-70}, {689,416,-70}, {689,421,-70}, {689,451,-70}, 
    {689,457,-70}, {689,736,-90}, {689,738,-90}, {689,760,-90}, 
    {689,777,-90}, {689,1032,-90}, {689,1113,-90}, {689,1175,-90}, 
    {689,1404,-50}, {689,1411,-100}, {689,1632,-80}, {689,1633,-80}, 
    {689,1647,-80}, {689,1657,-80}, {689,1664,-80}, {689,2797,-100}, 
    {689,2798,-100}, {689,2812,-100}, {689,2822,-100}, {689,2829,-100}, 
    {689,2902,-100}, {689,2910,-100}, {689,2998,-100}, {689,3354,-50}, 
    {689,3819,-100}, {689,3820,-100}, {689,3827,-100}, {689,3834,-100}, 
    {689,3842,-100}, {694,2,-110}, {694,7,-110}, {694,18,-110}, 
    {694,29,-110}, {694,35,-110}, {694,45,-110}, {694,50,-110}, 
    {694,395,-70}, {694,398,-70}, {694,406,-70}, {694,416,-70}, 
    {694,421,-70}, {694,451,-70}, {694,457,-70}, {694,736,-90}, 
    {694,738,-90}, {694,760,-90}, {694,777,-90}, {694,1032,-90}, 
    {694,1113,-90}, {694,1175,-90}, {694,1404,-50}, {694,1411,-100}, 
    {694,1632,-80}, {694,1633,-80}, {694,1647,-80}, {694,1657,-80}, 
    {694,1664,-80}, {694,2797,-100}, {694,2798,-100}, {694,2812,-100}, 
    {694,2822,-100}, {694,2829,-100}, {694,2902,-100}, {694,2910,-100}, 
    {694,2998,-100}, {694,3354,-50}, {694,3819,-100}, {694,3820,-100}, 
    {694,3827,-100}, {694,3834,-100}, {694,3842,-100}, {736,1888,-10}, 
    {736,3905,-15}, {736,3954,-15}, {736,4034,-20}, {736,4037,-20}, 
    {736,4053,-20}, {738,1888,-10}, {738,3905,-15}, {738,3954,-15}, 
    {738,4034,-20}, {738,4037,-20}, {738,4053,-20}, {760,1888,-10}, 
    {760,3905,-15}, {760,3954,-15}, {760,4034,-20}, {760,4037,-20}, 
    {760,4053,-20}, {777,1888,-10}, {777,3905,-15}, {777,3954,-15}, 
    {777,4034,-20}, {777,4037,-20}, {777,4053,-20}, {1032,1888,-10}, 
    {1032,3905,-15}, {1032,3954,-15}, {1032,4034,-20}, {1032,4037,-20}, 
    {1032,4053,-20}, {1113,1888,-10}, {1113,3905,-15}, {1113,3954,-15}, 
    {1113,4034,-20}, {1113,4037,-20}, {1113,4053,-20}, {1175,1888,-10}, 
    {1175,3905,-15}, {1175,3954,-15}, {1175,4034,-20}, {1175,4037,-20}, 
    {1175,4053,-20}, {1194,2418,-10}, {1194,2503,-10}, {1194,3819,-20}, 
    {1194,3820,-20}, {1194,3827,-20}, {1194,3834,-20}, {1194,3842,-20}, 
    {1194,3905,-20}, {1194,4034,-20}, {1194,4037,-20}, {1194,4053,-20}, 
    {1314,1990,-10}, {1314,2304,-20}, {1314,2418,-20}, {1314,2503,-20}, 
    {1314,4034,-10}, {1314,4037,-10}, {1314,4053,-10}, {1334,1990,-10}, 
    {1334,2304,-20}, {1334,2418,-20}, {1334,2503,-20}, {1334,4034,-10}, 
    {1334,4037,-10}, {1334,4053,-10}, {1404,3501,-40}, 
    {1411,3142,-120}, {1411,3146,-120}, {1411,3501,-40}, 
    {1483,1483,-10}, {1483,3905,-15}, {1483,3954,-15}, {1483,4034,-15}, 
    {1483,4037,-15}, {1483,4053,-15}, {1632,1411,10}, {1632,2998,20}, 
    {1632,3905,-15}, {1632,3954,-15}, {1632,4023,-15}, {1632,4034,-15}, 
    {1632,4037,-15}, {1632,4053,-15}, {1633,1411,10}, {1633,2998,20}, 
    {1633,3905,-15}, {1633,3954,-15}, {1633,4023,-15}, {1633,4034,-15}, 
    {1633,4037,-15}, {1633,4053,-15}, {1647,1411,10}, {1647,2998,20}, 
    {1647,3905,-15}, {1647,3954,-15}, {1647,4023,-15}, {1647,4034,-15}, 
    {1647,4037,-15}, {1647,4053,-15}, {1657,1411,10}, {1657,2998,20}, 
    {1657,3905,-15}, {1657,3954,-15}, {1657,4023,-15}, {1657,4034,-15}, 
    {1657,4037,-15}, {1657,4053,-15}, {1664,1411,10}, {1664,2998,20}, 
    {1664,3905,-15}, {1664,3954,-15}, {1664,4023,-15}, {1664,4034,-15}, 
    {1664,4037,-15}, {1664,4053,-15}, {1789,1411,-10}, {1789,1632,-10}, 
    {1789,1633,-10}, {1789,1647,-10}, {1789,1657,-10}, {1789,1664,-10}, 
    {1789,2797,-20}, {1789,2798,-20}, {1789,2812,-20}, {1789,2822,-20}, 
    {1789,2829,-20}, {1789,2902,-20}, {1789,2910,-20}, {1789,2998,-10}, 
    {1789,3142,30}, {1789,3146,30}, {1888,1632,10}, {1888,1633,10}, 
    {1888,1647,10}, {1888,1657,10}, {1888,1664,10}, {1888,1888,-10}, 
    {1990,4034,-20}, {1990,4037,-20}, {1990,4053,-20}, {2304,2797,-15}, 
    {2304,2798,-15}, {2304,2812,-15}, {2304,2822,-15}, {2304,2829,-15}, 
    {2304,2902,-15}, {2304,2910,-15}, {2418,3954,-15}, {2418,4034,-15}, 
    {2418,4037,-15}, {2418,4053,-15}, {2503,3954,-15}, {2503,4034,-15}, 
    {2503,4037,-15}, {2503,4053,-15}, {2513,3819,-20}, {2513,3820,-20}, 
    {2513,3827,-20}, {2513,3834,-20}, {2513,3842,-20}, {2513,4034,-30}, 
    {2513,4037,-30}, {2513,4053,-30}, {2650,3819,-10}, {2650,3820,-10}, 
    {2650,3827,-10}, {2650,3834,-10}, {2650,3842,-10}, {2650,3905,-40}, 
    {2650,4034,-20}, {2650,4037,-20}, {2650,4053,-20}, {2772,3819,-10}, 
    {2772,3820,-10}, {2772,3827,-10}, {2772,3834,-10}, {2772,3842,-10}, 
    {2772,3905,-40}, {2772,4034,-20}, {2772,4037,-20}, {2772,4053,-20}, 
    {2797,3905,-20}, {2797,3954,-15}, {2797,4023,-30}, {2797,4034,-20}, 
    {2797,4037,-20}, {2797,4053,-20}, {2798,3905,-20}, {2798,3954,-15}, 
    {2798,4023,-30}, {2798,4034,-20}, {2798,4037,-20}, {2798,4053,-20}, 
    {2812,3905,-20}, {2812,3954,-15}, {2812,4023,-30}, {2812,4034,-20}, 
    {2812,4037,-20}, {2812,4053,-20}, {2822,3905,-20}, {2822,3954,-15}, 
    {2822,4023,-30}, {2822,4034,-20}, {2822,4037,-20}, {2822,4053,-20}, 
    {2829,3905,-20}, {2829,3954,-15}, {2829,4023,-30}, {2829,4034,-20}, 
    {2829,4037,-20}, {2829,4053,-20}, {2902,3905,-20}, {2902,3954,-15}, 
    {2902,4023,-30}, {2902,4034,-20}, {2902,4037,-20}, {2902,4053,-20}, 
    {2910,3905,-20}, {2910,3954,-15}, {2910,4023,-30}, {2910,4034,-20}, 
    {2910,4037,-20}, {2910,4053,-20}, {2924,4034,-15}, {2924,4037,-15}, 
    {2924,4053,-15}, {2998,3142,-120}, {2998,3146,-120}, 
    {2998,3501,-40}, {3142,3501,-80}, {3143,3143,-46}, {3146,1483,-80}, 
    {3146,2418,-20}, {3146,2503,-20}, {3146,3146,-46}, {3146,3151,-40}, 
    {3146,3282,-60}, {3146,3320,-60}, {3146,3501,-80}, {3146,3905,-20}, 
    {3151,1314,-20}, {3151,1334,-20}, {3151,1411,-60}, {3151,1483,-20}, 
    {3151,1888,-15}, {3151,2128,-20}, {3151,2797,-20}, {3151,2798,-20}, 
    {3151,2812,-20}, {3151,2822,-20}, {3151,2829,-20}, {3151,2902,-20}, 
    {3151,2910,-20}, {3151,2998,-60}, {3151,3086,-20}, {3151,3282,-15}, 
    {3151,3320,-15}, {3151,3559,20}, {3151,3905,10}, {3151,4034,10}, 
    {3151,4037,10}, {3151,4053,10}, {3282,3954,-15}, {3320,3954,-15}, 
    {3354,3501,-40}, {3501,572,-100}, {3501,660,-80}, {3501,670,-80}, 
    {3501,688,-120}, {3501,689,-120}, {3501,694,-120}, {3501,3138,-80}, 
    {3501,3143,-60}, {3905,736,-20}, {3905,738,-20}, {3905,760,-20}, 
    {3905,777,-20}, {3905,1032,-20}, {3905,1113,-20}, {3905,1175,-20}, 
    {3905,1411,-80}, {3905,2797,-30}, {3905,2798,-30}, {3905,2812,-30}, 
    {3905,2822,-30}, {3905,2829,-30}, {3905,2902,-30}, {3905,2910,-30}, 
    {3905,2998,-80}, {3954,1411,-40}, {3954,2797,-20}, {3954,2798,-20}, 
    {3954,2812,-20}, {3954,2822,-20}, {3954,2829,-20}, {3954,2902,-20}, 
    {3954,2910,-20}, {3954,2998,-40}, {4023,1632,-10}, {4023,1633,-10}, 
    {4023,1647,-10}, {4023,1657,-10}, {4023,1664,-10}, {4034,736,-30}, 
    {4034,738,-30}, {4034,760,-30}, {4034,777,-30}, {4034,1032,-30}, 
    {4034,1113,-30}, {4034,1175,-30}, {4034,1411,-80}, {4034,1632,-10}, 
    {4034,1633,-10}, {4034,1647,-10}, {4034,1657,-10}, {4034,1664,-10}, 
    {4034,2797,-25}, {4034,2798,-25}, {4034,2812,-25}, {4034,2822,-25}, 
    {4034,2829,-25}, {4034,2902,-25}, {4034,2910,-25}, {4034,2998,-80}, 
    {4037,736,-30}, {4037,738,-30}, {4037,760,-30}, {4037,777,-30}, 
    {4037,1032,-30}, {4037,1113,-30}, {4037,1175,-30}, {4037,1411,-80}, 
    {4037,1632,-10}, {4037,1633,-10}, {4037,1647,-10}, {4037,1657,-10}, 
    {4037,1664,-10}, {4037,2797,-25}, {4037,2798,-25}, {4037,2812,-25}, 
    {4037,2822,-25}, {4037,2829,-25}, {4037,2902,-25}, {4037,2910,-25}, 
    {4037,2998,-80}, {4053,736,-30}, {4053,738,-30}, {4053,760,-30}, 
    {4053,777,-30}, {4053,1032,-30}, {4053,1113,-30}, {4053,1175,-30}, 
    {4053,1411,-80}, {4053,1632,-10}, {4053,1633,-10}, {4053,1647,-10}, 
    {4053,1657,-10}, {4053,1664,-10}, {4053,2797,-25}, {4053,2798,-25}, 
    {4053,2812,-25}, {4053,2822,-25}, {4053,2829,-25}, {4053,2902,-25}, 
    {4053,2910,-25}, {4053,2998,-80}, {4135,1632,10}, {4135,1633,10}, 
    {4135,1647,10}, {4135,1657,10}, {4135,1664,10}, {4156,1632,10}, 
    {4156,1633,10}, {4156,1647,10}, {4156,1657,10}, {4156,1664,10}, 
    {NOGLYPH,NOGLYPH,0}
};
static const ligature helvetica_boldoblique_ligs[] = {
    {1789,2134,1809}, {1789,2418,1852}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair courier_kerns[] = {
    
    {NOGLYPH,NOGLYPH,0}
};
static const ligature courier_ligs[] = {
    {1789,2134,1809}, {1789,2418,1852}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair courier_oblique_kerns[] = {
    
    {NOGLYPH,NOGLYPH,0}
};
static const ligature courier_oblique_ligs[] = {
    {1789,2134,1809}, {1789,2418,1852}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair courier_bold_kerns[] = {
    
    {NOGLYPH,NOGLYPH,0}
};
static const ligature courier_bold_ligs[] = {
    {1789,2134,1809}, {1789,2418,1852}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};
static const kern_pair courier_boldoblique_kerns[] = {
    
    {NOGLYPH,NOGLYPH,0}
};
static const ligature courier_boldoblique_ligs[] = {
    {1789,2134,1809}, {1789,2418,1852}, 
    {NOGLYPH,NOGLYPH,NOGLYPH}
};

static const struct ps_std_font_data {
    char const *name;
    kern_pair const *kerns;
    ligature const *ligs;
    int widths[lenof(ps_std_glyphs)-1];
} ps_std_fonts[] = {
    { "Times-Roman",
	times_roman_kerns, times_roman_ligs, {
	722, 667, 667, 722, 611, 556, 722, 722, 333, 389, 722, 611, 889, 
	722, 722, 556, 722, 667, 556, 611, 722, 722, 944, 722, 722, 611, 
	444, 500, 444, 500, 444, 333, 500, 500, 278, 278, 500, 278, 778, 
	500, 500, 500, 500, 333, 389, 278, 500, 500, 722, 500, 500, 444, 
	889, 722, 722, 722, 722, 722, 722, 667, 611, 611, 611, 611, 722, 
	333, 333, 333, 333, 611, 722, 889, 722, 722, 722, 722, 722, 722, 
	556, 556, 722, 722, 722, 722, 722, 722, 611, 667, 444, 444, 444, 
	444, 444, 444, 444, 444, 444, 444, 444, 500, 278, 278, 278, 278, 
	278, 500, 722, 500, 500, 500, 500, 500, 500, 389, 500, 500, 500, 
	500, 500, 500, 500, 444, 333, 778, 469, 541, 500, 921, 278, 200, 
	480, 480, 333, 333, 333, 200, 350, 333, 333, 500, 333, 278, 250, 
	760, 500, 500, 500, 400, 333, 564, 500, 333, 278, 500, 1000, 1000, 
	500, 564, 333, 333, 556, 500, 556, 500, 500, 167, 500, 333, 564, 
	500, 500, 333, 333, 333, 333, 564, 564, 333, 564, 500, 564, 500, 
	500, 333, 500, 750, 750, 300, 276, 310, 453, 333, 333, 833, 250, 
	250, 1000, 564, 564, 444, 444, 408, 444, 444, 444, 333, 333, 333, 
	180, 760, 333, 500, 278, 500, 500, 278, 250, 500, 500, 750, 300, 
	333, 980, 500, 300, 500, 500, 500, 
    }},
    { "Times-Italic",
	times_italic_kerns, times_italic_ligs, {
	611, 611, 667, 722, 611, 611, 722, 722, 333, 444, 667, 556, 833, 
	667, 722, 611, 722, 611, 500, 556, 722, 611, 833, 611, 556, 556, 
	500, 500, 444, 500, 444, 278, 500, 500, 278, 278, 444, 278, 722, 
	500, 500, 500, 500, 389, 389, 278, 500, 444, 667, 444, 444, 389, 
	889, 611, 611, 611, 611, 611, 611, 667, 611, 611, 611, 611, 722, 
	333, 333, 333, 333, 556, 667, 944, 722, 722, 722, 722, 722, 722, 
	500, 611, 722, 722, 722, 722, 556, 556, 556, 667, 500, 500, 500, 
	500, 500, 500, 444, 444, 444, 444, 444, 500, 278, 278, 278, 278, 
	278, 500, 667, 500, 500, 500, 500, 500, 500, 389, 500, 500, 500, 
	500, 500, 444, 444, 389, 333, 778, 422, 541, 500, 920, 278, 275, 
	400, 400, 389, 389, 333, 275, 350, 333, 333, 500, 333, 333, 250, 
	760, 500, 500, 500, 400, 333, 675, 500, 333, 278, 500, 889, 889, 
	500, 675, 333, 389, 500, 500, 500, 500, 500, 167, 500, 333, 675, 
	500, 500, 333, 333, 333, 333, 675, 675, 333, 675, 500, 675, 500, 
	500, 333, 500, 750, 750, 300, 276, 310, 523, 333, 333, 833, 250, 
	250, 1000, 675, 675, 500, 500, 420, 556, 556, 556, 333, 333, 333, 
	214, 760, 333, 500, 333, 500, 500, 278, 250, 500, 500, 750, 300, 
	333, 980, 500, 300, 500, 500, 500, 
    }},
    { "Times-Bold",
	times_bold_kerns, times_bold_ligs, {
	722, 667, 722, 722, 667, 611, 778, 778, 389, 500, 778, 667, 944, 
	722, 778, 611, 778, 722, 556, 667, 722, 722, 1000, 722, 722, 667, 
	500, 556, 444, 556, 444, 333, 500, 556, 278, 333, 556, 278, 833, 
	556, 500, 556, 556, 444, 389, 333, 556, 500, 722, 500, 500, 444, 
	1000, 722, 722, 722, 722, 722, 722, 722, 667, 667, 667, 667, 722, 
	389, 389, 389, 389, 667, 722, 1000, 778, 778, 778, 778, 778, 778, 
	556, 611, 722, 722, 722, 722, 722, 722, 667, 722, 500, 500, 500, 
	500, 500, 500, 444, 444, 444, 444, 444, 500, 278, 278, 278, 278, 
	278, 556, 722, 500, 500, 500, 500, 500, 500, 389, 556, 556, 556, 
	556, 556, 500, 500, 444, 333, 833, 581, 520, 500, 930, 278, 220, 
	394, 394, 333, 333, 333, 220, 350, 333, 333, 500, 333, 333, 250, 
	747, 500, 500, 500, 400, 333, 570, 500, 333, 278, 500, 1000, 1000, 
	500, 570, 333, 333, 556, 500, 556, 500, 500, 167, 556, 333, 570, 
	500, 500, 333, 333, 333, 333, 570, 570, 333, 570, 556, 570, 500, 
	500, 333, 500, 750, 750, 300, 300, 330, 540, 333, 333, 1000, 250, 
	250, 1000, 570, 570, 500, 500, 555, 500, 500, 500, 333, 333, 333, 
	278, 747, 333, 500, 333, 500, 500, 278, 250, 500, 500, 750, 300, 
	333, 1000, 500, 300, 500, 500, 500, 
    }},
    { "Times-BoldItalic",
	times_bolditalic_kerns, times_bolditalic_ligs, {
	667, 667, 667, 722, 667, 667, 722, 778, 389, 500, 667, 611, 889, 
	722, 722, 611, 722, 667, 556, 611, 722, 667, 889, 667, 611, 611, 
	500, 500, 444, 500, 444, 333, 500, 556, 278, 278, 500, 278, 778, 
	556, 500, 500, 500, 389, 389, 278, 556, 444, 667, 500, 444, 389, 
	944, 667, 667, 667, 667, 667, 667, 667, 667, 667, 667, 667, 722, 
	389, 389, 389, 389, 611, 722, 944, 722, 722, 722, 722, 722, 722, 
	556, 611, 722, 722, 722, 722, 611, 611, 611, 722, 500, 500, 500, 
	500, 500, 500, 444, 444, 444, 444, 444, 500, 278, 278, 278, 278, 
	278, 556, 722, 500, 500, 500, 500, 500, 500, 389, 500, 556, 556, 
	556, 556, 444, 444, 389, 333, 778, 570, 570, 500, 832, 278, 220, 
	348, 348, 333, 333, 333, 220, 350, 333, 333, 500, 333, 333, 250, 
	747, 500, 500, 500, 400, 333, 570, 500, 333, 278, 500, 1000, 1000, 
	500, 570, 389, 389, 556, 500, 556, 500, 500, 167, 500, 333, 570, 
	500, 500, 333, 333, 333, 333, 570, 606, 333, 606, 576, 570, 500, 
	500, 333, 500, 750, 750, 300, 266, 300, 500, 333, 333, 833, 250, 
	250, 1000, 570, 570, 500, 500, 555, 500, 500, 500, 333, 333, 333, 
	278, 747, 333, 500, 333, 500, 500, 278, 250, 500, 500, 750, 300, 
	333, 1000, 500, 300, 500, 500, 500, 
    }},
    { "Helvetica",
	helvetica_kerns, helvetica_ligs, {
	667, 667, 722, 722, 667, 611, 778, 722, 278, 500, 667, 556, 833, 
	722, 778, 667, 778, 722, 667, 611, 722, 667, 944, 667, 667, 611, 
	556, 556, 500, 556, 556, 278, 556, 556, 222, 222, 500, 222, 833, 
	556, 556, 556, 556, 333, 500, 278, 556, 500, 722, 500, 500, 500, 
	1000, 667, 667, 667, 667, 667, 667, 722, 667, 667, 667, 667, 722, 
	278, 278, 278, 278, 556, 722, 1000, 778, 778, 778, 778, 778, 778, 
	667, 667, 722, 722, 722, 722, 667, 667, 611, 889, 556, 556, 556, 
	556, 556, 556, 500, 556, 556, 556, 556, 556, 278, 278, 278, 278, 
	222, 556, 944, 556, 556, 556, 556, 611, 556, 500, 556, 556, 556, 
	556, 556, 500, 500, 500, 333, 667, 469, 584, 389, 1015, 278, 260, 
	334, 334, 278, 278, 333, 260, 350, 333, 333, 556, 333, 278, 278, 
	737, 556, 556, 556, 400, 333, 584, 556, 333, 278, 556, 1000, 1000, 
	556, 584, 278, 333, 500, 556, 500, 556, 556, 167, 611, 333, 584, 
	556, 556, 333, 333, 333, 333, 584, 584, 333, 584, 556, 584, 556, 
	556, 333, 556, 834, 834, 333, 370, 365, 537, 333, 333, 889, 278, 
	278, 1000, 584, 584, 556, 611, 355, 333, 333, 333, 222, 222, 222, 
	191, 737, 333, 556, 278, 556, 556, 278, 278, 556, 556, 834, 333, 
	333, 1000, 556, 333, 556, 556, 556, 
    }},
    { "Helvetica-Oblique",
	helvetica_oblique_kerns, helvetica_oblique_ligs, {
	667, 667, 722, 722, 667, 611, 778, 722, 278, 500, 667, 556, 833, 
	722, 778, 667, 778, 722, 667, 611, 722, 667, 944, 667, 667, 611, 
	556, 556, 500, 556, 556, 278, 556, 556, 222, 222, 500, 222, 833, 
	556, 556, 556, 556, 333, 500, 278, 556, 500, 722, 500, 500, 500, 
	1000, 667, 667, 667, 667, 667, 667, 722, 667, 667, 667, 667, 722, 
	278, 278, 278, 278, 556, 722, 1000, 778, 778, 778, 778, 778, 778, 
	667, 667, 722, 722, 722, 722, 667, 667, 611, 889, 556, 556, 556, 
	556, 556, 556, 500, 556, 556, 556, 556, 556, 278, 278, 278, 278, 
	222, 556, 944, 556, 556, 556, 556, 611, 556, 500, 556, 556, 556, 
	556, 556, 500, 500, 500, 333, 667, 469, 584, 389, 1015, 278, 260, 
	334, 334, 278, 278, 333, 260, 350, 333, 333, 556, 333, 278, 278, 
	737, 556, 556, 556, 400, 333, 584, 556, 333, 278, 556, 1000, 1000, 
	556, 584, 278, 333, 500, 556, 500, 556, 556, 167, 611, 333, 584, 
	556, 556, 333, 333, 333, 333, 584, 584, 333, 584, 556, 584, 556, 
	556, 333, 556, 834, 834, 333, 370, 365, 537, 333, 333, 889, 278, 
	278, 1000, 584, 584, 556, 611, 355, 333, 333, 333, 222, 222, 222, 
	191, 737, 333, 556, 278, 556, 556, 278, 278, 556, 556, 834, 333, 
	333, 1000, 556, 333, 556, 556, 556, 
    }},
    { "Helvetica-Bold",
	helvetica_bold_kerns, helvetica_bold_ligs, {
	722, 722, 722, 722, 667, 611, 778, 722, 278, 556, 722, 611, 833, 
	722, 778, 667, 778, 722, 667, 611, 722, 667, 944, 667, 667, 611, 
	556, 611, 556, 611, 556, 333, 611, 611, 278, 278, 556, 278, 889, 
	611, 611, 611, 611, 389, 556, 333, 611, 556, 778, 556, 556, 500, 
	1000, 722, 722, 722, 722, 722, 722, 722, 667, 667, 667, 667, 722, 
	278, 278, 278, 278, 611, 722, 1000, 778, 778, 778, 778, 778, 778, 
	667, 667, 722, 722, 722, 722, 667, 667, 611, 889, 556, 556, 556, 
	556, 556, 556, 556, 556, 556, 556, 556, 611, 278, 278, 278, 278, 
	278, 611, 944, 611, 611, 611, 611, 611, 611, 556, 611, 611, 611, 
	611, 611, 556, 556, 500, 333, 722, 584, 584, 389, 975, 278, 280, 
	389, 389, 333, 333, 333, 280, 350, 333, 333, 556, 333, 333, 278, 
	737, 556, 556, 556, 400, 333, 584, 556, 333, 278, 556, 1000, 1000, 
	556, 584, 333, 333, 611, 556, 611, 556, 556, 167, 611, 333, 584, 
	556, 556, 333, 333, 333, 333, 584, 584, 333, 584, 611, 584, 556, 
	556, 333, 556, 834, 834, 333, 370, 365, 556, 333, 333, 889, 278, 
	278, 1000, 584, 584, 611, 611, 474, 500, 500, 500, 278, 278, 278, 
	238, 737, 333, 556, 333, 556, 556, 278, 278, 556, 556, 834, 333, 
	333, 1000, 556, 333, 556, 556, 556, 
    }},
    { "Helvetica-BoldOblique",
	helvetica_boldoblique_kerns, helvetica_boldoblique_ligs, {
	722, 722, 722, 722, 667, 611, 778, 722, 278, 556, 722, 611, 833, 
	722, 778, 667, 778, 722, 667, 611, 722, 667, 944, 667, 667, 611, 
	556, 611, 556, 611, 556, 333, 611, 611, 278, 278, 556, 278, 889, 
	611, 611, 611, 611, 389, 556, 333, 611, 556, 778, 556, 556, 500, 
	1000, 722, 722, 722, 722, 722, 722, 722, 667, 667, 667, 667, 722, 
	278, 278, 278, 278, 611, 722, 1000, 778, 778, 778, 778, 778, 778, 
	667, 667, 722, 722, 722, 722, 667, 667, 611, 889, 556, 556, 556, 
	556, 556, 556, 556, 556, 556, 556, 556, 611, 278, 278, 278, 278, 
	278, 611, 944, 611, 611, 611, 611, 611, 611, 556, 611, 611, 611, 
	611, 611, 556, 556, 500, 333, 722, 584, 584, 389, 975, 278, 280, 
	389, 389, 333, 333, 333, 280, 350, 333, 333, 556, 333, 333, 278, 
	737, 556, 556, 556, 400, 333, 584, 556, 333, 278, 556, 1000, 1000, 
	556, 584, 333, 333, 611, 556, 611, 556, 556, 167, 611, 333, 584, 
	556, 556, 333, 333, 333, 333, 584, 584, 333, 584, 611, 584, 556, 
	556, 333, 556, 834, 834, 333, 370, 365, 556, 333, 333, 889, 278, 
	278, 1000, 584, 584, 611, 611, 474, 500, 500, 500, 278, 278, 278, 
	238, 737, 333, 556, 333, 556, 556, 278, 278, 556, 556, 834, 333, 
	333, 1000, 556, 333, 556, 556, 556, 
    }},
    { "Courier",
	courier_kerns, courier_ligs, {
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 
    }},
    { "Courier-Oblique",
	courier_oblique_kerns, courier_oblique_ligs, {
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 
    }},
    { "Courier-Bold",
	courier_bold_kerns, courier_bold_ligs, {
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 
    }},
    { "Courier-BoldOblique",
	courier_boldoblique_kerns, courier_boldoblique_ligs, {
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 
	600, 600, 600, 600, 600, 600, 600, 
    }},
};

void init_std_fonts(void) {
    int i, j;
    ligature const *lig;
    kern_pair const *kern;
    static int done = FALSE;

    if (done) return;
    for (i = 0; i < (int)lenof(ps_std_fonts); i++) {
	font_info *fi = snew(font_info);
	fi->fontfile = NULL;
	fi->name = ps_std_fonts[i].name;
	fi->widths = newtree234(width_cmp);
	for (j = 0; j < (int)lenof(fi->bmp); j++)
	    fi->bmp[j] = NOGLYPH;
	for (j = 0; j < (int)lenof(ps_std_glyphs) - 1; j++) {
	    glyph_width *w = snew(glyph_width);
	    wchar_t ucs;
	    w->glyph = glyph_intern(ps_std_glyphs[j]);
	    w->width = ps_std_fonts[i].widths[j];
	    add234(fi->widths, w);
	    ucs = ps_glyph_to_unicode(w->glyph);
	    assert(ucs != 0xFFFF);
	    fi->bmp[ucs] = w->glyph;
	}
	fi->kerns = newtree234(kern_cmp);
	for (kern = ps_std_fonts[i].kerns; kern->left != NOGLYPH; kern++)
	    add234(fi->kerns, (void *)kern);
	fi->ligs = newtree234(lig_cmp);
	for (lig = ps_std_fonts[i].ligs; lig->left != NOGLYPH; lig++)
	    add234(fi->ligs, (void *)lig);
	fi->next = all_fonts;
	all_fonts = fi;
    }
    done = TRUE;
}

const int *ps_std_font_widths(char const *fontname)
{
    int i;

    for (i = 0; i < (int)lenof(ps_std_fonts); i++)
	if (!strcmp(ps_std_fonts[i].name, fontname))
	    return ps_std_fonts[i].widths;

    return NULL;
}

const kern_pair *ps_std_font_kerns(char const *fontname)
{
    int i;

    for (i = 0; i < (int)lenof(ps_std_fonts); i++)
	if (!strcmp(ps_std_fonts[i].name, fontname))
	    return ps_std_fonts[i].kerns;

    return NULL;
}
