// -*- C++ -*-
//
// BELLECharmAnalysis.cc is a part of Herwig++ - A multi-purpose Monte Carlo event generator
// Copyright (C) 2002-2011 The Herwig Collaboration
//
// Herwig++ is licenced under version 2 of the GPL, see COPYING for details.
// Please respect the MCnet academic guidelines, see GUIDELINES for details.
//
//
// This is the implementation of the non-inlined, non-templated member
// functions of the BELLECharmAnalysis class.
//

#include "BELLECharmAnalysis.h"
#include "ThePEG/PDT/EnumParticles.h"
#include "ThePEG/Interface/Switch.h"
#include "ThePEG/Interface/ClassDocumentation.h"
#include "ThePEG/Persistency/PersistentOStream.h"
#include "ThePEG/Persistency/PersistentIStream.h"
#include "ThePEG/EventRecord/Event.h"
#include "ThePEG/Repository/CurrentGenerator.h"

using namespace Herwig;

void BELLECharmAnalysis::analyze(tEventPtr event, long, int, int) {
  _weight = event->weight();
  _s = (event->incoming().first ->momentum()+
	event->incoming().second->momentum()).m2();  
  set<tPPtr> particles;
  StepVector steps = event->primaryCollision()->steps();
  for ( StepVector::const_iterator it = steps.begin()+2;
	it != steps.end(); ++it ) {
    (**it).select(inserter(particles), ThePEG::AllSelector());
  }
  tPVector output;
  for(set<tPPtr>::const_iterator it = particles.begin(); 
      it != particles.end(); ++it) {
    long id = abs((*it)->id());
    if(id==ParticleID::Dplus   || id==ParticleID::Dstarplus    ||
       id==ParticleID::D0      || id==ParticleID::Dstar0       || 
       id==ParticleID::D_splus || id==ParticleID::Lambda_cplus )
      output.push_back(*it);
  }
  analyze(output);
}

void BELLECharmAnalysis::analyze(const tPVector & particles) {
  for(unsigned int ix=0;ix<particles.size();++ix) analyze(particles[ix]);
}

void BELLECharmAnalysis::analyze(tPPtr particle) {
  double xp = particle->momentum().vect().mag()/
    sqrt(0.25*_s-sqr(particle->mass()));
  int id = abs(particle->id());
  if(id==ParticleID::Dstarplus) {
    _histDstarplus  ->addWeighted( xp,_weight);
    _statDstar      += _weight;
  }
  else if(id==ParticleID::Dstar0) {
    _histDstar0     ->addWeighted( xp,_weight);
    _statDstar      += _weight;
  }
  else if(id==ParticleID::D0) {
    _histD0         ->addWeighted( xp,_weight);
    _statD          += _weight;
  }
  else if(id==ParticleID::Dplus) {
    _histDplus      ->addWeighted( xp,_weight);
    _statD          += _weight;
  }
  else if(id==ParticleID::D_splus) {
    _histDs         ->addWeighted( xp,_weight);
    _statDs         += _weight;
  }
  else if(id==ParticleID::Lambda_cplus) {
    _histLambda     ->addWeighted( xp,_weight);
    _statLambda     += _weight;
  }
}

void BELLECharmAnalysis::persistentOutput(PersistentOStream & os) const {
  os << _onshell;
}

void BELLECharmAnalysis::persistentInput(PersistentIStream & is, int) {
  is >> _onshell;
}

ClassDescription<BELLECharmAnalysis> BELLECharmAnalysis::initBELLECharmAnalysis;
// Definition of the static class description member.

void BELLECharmAnalysis::Init() {

  static ClassDocumentation<BELLECharmAnalysis> documentation
    ("The BELLECharmAnalysis class implements the analysis of hep-ex/0506068",
     "The results of Herwig++ were compared with the data of the Belle collaboration"
     "\\cite{Seuster:2005tr} on charm fragmentation functions.",
     "\\bibitem{Seuster:2005tr} R.~Seuster {\\it et al.}  [Belle Collaboration],"
     "Phys.\\ Rev.\\  D {\\bf 73} (2006) 032002 [arXiv:hep-ex/0506068].\n"
     "%%CITATION = PHRVA,D73,032002;%%\n");

  static Switch<BELLECharmAnalysis,bool> interfaceResonance
    ("Resonance",
     "Whether we are doing Upsilon(4S)->B,B or continuum production",
     &BELLECharmAnalysis::_onshell, false, false, false);
  static SwitchOption interfaceResonanceContinuum
    (interfaceResonance,
     "Continuum",
     "Continuum production",
     false);
  static SwitchOption interfaceResonanceUpsilon4S
    (interfaceResonance,
     "Upsilon4S",
     "At Upsilon(4S) resonance",
     true);
}

void BELLECharmAnalysis::dofinish() {
  useMe();
  AnalysisHandler::dofinish();
  string fname = generator()->filename() + 
    string("-") + name() + string(".top");
  ofstream output(fname.c_str());
  using namespace HistogramOptions;
  double chisq=0.,minfrac=0.05;
  unsigned int ndegrees;
  _histDstarplus->chiSquared(chisq,ndegrees,minfrac);
  generator()->log() << "Chi Square = " << chisq << " for " << ndegrees 
			  << " degrees of freedom for BELLE D*+ distribution\n";
  _histDstar0   ->chiSquared(chisq,ndegrees,minfrac);
  generator()->log() << "Chi Square = " << chisq << " for " << ndegrees 
			  << " degrees of freedom for BELLE D*0 distribution\n";
  _histD0       ->chiSquared(chisq,ndegrees,minfrac);
  generator()->log() << "Chi Square = " << chisq << " for " << ndegrees 
			  << " degrees of freedom for BELLE D0 distribution\n";
  _histDplus    ->chiSquared(chisq,ndegrees,minfrac);
  generator()->log() << "Chi Square = " << chisq << " for " << ndegrees 
			  << " degrees of freedom for BELLE D+ distribution\n";
  _histDs       ->chiSquared(chisq,ndegrees,minfrac);
  generator()->log() << "Chi Square = " << chisq << " for " << ndegrees 
			  << " degrees of freedom for BELLE D_s+ distribution\n";
  _histLambda   ->chiSquared(chisq,ndegrees,minfrac);
  generator()->log() << "Chi Square = " << chisq << " for " << ndegrees 
			  << " degrees of freedom for BELLE Lambda_c+ distribution\n";
  _histDstarplus->topdrawOutput(output,Frame|Errorbars,
				"RED",
				"D2*+3",
				" X  X",
				"1/SdS/dx0p1",
				"  G G   X X",
				"x0p1",
				" X X");
  _histDstar0->topdrawOutput(output,Frame|Errorbars,
			     "RED",
			     "D2*03",
			     " X  X",
			     "1/SdS/dx0p1",
			     "  G G   X X",
			     "x0p1",
			     " X X");
  _histD0->topdrawOutput(output,Frame|Errorbars,
			 "RED",
			 "D203",
			 " X X",
			 "1/SdS/dx0p1",
			 "  G G   X X",
			 "x0p1",
			 " X X");
  _histDplus->topdrawOutput(output,Frame|Errorbars,
			    "RED",
			    "D2+3",
			    " X X",
			    "1/SdS/dx0p1",
			    "  G G   X X",
			    "x0p1",
			    " X X");
  _histDs->topdrawOutput(output,Frame|Errorbars,
			 "RED",
			 "D0s12+3",
			 " X XX X",
			 "1/SdS/dx0p1",
			 "  G G   X X",
			 "x0p1",
			 " X X");
  _histLambda->topdrawOutput(output,Frame|Errorbars,
			     "RED",
			     "L0c12+3",
			     "FX XX X",
			     "1/SdS/dx0p1",
			     "  G G   X X",
			     "x0p1",
			     " X X");
  // and the ratios
  double denom1 = _statD.total();
  double denom2 = denom1+_statDs.total();
  double data[3] ={_statDstar .total()/denom1,
		   _statDs    .total()/denom2,
		   _statLambda.total()/denom2};
  double derror[3] = {data[0]*sqrt(1./denom1+1./_statDstar .total()),
		      data[1]*sqrt(1./denom2+1./_statDs    .total()),
		      data[2]*sqrt(1./denom2+1./_statLambda.total())};
  double ratio[3]={0.527,0.099,0.081};
  double error[3]={0.027,0.004,0.004};
  for(unsigned int ix=0;ix<3;++ix) {
    double chisq = sqr(data[ix]-ratio[ix])/(sqr(derror[ix])+sqr(error[ix]));
    generator()->log() << "Chi Square = " << chisq 
			    << " for 1 degree of freedon for BELLE ";
    if(ix==0) {
      generator()->log() << "Ratio of D*/D = ";
    }
    else if(ix==1) {
      generator()->log() << "Ratio of D_s/D = ";
    }
    else if(ix==2) {
      generator()->log() << "Ratio of Lambda_c/D = ";
    }  
    generator()->log() << "Hw++ = " << data[ix]  << " +/- " << derror[ix] << "\t"
			    << "Exp = "  << ratio[ix] << " +/- " << error[ix]  << "\n";
  }
}

void BELLECharmAnalysis::doinitrun() {
  AnalysisHandler::doinitrun();
  // the bin sizes (same for all the plots)
  double vals[]={0.00,0.02,0.04,0.06,0.08,0.10,0.12,0.14,0.16,0.18,
		 0.20,0.22,0.24,0.26,0.28,0.30,0.32,0.34,0.36,0.38,
		 0.40,0.42,0.44,0.46,0.48,0.50,0.52,0.54,0.56,0.58,
		 0.60,0.62,0.64,0.66,0.68,0.70,0.72,0.74,0.76,0.78,
		 0.80,0.82,0.84,0.86,0.88,0.90,0.92,0.94,0.96,0.98,
		 1.00,1.02,1.04,1.06,1.08,1.10,1.12,1.14,1.16,1.18,
		 1.20};
  // the data and errors for D*+
  // continuum
  double DstarContAdata []={0.0000  ,0.0000  ,0.0000  ,0.0000  ,0.1086  ,
			    0.0920  ,0.2600  ,0.1734  ,0.2212  ,0.4413  ,
			    0.3998  ,0.5073  ,0.6654  ,0.7496  ,1.0015  ,
			    0.8738  ,1.2712  ,1.2758  ,1.5916  ,1.7495  ,
			    1.9353  ,2.1832  ,2.3189  ,2.5587  ,2.8917  ,
			    3.0588  ,3.2292  ,3.4595  ,3.7583  ,3.9273  ,
			    3.8202  ,3.8493  ,3.8297  ,3.7796  ,3.7061  ,
			    3.4861  ,3.3356  ,3.2040  ,2.9714  ,2.7053  ,
			    2.4643  ,2.2032  ,1.9259  ,1.7746  ,1.5298  ,
			    1.2213  ,0.9208  ,0.7591  ,0.5077  ,0.3008  ,
			    0.0752  ,0.0000  ,0.0007  ,0.0003  ,0.0003  ,
			    0.0000  ,0.0000  ,0.0004  ,0.0007  ,0.0000  };
  double DstarContAerror[]={0.0000  ,0.0000  ,0.0000  ,0.0000  ,0.0945  ,
			    0.0651  ,0.0782  ,0.0672  ,0.0559  ,0.0606  ,
			    0.0703  ,0.0802  ,0.1147  ,0.1003  ,0.0519  ,
			    0.0818  ,0.0591  ,0.0688  ,0.0608  ,0.0508  ,
			    0.0770  ,0.0695  ,0.1046  ,0.0685  ,0.0827  ,
			    0.0743  ,0.0942  ,0.0873  ,0.0777  ,0.0740  ,
			    0.0854  ,0.0804  ,0.0694  ,0.0720  ,0.0725  ,
			    0.0738  ,0.0674  ,0.0712  ,0.0533  ,0.0598  ,
			    0.0595  ,0.0542  ,0.0582  ,0.0470  ,0.0481  ,
			    0.0491  ,0.0377  ,0.0343  ,0.0412  ,0.0287  ,
			    0.0096  ,0.0000  ,0.0007  ,0.0000  ,0.0000  ,
			    0.0000  ,0.0000  ,0.0001  ,0.0007  ,0.0000  };
  double DstarContAsyst []={0.0000  ,0.0000  ,0.0000  ,0.0000  ,0.1089  ,
			    0.1010  ,0.1116  ,0.0260  ,0.0331  ,0.0470  ,
			    0.0298  ,0.1081  ,0.0374  ,0.0403  ,0.0775  ,
			    0.0257  ,0.0770  ,0.0456  ,0.0729  ,0.0675  ,
			    0.0668  ,0.0638  ,0.0705  ,0.0760  ,0.1014  ,
			    0.0834  ,0.1208  ,0.1058  ,0.1048  ,0.1141  ,
			    0.1179  ,0.1175  ,0.1137  ,0.1305  ,0.1276  ,
			    0.0944  ,0.1488  ,0.0804  ,0.1300  ,0.0827  ,
			    0.0611  ,0.0556  ,0.0498  ,0.0503  ,0.0384  ,
			    0.0339  ,0.0235  ,0.0210  ,0.0177  ,0.0221  ,
			    0.0212  ,0.0000  ,0.0005  ,0.0004  ,0.0003  ,
			    0.0000  ,0.0002  ,0.0003  ,0.0004  ,0.0000};
  double DstarContBdata []={0.0023  ,0.0192  ,0.1091  ,0.0119  ,0.2982  ,
			    0.1810  ,0.5365  ,0.5271  ,0.4393  ,0.5188  ,
			    1.2347  ,0.9356  ,1.2284  ,1.3936  ,1.6509  ,
			    1.5206  ,1.0046  ,1.5886  ,1.5468  ,2.4733  ,
			    2.1732  ,2.2865  ,2.4508  ,3.2556  ,2.9489  ,
			    3.4497  ,3.2338  ,4.0084  ,3.3482  ,3.7872  ,
			    4.0491  ,4.0065  ,3.8393  ,3.8573  ,3.9580  ,
			    3.4841  ,3.2875  ,3.0220  ,3.0135  ,2.6343  ,
			    2.1812  ,2.1588  ,2.1708  ,1.8158  ,1.4513  ,
			    1.1350  ,1.0107  ,0.7388  ,0.4409  ,0.1652  ,
			    0.0076  ,0.0000  ,0.0000  ,0.0000  ,0.0000  ,
			    0.0002  ,0.0000  ,0.0003  ,0.0000  ,0.0000};
  double DstarContBerror[]={0.0029  ,0.0235  ,0.0694  ,0.2135  ,0.1164  ,
			    0.1500  ,0.1942  ,0.2272  ,0.2671  ,0.2697  ,
			    0.2797  ,0.3122  ,0.2844  ,0.3007  ,0.3027  ,
			    0.2881  ,0.3061  ,0.2927  ,0.2742  ,0.2868  ,
			    0.2623  ,0.2652  ,0.2541  ,0.2541  ,0.2410  ,
			    0.2388  ,0.2307  ,0.2307  ,0.2112  ,0.2102  ,
			    0.2116  ,0.1995  ,0.1875  ,0.1811  ,0.1778  ,
			    0.1649  ,0.1562  ,0.1460  ,0.1405  ,0.1307  ,
			    0.1171  ,0.1131  ,0.1126  ,0.1016  ,0.0929  ,
			    0.0801  ,0.0792  ,0.0712  ,0.0584  ,0.0415  ,
			    0.0019  ,0.0000  ,0.0000  ,0.0000  ,0.0000  ,
			    0.0003  ,0.0000  ,0.0003  ,0.0000  ,0.0000};
  double DstarContBsyst []={0.0289  ,0.0279  ,0.1111  ,0.1412  ,0.1884  ,
			    0.2026  ,0.2951  ,0.4560  ,0.3143  ,0.2830  ,
			    0.8143  ,0.5703  ,0.5426  ,0.8053  ,0.8518  ,
			    0.7047  ,0.6345  ,0.4422  ,0.6728  ,0.7195  ,
			    0.5979  ,0.5913  ,0.4242  ,0.8650  ,0.3603  ,
			    0.3100  ,0.4185  ,0.5449  ,0.7992  ,0.6498  ,
			    0.3426  ,0.4139  ,0.4198  ,0.3723  ,0.1777  ,
			    0.3743  ,0.3991  ,0.5463  ,0.2445  ,0.3707  ,
			    0.5247  ,0.2772  ,0.1936  ,0.1444  ,0.2310  ,
			    0.2199  ,0.0455  ,0.1016  ,0.1170  ,0.1720  ,
			    0.0753  ,0.0000  ,0.0008  ,0.0003  ,0.0003  ,
			    0.0003  ,0.0000  ,0.0001  ,0.0008  ,0.0004};
  // upsilon(4s)  
  double DstarResAdata [] ={0.0000  ,0.0000  ,0.0000  ,0.4738  ,1.5171  ,
			    2.3419  ,2.8561  ,3.4064  ,3.8241  ,4.1637  ,
			    4.6871  ,5.0245  ,5.3191  ,5.3183  ,4.9756  ,
			    5.3471  ,5.4127  ,5.6859  ,5.0897  ,4.3337  ,
			    4.2906  ,4.2940  ,3.8785  ,3.4359  ,3.1051  ,
			    3.1234  ,3.2755  ,3.5321  ,3.6719  ,3.7688  ,
			    3.7823  ,3.8011  ,3.9374  ,3.7599  ,3.6048  ,
			    3.5008  ,3.2785  ,3.0835  ,2.9116  ,2.6687  ,
			    2.4073  ,2.1714  ,1.9326  ,1.6977  ,1.4383  ,
			    1.1891  ,0.9458  ,0.7230  ,0.4886  ,0.2788  ,
			    0.0560  ,0.0004  ,0.0000  ,0.0001  ,0.0001  ,
			    0.0001  ,0.0004  ,0.0001  ,0.0001  ,0.0002};
  double DstarResAerror[] ={0.0000  ,0.0000  ,0.0000  ,0.0827  ,0.0814  ,
			    0.0742  ,0.0695  ,0.0660  ,0.0617  ,0.0582  ,
			    0.0898  ,0.0762  ,0.0725  ,0.0664  ,0.0639  ,
			    0.0607  ,0.0604  ,0.0563  ,0.0541  ,0.0488  ,
			    0.0506  ,0.0500  ,0.0440  ,0.0417  ,0.0398  ,
			    0.0393  ,0.0406  ,0.0394  ,0.0415  ,0.0402  ,
			    0.0394  ,0.0391  ,0.0347  ,0.0374  ,0.0372  ,
			    0.0355  ,0.0345  ,0.0328  ,0.0319  ,0.0292  ,
			    0.0285  ,0.0281  ,0.0254  ,0.0249  ,0.0247  ,
			    0.0240  ,0.0232  ,0.0231  ,0.0235  ,0.0185  ,
			    0.0059  ,0.0002  ,0.0000  ,0.0001  ,0.0000  ,
			    0.0000  ,0.0004  ,0.0000  ,0.0001  ,0.0002};
   double DstarResAsyst []={0.0000  ,0.0000  ,0.0000  ,0.4952  ,0.1372  ,
			    0.1752  ,0.1725  ,0.1311  ,0.1107  ,0.1715  ,
			    0.1424  ,0.1415  ,0.1428  ,0.1425  ,0.1335  ,
			    0.1851  ,0.1591  ,0.1564  ,0.1502  ,0.1209  ,
			    0.1471  ,0.1470  ,0.1160  ,0.0948  ,0.0846  ,
			    0.0896  ,0.0879  ,0.0911  ,0.0935  ,0.1639  ,
			    0.0945  ,0.1078  ,0.1546  ,0.0968  ,0.0927  ,
			    0.0944  ,0.1009  ,0.0796  ,0.0738  ,0.0717  ,
			    0.0653  ,0.0757  ,0.0494  ,0.0447  ,0.0413  ,
			    0.0306  ,0.0238  ,0.0204  ,0.0163  ,0.0303  ,
			    0.0167  ,0.0004  ,0.0001  ,0.0001  ,0.0002  ,
			    0.0003  ,0.0004  ,0.0001  ,0.0002  ,0.0003};
  double DstarResBdata [] ={0.1165  ,0.4929  ,1.0685  ,1.4522  ,2.3575  ,
			    3.2876  ,4.0676  ,4.6098  ,5.0437  ,5.4440  ,
			    6.0972  ,5.8901  ,6.4027  ,6.1881  ,5.9507  ,
			    5.7318  ,6.1615  ,6.4023  ,5.5579  ,4.9284  ,
			    4.5515  ,4.7132  ,3.8666  ,3.7695  ,3.0282  ,
			    3.3300  ,3.4129  ,3.4566  ,3.6621  ,3.6642  ,
			    3.5726  ,3.5974  ,3.5789  ,3.4910  ,3.4001  ,
			    3.3657  ,3.0850  ,2.9208  ,2.7965  ,2.5201  ,
			    2.3858  ,2.0900  ,1.8334  ,1.6066  ,1.3657  ,
			    1.1287  ,0.8791  ,0.6908  ,0.4402  ,0.2154  ,
			    0.0074  ,0.0000  ,0.0000  ,0.0000  ,0.0000  ,
			    0.0003  ,0.0001  ,0.0000  ,0.0000  ,0.0000};
  double DstarResBerror[] ={0.0551  ,0.0953  ,0.1464  ,0.1641  ,0.2045  ,
			    0.2484  ,0.2532  ,0.2739  ,0.2811  ,0.2803  ,
			    0.2781  ,0.2721  ,0.2662  ,0.2609  ,0.2462  ,
			    0.2263  ,0.2298  ,0.2155  ,0.1894  ,0.1785  ,
			    0.1646  ,0.1637  ,0.1423  ,0.1364  ,0.1200  ,
			    0.1188  ,0.1156  ,0.1084  ,0.1060  ,0.0991  ,
			    0.0969  ,0.0921  ,0.0869  ,0.0835  ,0.0803  ,
			    0.0770  ,0.0717  ,0.0685  ,0.0654  ,0.0610  ,
			    0.0585  ,0.0541  ,0.0505  ,0.0474  ,0.0453  ,
			    0.0420  ,0.0402  ,0.0405  ,0.0371  ,0.0329  ,
			    0.0009  ,0.0000  ,0.0000  ,0.0000  ,0.0000  ,
			    0.0002  ,0.0001  ,0.0001  ,0.0000  ,0.0000};
  double DstarResBsyst [] ={0.1372  ,0.4965  ,1.0767  ,0.9674  ,0.7464  ,
			    1.0696  ,1.3322  ,1.2031  ,1.6356  ,1.5783  ,
			    1.4752  ,1.4304  ,1.0702  ,1.2437  ,0.9721  ,
			    0.9081  ,1.0710  ,0.9747  ,0.7575  ,0.8353  ,
			    0.5420  ,0.5751  ,0.6969  ,0.4650  ,0.6019  ,
			    0.3728  ,0.3793  ,0.5501  ,0.4483  ,0.6431  ,
			    0.6985  ,0.6916  ,0.7151  ,0.6843  ,0.5956  ,
			    0.5118  ,0.5377  ,0.4870  ,0.4397  ,0.4362  ,
			    0.3022  ,0.3202  ,0.3053  ,0.2776  ,0.2307  ,
			    0.1823  ,0.1645  ,0.1084  ,0.1023  ,0.0940  ,
			    0.0546  ,0.0005  ,0.0000  ,0.0001  ,0.0002  ,
			    0.0004  ,0.0004  ,0.0001  ,0.0001  ,0.0002};
  // combine the data
  vector<double> bins(vals,vals+61),data(60),error(60);
  double norm(0.);
  for(unsigned int ix=0;ix<60;++ix) {
    if(_onshell) {
      data[ix]  = DstarResAdata[ix]+DstarResBdata[ix]
	        -(DstarContAdata[ix]+DstarContBdata[ix]);
      error[ix] = sqrt(sqr(DstarResAerror[ix])+sqr(DstarResAsyst[ix])+
		       sqr(DstarResBerror[ix])+sqr(DstarResBsyst[ix])+
		       sqr(DstarContAerror[ix])+sqr(DstarContAsyst[ix])+
		       sqr(DstarContBerror[ix])+sqr(DstarContBsyst[ix]));
    }
    else {
      data[ix]  = DstarContAdata[ix]+DstarContBdata[ix];
      error[ix] = sqrt(sqr(DstarContAerror[ix])+sqr(DstarContAsyst[ix])+
		       sqr(DstarContBerror[ix])+sqr(DstarContBsyst[ix]));
    }
    norm +=data[ix];
  }
  norm*=0.02;
  for(unsigned int ix=0;ix<data.size();++ix) {
    data [ix] /=norm;
    error[ix] /=norm;
  }
  _histDstarplus = new_ptr(Histogram(bins,data,error));
  // data and errors for D*0
  double DstarContCdata [] = {0.0005  ,0.0429  ,0.0628  ,0.0637  ,0.1361  ,
			      0.2393  ,0.2248  ,0.2236  ,0.0759  ,0.2966  ,
			      0.3302  ,0.3380  ,0.6534  ,0.6235  ,0.7017  ,
			      0.9976  ,0.9747  ,1.0301  ,0.9653  ,1.3007  ,
			      1.2644  ,1.7510  ,1.8537  ,2.1379  ,2.4625  ,
			      2.6152  ,2.4718  ,2.6468  ,2.6830  ,3.2672  ,
			      2.5735  ,3.0625  ,2.9439  ,2.8022  ,2.7553  ,
			      2.7874  ,2.6514  ,2.6271  ,2.3999  ,2.0013  ,
			      1.8978  ,1.8005  ,1.4439  ,1.2328  ,1.2430  ,
			      1.0324  ,0.7418  ,0.6244  ,0.3600  ,0.1391  ,
			      0.0067  ,0.0000  ,0.0000  ,0.0001  ,0.0000  ,
			      0.0000  ,0.0000  ,0.0000  ,0.0000  ,0.0000};
  double DstarContCerror[] = {0.0019  ,0.0349  ,0.0552  ,0.0839  ,0.1468  ,
			      0.0934  ,0.1193  ,0.1710  ,0.1373  ,0.1408  ,
			      0.1439  ,0.1434  ,0.1367  ,0.1373  ,0.1327  ,
			      0.1394  ,0.1319  ,0.1305  ,0.1227  ,0.1256  ,
			      0.1245  ,0.1265  ,0.1240  ,0.1305  ,0.1326  ,
			      0.1325  ,0.1265  ,0.1303  ,0.1262  ,0.1362  ,
			      0.1201  ,0.1269  ,0.1221  ,0.1162  ,0.1144  ,
			      0.1137  ,0.1082  ,0.1059  ,0.0993  ,0.0911  ,
			      0.0875  ,0.0841  ,0.0747  ,0.0677  ,0.0699  ,
			      0.0644  ,0.0559  ,0.0526  ,0.0440  ,0.0283  ,
			      0.0018  ,0.0000  ,0.0000  ,0.0005  ,0.0000  ,
			      0.0000  ,0.0000  ,0.0000  ,0.0000  ,0.0000};
  double DstarContCsyst [] = {0.0160  ,0.0541  ,0.0935  ,0.1248  ,0.1730  ,
			      0.1432  ,0.3045  ,0.2947  ,0.3041  ,0.2320  ,
			      0.6625  ,0.3682  ,0.4239  ,0.5345  ,0.4599  ,
			      0.4959  ,0.3333  ,0.2009  ,0.1674  ,0.4672  ,
			      0.1038  ,0.1506  ,0.1571  ,0.3942  ,0.2303  ,
			      0.1200  ,0.3101  ,0.3079  ,0.6658  ,0.4874  ,
			      0.1710  ,0.3015  ,0.3281  ,0.2784  ,0.1781  ,
			      0.3159  ,0.3318  ,0.4386  ,0.1970  ,0.3156  ,
			      0.4567  ,0.2463  ,0.0822  ,0.1441  ,0.2078  ,
			      0.1855  ,0.0286  ,0.0850  ,0.1021  ,0.1365  ,
			      0.0627  ,0.0000  ,0.0007  ,0.0003  ,0.0008  ,
			      0.0002  ,0.0000  ,0.0003  ,0.0006  ,0.0000};
  double DstarResCdata  [] = {0.0499  ,0.2866  ,0.7566  ,1.2445  ,1.7512  ,
			      2.0055  ,2.7251  ,3.3537  ,3.7734  ,3.9434  ,
			      3.9614  ,4.6476  ,4.4744  ,4.6342  ,4.3167  ,
			      4.6727  ,4.8370  ,5.1344  ,4.5411  ,4.1157  ,
			      4.0264  ,3.8603  ,3.4919  ,2.8508  ,2.4374  ,
			      2.2697  ,2.3825  ,2.5889  ,2.7219  ,2.9223  ,
			      2.8554  ,2.8909  ,2.9134  ,2.7963  ,2.6736  ,
			      2.5551  ,2.5038  ,2.3519  ,2.1896  ,2.0004  ,
			      1.8195  ,1.5966  ,1.4875  ,1.2618  ,1.0930  ,
			      0.9624  ,0.7382  ,0.5387  ,0.3747  ,0.1922  ,
			      0.0100  ,0.0004  ,0.0000  ,0.0002  ,0.0000  ,
			      0.0003  ,0.0000  ,0.0000  ,0.0000  ,0.0001};
  double DstarResCerror [] = {0.0137  ,0.0371  ,0.0709  ,0.0813  ,0.0968  ,
			      0.0982  ,0.1108  ,0.1218  ,0.1224  ,0.1208  ,
			      0.1185  ,0.1254  ,0.1193  ,0.1205  ,0.1141  ,
			      0.1137  ,0.1109  ,0.1141  ,0.1029  ,0.0979  ,
			      0.0944  ,0.0865  ,0.0811  ,0.0716  ,0.0656  ,
			      0.0623  ,0.0605  ,0.0625  ,0.0613  ,0.0633  ,
			      0.0607  ,0.0597  ,0.0591  ,0.0567  ,0.0540  ,
			      0.0526  ,0.0512  ,0.0485  ,0.0461  ,0.0439  ,
			      0.0415  ,0.0384  ,0.0373  ,0.0341  ,0.0335  ,
			      0.0335  ,0.0318  ,0.0286  ,0.0299  ,0.0243  ,
			      0.0010  ,0.0003  ,0.0000  ,0.0002  ,0.0000  ,
			      0.0002  ,0.0000  ,0.0003  ,0.0000  ,0.0001};
  double DstarResCsyst  [] = {0.1002  ,0.4131  ,0.9090  ,0.7991  ,0.6240  ,
			      0.6420  ,0.8793  ,0.7754  ,0.7611  ,0.7890  ,
			      0.9126  ,0.5415  ,0.5097  ,0.5498  ,0.4939  ,
			      0.2492  ,0.3606  ,0.3842  ,0.2612  ,0.3417  ,
			      0.3242  ,0.2607  ,0.3608  ,0.1207  ,0.3845  ,
			      0.1929  ,0.2109  ,0.4158  ,0.3605  ,0.4936  ,
			      0.5290  ,0.5272  ,0.5955  ,0.5716  ,0.4944  ,
			      0.4321  ,0.4581  ,0.4050  ,0.3519  ,0.3660  ,
			      0.2370  ,0.2660  ,0.2606  ,0.2215  ,0.1866  ,
			      0.1552  ,0.1378  ,0.0915  ,0.0835  ,0.0781  ,
			      0.0454  ,0.0005  ,0.0000  ,0.0001  ,0.0001  ,
			      0.0002  ,0.0003  ,0.0002  ,0.0001  ,0.0002};
  norm=0.;
  for(unsigned int ix=0;ix<60;++ix) {
    if(_onshell) {
      data[ix]  = DstarResCdata[ix]-DstarContCdata[ix];
      error[ix] = sqrt(sqr(DstarResCerror[ix])+sqr(DstarResCsyst[ix])+
		       sqr(DstarContCerror[ix])+sqr(DstarContCsyst[ix]));
    }
    else {
      data[ix]  = DstarContCdata[ix];
      error[ix] = sqrt(sqr(DstarContCerror[ix])+sqr(DstarContCsyst[ix]));
    }
    norm +=data[ix];
  }
  norm*=0.02;
  for(unsigned int ix=0;ix<data.size();++ix) {
    data [ix] /=norm;
    error[ix] /=norm;
  }
  _histDstar0 = new_ptr(Histogram(bins,data,error));
  // D0
  double D0Contdata []={ 0.0169 , 0.0833 , 0.0618 , 0.1986 , 0.4482 , 
			 0.7078 , 1.2171 , 1.2134 , 1.0833 , 1.7438 , 
			 2.0325 , 2.2870 , 2.7788 , 3.3657 , 3.8424 , 
			 4.0673 , 5.2573 , 5.9915 , 6.9013 , 7.9586 , 
			 8.7103 , 9.7516 ,10.780  ,11.833  ,12.588  ,
			 13.333 ,14.118  ,14.520  ,14.634  ,14.631  ,
			 14.792 ,14.038  ,13.677  ,12.910  ,12.316  ,
			 11.687 ,10.443  , 9.6756 , 8.5980 , 7.7212 ,
			 6.6372 , 5.6244 , 4.4447 , 3.6675 , 2.7390 , 
			 1.8597 , 1.1354 , 0.5906 , 0.3842 , 0.1368 , 
			 0.0005 , 0.0006 , 0.0000 , 0.0001 , 0.0000 , 
			 0.0000 , 0.0010 , 0.0000 , 0.0000 , 0.0000};
  double D0Conterror[]={0.0292 ,0.0587 ,0.1030 ,0.1188 ,0.1706 ,
			0.1790 ,0.1868 ,0.2096 ,0.2049 ,0.2129 ,
			0.2114 ,0.2049 ,0.2062 ,0.2021 ,0.1936 ,
			0.1905 ,0.1858 ,0.1839 ,0.1804 ,0.1793 ,
			0.1770 ,0.1754 ,0.1747 ,0.1741 ,0.1726 ,
			0.1709 ,0.1700 ,0.1681 ,0.1651 ,0.1617 ,
			0.1600 ,0.1539 ,0.1503 ,0.1443 ,0.1405 ,
			0.1347 ,0.1267 ,0.1221 ,0.1157 ,0.1091 ,
			0.1016 ,0.0952 ,0.0865 ,0.0799 ,0.0631 ,
			0.0548 ,0.0464 ,0.0367 ,0.0422 ,0.0345 ,
			0.0171 ,0.0084 ,0.0000 ,0.0284 ,0.0000 ,
			0.0000 ,0.0037 ,0.0000 ,0.0000 ,0.0000};
  double D0Contsyst []={ 0.0034  ,0.0371  ,0.0672  ,0.0409  ,0.1235  ,
			 0.1347  ,0.1904  ,0.1068  ,0.1620  ,0.1482  ,
			 0.1350  ,0.1249  ,0.2211  ,0.1672  ,0.1875  ,
			 0.2001  ,0.2818  ,0.2955  ,0.2418  ,0.2760  ,
			 0.2875  ,0.3735  ,0.5162  ,0.4231  ,0.4262  ,
			 0.4586  ,0.5054  ,0.5626  ,0.6052  ,0.6230  ,
			 0.6430  ,0.7133  ,0.6407  ,0.5828  ,0.5499  ,
			 0.5147  ,0.4593  ,0.5165  ,0.4916  ,0.5091  ,
			 0.4028  ,0.3546  ,0.3498  ,0.2157  ,0.1926  ,
			 0.1333  ,0.0482  ,0.0375  ,0.0197  ,0.0362  ,
			 0.0036  ,0.0024  ,0.0000  ,0.0031  ,0.0000  ,
			 0.0000  ,0.0006  ,0.0000  ,0.0016  ,0.0000};
  double D0Resdata  []={0.2158 , 1.3354 , 3.5464 , 6.2104 , 9.1055 ,
			11.743 ,14.185  ,16.600  ,18.420  ,19.861  ,
			21.068 ,22.052  ,22.610  ,21.903  ,22.920  ,
			23.370 ,23.409  ,22.102  ,21.291  ,20.201  ,
			18.593 ,17.130  ,15.455  ,14.157  ,13.408  ,
			13.367 ,13.600  ,14.146  ,14.318  ,14.529  ,
			14.361 ,13.890  ,13.497  ,12.769  ,12.112  ,
			11.349 ,10.411  , 9.4833 , 8.6169 , 7.6538 ,
			6.6222 , 5.5604 , 4.5685 , 3.5991 , 2.6699 , 
			1.7757 , 1.0773 , 0.5975 , 0.3697 , 0.1322 , 
			0.0012 , 0.0000 , 0.0000 , 0.0000 , 0.0006 , 
			0.0000 , 0.0000 , 0.0000 , 0.0000 , 0.0013 };
  double D0Reserror []={0.0139 ,0.0343 ,0.0577 ,0.0799 ,0.0980 ,
			0.1111 ,0.1227 ,0.1305 ,0.1371 ,0.1411 ,
			0.1435 ,0.1463 ,0.1492 ,0.1447 ,0.1447 ,
			0.1440 ,0.1404 ,0.1366 ,0.1321 ,0.1251 ,
			0.1175 ,0.1098 ,0.1018 ,0.0951 ,0.0894 ,
			0.0862 ,0.0841 ,0.4752 ,0.0815 ,0.0798 ,
			0.1961 ,0.1058 ,0.0736 ,0.1009 ,0.0884 ,
			0.0756 ,0.0735 ,0.0688 ,0.0575 ,0.0616 ,
			0.0522 ,0.0498 ,0.0472 ,0.0445 ,0.0391 ,
			0.0337 ,0.0292 ,0.0241 ,0.0307 ,0.0554 ,
			0.0171 ,0.0000 ,0.0000 ,0.0000 ,0.0020 ,
			0.0000 ,0.0000 ,0.0000 ,0.0000 ,0.0018};
  double D0Ressyst  []={0.0241  ,0.0936  ,0.1466  ,0.1804  ,0.2883  ,
			0.3625  ,0.3514  ,0.4124  ,0.4660  ,0.5604  ,
			0.6975  ,0.6903  ,0.7772  ,0.7347  ,0.8244  ,
			0.8277  ,0.8621  ,0.8435  ,0.8492  ,0.7592  ,
			0.6783  ,0.6762  ,0.6239  ,0.5569  ,0.5639  ,
			0.5113  ,0.5373  ,0.5896  ,0.6189  ,0.6497  ,
			0.6166  ,0.6128  ,0.6263  ,0.6027  ,0.5568  ,
			0.5054  ,0.4987  ,0.5208  ,0.4783  ,0.4779  ,
			0.4491  ,0.3651  ,0.3119  ,0.2262  ,0.2316  ,
			0.1255  ,0.0737  ,0.0297  ,0.0277  ,0.0221  ,
			0.0018  ,0.0000  ,0.0000  ,0.0003  ,0.0009  ,
			0.0000  ,0.0000  ,0.0004  ,0.0000  ,0.0013};
  norm=0.;
  for(unsigned int ix=0;ix<60;++ix) {
    if(_onshell) {
      data[ix]  = D0Resdata[ix]-D0Contdata[ix];
      error[ix] = sqrt(sqr(D0Reserror[ix])+sqr(D0Ressyst[ix])+
		       sqr(D0Conterror[ix])+sqr(D0Contsyst[ix]));
    }
    else {
      data[ix]  = D0Contdata[ix];
      error[ix] = sqrt(sqr(D0Conterror[ix])+sqr(D0Contsyst[ix]));
    }
    norm +=data[ix];
  }
  norm*=0.02;
  for(unsigned int ix=0;ix<data.size();++ix) {
    data [ix] /=norm;
    error[ix] /=norm;
  }
  _histD0 = new_ptr(Histogram(bins,data,error));
  // D+
  double DplusContdata [] = {0.0000  , 0.0000  , 0.5818  , 0.6357  , 0.1329  ,
			     0.8692  , 0.4138  , 1.6552  , 2.5470  , 1.3803  ,
			     2.7601  , 2.1947  , 3.9075  , 3.6200  , 4.2296  ,
			     4.9135  , 5.2604  , 5.9759  , 7.3663  , 8.1743  ,
			     8.4325  , 9.7974  ,11.461   ,12.077   ,12.934   ,
			     13.708  ,14.067   ,14.725   ,14.963   ,15.423   ,
			     15.320  ,15.190   ,15.055   ,14.245   ,13.537   ,
			     12.619  ,11.922   ,11.056   , 9.7519  , 9.0101  ,
			     8.1814  , 6.8664  , 5.6215  , 4.8304  , 3.8260  , 
			     2.9192  , 1.9412  , 1.2589  , 0.8647  , 0.4013  , 
			     0.0000  , 0.0030  , 0.0001  , 0.0019  , 0.0024  , 
			     0.0007  , 0.0000  , 0.0000  , 0.0005  , 0.0000};
  double DplusConterror[] = {0.0000  ,0.0000  ,0.2483  ,0.3362  ,0.3210  ,
			     0.4320  ,0.3511  ,0.5107  ,0.4657  ,0.4867  ,
			     0.4947  ,0.4704  ,0.4833  ,0.4373  ,0.4511  ,
			     0.4313  ,0.4162  ,0.4001  ,0.4001  ,0.3773  ,
			     0.3670  ,0.3562  ,0.3507  ,0.3345  ,0.3275  ,
			     0.3147  ,0.3010  ,0.2905  ,0.2802  ,0.2721  ,
			     0.2575  ,0.2451  ,0.2352  ,0.2222  ,0.2106  ,
			     0.1959  ,0.1855  ,0.1750  ,0.1614  ,0.1526  ,
			     0.1416  ,0.1297  ,0.1156  ,0.1075  ,0.0861  ,
			     0.0780  ,0.0672  ,0.0567  ,0.0592  ,0.0781  ,
			     0.0000  ,0.0037  ,0.0204  ,0.0031  ,0.0030  ,
			     0.0028  ,0.0000  ,0.0000  ,0.0039  ,0.0000};
  double DplusContsyst [] = {0.0089  ,0.0387  ,0.0883  ,0.1988  ,0.1961  ,
			     0.1366  ,0.1952  ,0.3483  ,0.8036  ,0.4267  ,
			     0.3463  ,0.2670  ,0.3700  ,0.6879  ,0.4151  ,
			     0.3116  ,0.3172  ,0.2360  ,0.6964  ,0.5013  ,
			     0.5977  ,0.4255  ,0.6292  ,0.4448  ,0.4600  ,
			     0.5626  ,0.5610  ,0.5647  ,0.5255  ,0.5867  ,
			     0.5850  ,0.6885  ,0.5657  ,0.5309  ,0.5721  ,
			     0.5157  ,0.5405  ,0.4071  ,0.3945  ,0.3568  ,
			     0.3436  ,0.2785  ,0.2288  ,0.1772  ,0.1493  ,
			     0.1128  ,0.0755  ,0.0946  ,0.0340  ,0.0888  ,
			     0.0000  ,0.0030  ,0.0024  ,0.0021  ,0.0013  ,
			     0.0010  ,0.0000  ,0.0012  ,0.0008  ,0.0007};
  double DplusResdata  [] = {0.1550  , 1.5006  , 3.1119  , 6.4047  , 8.9433  ,
			     11.700   ,12.478   ,15.654   ,16.602   ,18.075   ,
			     19.110   ,20.501   ,21.173   ,20.674   ,22.020   ,
			     21.443   ,22.801   ,23.363   ,22.590   ,20.531   ,
			     18.777   ,19.212   ,17.918   ,15.831   ,14.004   ,
			     13.599   ,13.707   ,14.565   ,15.145   ,15.326   ,
			     15.049   ,14.756   ,14.570   ,14.051   ,13.479   ,
			     12.685   ,11.859   ,11.030   , 9.9618  , 9.1037  ,
			     7.8287  , 6.8724  , 5.6991  , 4.7644  , 3.6555  , 
			     2.7980  , 1.9429  , 1.2478  , 0.7765  , 0.5288  , 
			     0.0045  , 0.0000  , 0.0000  , 0.0000  , 0.0000  , 
			     0.0000  , 0.0007  , 0.0013  , 0.0023  , 0.0000};
  double DplusReserror [] = {0.0407  ,0.1314  ,0.1924  ,0.2836  ,0.3264  ,
			     0.3597  ,0.3502  ,0.3808  ,0.3652  ,0.3687  ,
			     0.3687  ,0.3628  ,0.3605  ,0.3390  ,0.3315  ,
			     0.3106  ,0.2970  ,0.2881  ,0.2702  ,0.2409  ,
			     0.2328  ,0.2205  ,0.2086  ,0.1887  ,0.1721  ,
			     0.1684  ,0.1570  ,3.0100  ,0.4975  ,0.1386  ,
			     1.0654  ,0.2339  ,0.1170  ,0.1102  ,0.2722  ,
			     0.0977  ,0.1126  ,0.1151  ,0.0818  ,0.0781  ,
			     0.0715  ,0.0813  ,0.0737  ,0.0581  ,0.0479  ,
			     0.0461  ,0.0433  ,0.0371  ,0.1249  ,0.0812  ,
			     0.0070  ,0.0000  ,0.0000  ,0.0000  ,0.0000  ,
			     0.0000  ,0.0013  ,0.0013  ,0.0013  ,0.0000};
  double DplusRessyst  [] = {0.0114  ,0.0644  ,0.1420  ,0.3082  ,0.3491  ,
			     0.5140  ,0.6339  ,0.6431  ,0.7199  ,0.6802  ,
			     0.6784  ,0.8869  ,1.0151  ,0.8385  ,0.8824  ,
			     0.8406  ,0.9693  ,0.8910  ,0.8966  ,0.8640  ,
			     0.7759  ,0.7758  ,0.6574  ,0.5959  ,0.5031  ,
			     0.5601  ,0.4837  ,0.5873  ,0.5450  ,0.5444  ,
			     0.5404  ,0.5806  ,0.5533  ,0.5293  ,0.5188  ,
			     0.4884  ,0.4699  ,0.4564  ,0.3867  ,0.3564  ,
			     0.3258  ,0.2715  ,0.2275  ,0.1951  ,0.1517  ,
			     0.1071  ,0.0744  ,0.0695  ,0.0561  ,0.1143  ,
			     0.0012  ,0.0000  ,0.0000  ,0.0000  ,0.0000  ,
			     0.0000  ,0.0005  ,0.0004  ,0.0013  ,0.0000};
  norm=0.;
  for(unsigned int ix=0;ix<60;++ix) {
    if(_onshell) {
      data[ix]  = DplusResdata[ix]-DplusContdata[ix];
      error[ix] = sqrt(sqr(DplusReserror[ix])+sqr(DplusRessyst[ix])+
		       sqr(DplusConterror[ix])+sqr(DplusContsyst[ix]));
    }
    else {
      data[ix]  = DplusContdata[ix];
      error[ix] = sqrt(sqr(DplusConterror[ix])+sqr(DplusContsyst[ix]));
    }
    norm +=data[ix];
  }
  norm*=0.02;
  for(unsigned int ix=0;ix<data.size();++ix) {
    data [ix] /=norm;
    error[ix] /=norm;
  }
  _histDplus = new_ptr(Histogram(bins,data,error));
  // D_s
  double DsContdata []={0.0000  ,0.0215  ,0.0523  ,0.0000  ,0.0192  ,
			0.1315  ,0.2369  ,0.2176  ,0.2092  ,0.0000  ,
			0.1286  ,0.0000  ,0.1710  ,0.4137  ,0.6091  ,
			0.4799  ,0.3845  ,0.4657  ,0.6666  ,0.8618  ,
			0.7695  ,0.9957  ,0.6926  ,1.0742  ,1.2848  ,
			1.1490  ,1.4815  ,1.7732  ,1.6550  ,2.0110  ,
			2.2453  ,2.1992  ,2.2760  ,2.4153  ,2.3540  ,
			2.2840  ,2.1807  ,2.0987  ,2.0491  ,1.8800  ,
			1.7452  ,1.5000  ,1.3928  ,1.0368  ,0.8881  ,
			0.6345  ,0.4031  ,0.3867  ,0.1964  ,0.1030  ,
			0.0001  ,0.0000  ,0.0003  ,0.0000  ,0.0003  ,
			0.0003  ,0.0000  ,0.0003  ,0.0000  ,0.0000};
  double DsConterror[]={0.0000  ,0.0215  ,0.0409  ,0.0000  ,0.0874  ,
			0.1225  ,0.1223  ,0.1021  ,0.1049  ,0.0000  ,
			0.1247  ,0.0000  ,0.1248  ,0.1241  ,0.1218  ,
			0.1210  ,0.1126  ,0.1127  ,0.1092  ,0.1101  ,
			0.1064  ,0.1077  ,0.0966  ,0.1004  ,0.1016  ,
			0.0934  ,0.0970  ,0.0995  ,0.0943  ,0.0986  ,
			0.0990  ,0.0972  ,0.0950  ,0.0935  ,0.0912  ,
			0.0886  ,0.0851  ,0.0832  ,0.0806  ,0.0785  ,
			0.0757  ,0.0702  ,0.0685  ,0.0601  ,0.0555  ,
			0.0494  ,0.0409  ,0.0510  ,0.0586  ,0.0406  ,
			0.0007  ,0.0000  ,0.0003  ,0.0000  ,0.0003  ,
			0.0003  ,0.0000  ,0.0003  ,0.0000  ,0.0000 };
  double DsContsyst []={0.0000  ,0.0075  ,0.0086  ,0.0000  ,0.0719  ,
			0.0695  ,0.0345  ,0.1379  ,0.0277  ,0.1283  ,
			0.0769  ,0.0936  ,0.1908  ,0.0817  ,0.0814  ,
			0.0760  ,0.0835  ,0.0390  ,0.1038  ,0.0569  ,
			0.0855  ,0.0952  ,0.0733  ,0.0524  ,0.0602  ,
			0.0477  ,0.0827  ,0.1078  ,0.1010  ,0.0976  ,
			0.0789  ,0.0731  ,0.0954  ,0.1753  ,0.1315  ,
			0.1019  ,0.0752  ,0.1184  ,0.0687  ,0.0792  ,
			0.0727  ,0.0610  ,0.0469  ,0.0556  ,0.0445  ,
			0.0317  ,0.0155  ,0.0144  ,0.0331  ,0.0152  ,
			0.0002  ,0.0002  ,0.0001  ,0.0000  ,0.0000  ,
			0.0003  ,0.0000  ,0.0000  ,0.0000  ,0.0000};
  double DsResdata  []={0.0172  ,0.1635  ,0.4140  ,0.7689  ,1.0447  ,
			1.5749  ,1.4311  ,1.9089  ,2.1637  ,2.4421  ,
			2.9070  ,3.0959  ,3.3758  ,4.1464  ,5.2309  ,
			6.0358  ,6.8050  ,6.2537  ,4.9198  ,2.6739  ,
			1.1283  ,0.8664  ,0.8912  ,1.1026  ,1.1718  ,
			1.3390  ,1.5067  ,1.7534  ,1.9282  ,1.9846  ,
			2.0309  ,2.2831  ,2.3075  ,2.3346  ,2.4522  ,
			2.2865  ,2.1992  ,2.1336  ,2.0154  ,1.8252  ,
			1.6409  ,1.4550  ,1.2669  ,0.9831  ,0.8357  ,
			0.5785  ,0.4427  ,0.3233  ,0.2478  ,0.0932  ,
			0.0005  ,0.0000  ,0.0000  ,0.0000  ,0.0000  ,
			0.0001  ,0.0000  ,0.0000  ,0.0000  ,0.0000};
  double DsReserror []={0.0060  ,0.0327  ,0.0602  ,0.0802  ,0.0850  ,
			0.1283  ,0.0963  ,0.1264  ,0.1129  ,0.1212  ,
			0.1440  ,0.1321  ,0.1325  ,0.1383  ,0.1364  ,
			0.1277  ,0.1306  ,0.1175  ,0.1045  ,0.0769  ,
			0.0594  ,0.0501  ,0.0495  ,0.0499  ,0.0489  ,
			0.0945  ,0.0470  ,0.0479  ,0.0772  ,0.0475  ,
			0.0578  ,0.0475  ,0.0463  ,0.0455  ,0.0464  ,
			0.0441  ,0.0427  ,0.0420  ,0.0410  ,0.0396  ,
			0.0388  ,0.0373  ,0.0366  ,0.0334  ,0.0332  ,
			0.0297  ,0.0294  ,0.0330  ,0.0599  ,0.0316  ,
			0.0004  ,0.0000  ,0.0000  ,0.0002  ,0.0001  ,
			0.0001  ,0.0001  ,0.0000  ,0.0000  ,0.0000};
  double DsRessyst  []={0.0022  ,0.0124  ,0.0197  ,0.0272  ,0.0688  ,
			0.0534  ,0.0623  ,0.1163  ,0.1213  ,0.0866  ,
			0.1406  ,0.1193  ,0.1181  ,0.1573  ,0.1862  ,
			0.2050  ,0.2479  ,0.2265  ,0.1930  ,0.1158  ,
			0.0467  ,0.0381  ,0.0332  ,0.0418  ,0.0411  ,
			0.0479  ,0.0587  ,0.0657  ,0.0644  ,0.0752  ,
			0.0682  ,0.0959  ,0.0943  ,0.0860  ,0.0919  ,
			0.0824  ,0.0811  ,0.0756  ,0.0804  ,0.0703  ,
			0.0848  ,0.0767  ,0.0505  ,0.1049  ,0.0305  ,
			0.0420  ,0.0230  ,0.0132  ,0.0133  ,0.0071  ,
			0.0002  ,0.0001  ,0.0000  ,0.0001  ,0.0001  ,
			0.0000  ,0.0000  ,0.0000  ,0.0000  ,0.0000};
  norm=0.;
  for(unsigned int ix=0;ix<60;++ix) {
    if(_onshell) {
      data[ix]  = DsResdata[ix]-DsContdata[ix];
      error[ix] = sqrt(sqr(DsReserror[ix])+sqr(DsRessyst[ix])+
		       sqr(DsConterror[ix])+sqr(DsContsyst[ix]));
    }
    else {
      data[ix]  = DsContdata[ix];
      error[ix] = sqrt(sqr(DsConterror[ix])+sqr(DsContsyst[ix]));
    }
    norm +=data[ix];
  }
  norm*=0.02;
  for(unsigned int ix=0;ix<data.size();++ix) {
    data [ix] /=norm;
    error[ix] /=norm;
  }
  _histDs = new_ptr(Histogram(bins,data,error));
  // lambda_c
  double LambdaContdata []={0.0139  ,0.0000  ,0.0000  ,0.2306  ,0.0000  ,
			    0.0000  ,0.0000  ,0.1742  ,0.3650  ,0.0595  ,
			    0.0000  ,0.1462  ,0.1987  ,0.3494  ,0.5546  ,
			    0.7100  ,0.9732  ,1.2286  ,1.2439  ,0.8432  ,
			    1.3625  ,1.0046  ,1.8011  ,1.8175  ,2.2466  ,
			    2.0503  ,2.3058  ,2.6803  ,2.9076  ,2.9698  ,
			    2.9984  ,2.9810  ,2.6864  ,2.7354  ,2.6587  ,
			    2.4398  ,1.9472  ,1.8592  ,1.5798  ,1.2745  ,
			    1.0149  ,0.7507  ,0.5853  ,0.3793  ,0.2852  ,
			    0.1373  ,0.1369  ,0.1237  ,0.0495  ,0.0109  ,
			    0.0054  ,0.0000  ,0.0000  ,0.0000  ,0.0035  ,
			    0.0000  ,0.0000  ,0.0010  ,0.0000  ,0.0000};
  double LambdaConterror[]={0.0159  ,0.0000  ,0.0638  ,0.1087  ,0.0000  ,
			    0.0000  ,0.0000  ,0.2015  ,0.2347  ,0.2075  ,
			    0.0000  ,0.2120  ,0.2144  ,0.2115  ,0.2075  ,
			    0.1949  ,0.2042  ,0.1816  ,0.1794  ,0.1727  ,
			    0.1606  ,0.1500  ,0.1608  ,0.1471  ,0.1446  ,
			    0.1366  ,0.1349  ,0.1309  ,0.1304  ,0.1251  ,
			    0.1208  ,0.1163  ,0.1129  ,0.1074  ,0.1036  ,
			    0.0994  ,0.0922  ,0.0884  ,0.0830  ,0.0756  ,
			    0.0697  ,0.0653  ,0.0619  ,0.0548  ,0.0493  ,
			    0.0439  ,0.0457  ,0.0466  ,0.0530  ,0.0075  ,
			    0.0068  ,0.0000  ,0.0000  ,0.0000  ,0.0056  ,
			    0.0000  ,0.0000  ,0.0060  ,0.0000  ,0.0000};
  double LambdaContsyst []={0.0212  ,0.0065  ,0.0384  ,0.0460  ,0.0222  ,
			    0.0127  ,0.0022  ,0.0697  ,0.0694  ,0.1109  ,
			    0.0000  ,0.0827  ,0.1871  ,0.1175  ,0.5312  ,
			    0.2119  ,0.0841  ,0.1294  ,0.0923  ,0.2728  ,
			    0.0993  ,0.1427  ,0.2887  ,0.0801  ,0.1223  ,
			    0.1335  ,0.1354  ,0.1388  ,0.2330  ,0.1289  ,
			    0.1052  ,0.1177  ,0.1754  ,0.1521  ,0.1078  ,
			    0.0872  ,0.0853  ,0.0858  ,0.0924  ,0.0562  ,
			    0.0434  ,0.0416  ,0.0957  ,0.0204  ,0.0300  ,
			    0.0225  ,0.0870  ,0.0698  ,0.0301  ,0.0039  ,
			    0.0060  ,0.0000  ,0.0000  ,0.0005  ,0.0050  ,
			    0.0000  ,0.0017  ,0.0026  ,0.0000  ,0.0013};
  double LambdaResdata [] ={0.0754  ,0.4204  ,0.8565  ,1.4055  ,1.6783  ,
			    2.0153  ,2.4834  ,2.6237  ,2.8581  ,2.7140  ,
			    2.7020  ,2.3296  ,1.9372  ,1.6154  ,1.2013  ,
			    1.2710  ,1.1848  ,1.1810  ,1.1716  ,1.2500  ,
			    1.1281  ,1.5176  ,1.6848  ,1.8331  ,1.8510  ,
			    2.1514  ,2.5126  ,2.5917  ,2.7706  ,2.8117  ,
			    2.9390  ,2.8475  ,2.9111  ,2.7268  ,2.4615  ,
			    2.3572  ,2.1411  ,1.7850  ,1.5107  ,1.2671  ,
			    0.9243  ,0.7535  ,0.5556  ,0.3460  ,0.2931  ,
			    0.1487  ,0.1007  ,0.0470  ,0.0034  ,0.0038  ,
			    0.0018  ,0.0005  ,0.0000  ,0.0030  ,0.0000  ,
			    0.0021  ,0.0008  ,0.0000  ,0.0000  ,0.0000  };
  double LambdaReserror[] ={0.0227  ,0.0489  ,0.0673  ,0.0947  ,0.1075  ,
			    0.1135  ,0.1280  ,0.1356  ,0.1516  ,0.1427  ,
			    0.1489  ,0.1328  ,0.1239  ,0.1238  ,0.1056  ,
			    0.1090  ,0.1102  ,0.0975  ,0.0982  ,0.1028  ,
			    0.0740  ,0.0942  ,0.0882  ,0.0771  ,0.0722  ,
			    0.0729  ,0.3043  ,0.0667  ,0.1758  ,0.0686  ,
			    0.0620  ,0.0586  ,0.1371  ,0.0546  ,0.1111  ,
			    0.0941  ,0.0466  ,0.0770  ,0.1435  ,0.0367  ,
			    0.0640  ,0.0311  ,0.0292  ,0.0248  ,0.0227  ,
			    0.0209  ,0.0619  ,0.0198  ,0.0304  ,0.0033  ,
			    0.0030  ,0.0038  ,0.0000  ,0.0026  ,0.0000  ,
			    0.0023  ,0.0022  ,0.0000  ,0.0000  ,0.0000  };
  double LambdaRessyst [] ={0.0067  ,0.0199  ,0.0449  ,0.0837  ,0.0576  ,
			    0.1153  ,0.1054  ,0.1077  ,0.1346  ,0.0964  ,
			    0.1063  ,0.2158  ,0.1677  ,0.1223  ,0.1309  ,
			    0.1205  ,0.0527  ,0.0794  ,0.0610  ,0.1053  ,
			    0.1826  ,0.0706  ,0.0738  ,0.1364  ,0.1319  ,
			    0.1011  ,0.0910  ,0.0887  ,0.1353  ,0.1149  ,
			    0.1423  ,0.1238  ,0.1136  ,0.0949  ,0.0907  ,
			    0.0931  ,0.0893  ,0.1040  ,0.0685  ,0.0595  ,
			    0.0356  ,0.0497  ,0.0270  ,0.0218  ,0.0132  ,
			    0.0254  ,0.0256  ,0.0120  ,0.0178  ,0.0799  ,
			    0.0016  ,0.0006  ,0.0000  ,0.0015  ,0.0002  ,
			    0.0029  ,0.0013  ,0.0003  ,0.0000  ,0.0000};
  norm=0.;
  for(unsigned int ix=0;ix<60;++ix) {
    if(_onshell) {
      data[ix]  = LambdaResdata[ix]-LambdaContdata[ix];
      error[ix] = sqrt(sqr(LambdaReserror[ix])+sqr(LambdaRessyst[ix])+
		       sqr(LambdaConterror[ix])+sqr(LambdaContsyst[ix]));
    }
    else {
      data[ix]  = LambdaContdata[ix];
      error[ix] = sqrt(sqr(LambdaConterror[ix])+sqr(LambdaContsyst[ix]));
    }
    norm +=data[ix];
  }
  norm*=0.02;
  for(unsigned int ix=0;ix<data.size();++ix) {
    data [ix] /=norm;
    error[ix] /=norm;
  }
  _histLambda = new_ptr(Histogram(bins,data,error));
}
