/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include <xine.h>

#include "_xitk.h"
#include "image.h"
#include "mrlbrowser.h"
#include "browser.h"
#include "label.h"
#include "labelbutton.h"
#include "button.h"
#include "combo.h"

#include "utils.h"

#include "_xitk.h"
#include "button_list.h"

typedef struct {
  xine_mrl_t                 xmrl;
  char                      *disp, *last;
} _mrlb_item_t;

typedef struct {
  _mrlb_item_t              *items;
  char                     **f_list;
  int                       *f_indx, f_num, i_num;
} _mrlb_items_t;

typedef struct {
  xitk_widget_t              w;

  xitk_t                    *xitk;
  xitk_register_key_t        widget_key;

  xitk_window_t             *xwin;

  xitk_widget_list_t        *widget_list; /* File browser widget list */

  xine_t                    *xine;

  char                      *last_mrl_source;

  xitk_widget_t             *widget_origin; /* Current directory widget */

  int                        visible; /* Boolean status */

  _mrlb_items_t              items;

  xitk_widget_t             *list_title_w, *mrlb_list; /*  Browser list widget */
  xitk_widget_t             *autodir_buttons, *last_clicked;
  xitk_widget_t             *play_add[2]; /* for dis/enable */

  xitk_widget_t             *combo_filter;
  struct {
    const char             **names;
    int                      selected;
    int                    (*match) (void *data, const char *name, uint32_t n);
  }                          filter;

  int                        sel;
  unsigned int               have_callbacks;

  xitk_mrl_callback_t        add_callback;
  xitk_mrl_callback_t        play_callback;
  xitk_simple_callback_t     kill_callback;

  xitk_be_event_handler_t   *input_cb;
  void                      *input_cb_data;

  char                       list_title_s[64];
  char                       current_origin[XITK_PATH_MAX] ; /* Current directory */
} _mrlbrowser_private_t;

#undef DEBUG_MRLB

static void _mrlb_items_free (_mrlb_items_t *items) {
  free (items->items);
  items->items = NULL;
  items->i_num = 0;
  free (items->f_list);
  items->f_list = NULL;
  free (items->f_indx);
  items->f_indx = NULL;
  items->f_num = 0;
}

/*
 * Destroy the mrlbrowser.
 */
static void _mrlbrowser_destroy (_mrlbrowser_private_t *wp) {
  xitk_unregister_event_handler (wp->xitk, &wp->widget_key);

  xitk_window_destroy_window (wp->xwin);
  wp->xwin = NULL;

  _mrlb_items_free (&wp->items);

  XITK_FREE (wp->last_mrl_source);
}

static int notify_event(xitk_widget_t *w, const widget_event_t *event) {
  _mrlbrowser_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_MRLBROWSER)
    return 0;

  switch (event->type) {
    case WIDGET_EVENT_DESTROY:
      _mrlbrowser_destroy (wp);
      break;
    default: ;
  }
  return 0;
}

/*
 *
 */
static int _mrlbrowser_filter_mrls (_mrlbrowser_private_t *wp) {
  int nsel;

  if (!wp->items.f_list)
    return -1;

  if (_ZERO_TO_MAX_MINUS_1 (wp->sel, wp->items.f_num))
    wp->sel = wp->items.f_indx[wp->sel];
  else
    wp->sel = -1;
  nsel = -1;

  if (wp->filter.selected) { /* filtering is enabled */
    int i, j;
    for (i = j = 0; i < wp->items.i_num; i++) {
      int keep = 1;
      if (!(wp->items.items[i].xmrl.type & XINE_MRL_TYPE_file_directory) && wp->filter.match)
        keep = wp->filter.match (wp->w.userdata, wp->items.items[i].last, wp->filter.selected);
      if (keep) {
        wp->items.f_list[j] = wp->items.items[i].disp;
        wp->items.f_indx[j] = i;
        if (wp->sel == i)
          nsel = j;
        j++;
      }
    }
    wp->items.f_list[j] = NULL;
    wp->items.f_num = j;
  } else { /* no filtering */
    int i;
    for (i = 0; i < wp->items.i_num; i++) {
      wp->items.f_list[i] = wp->items.items[i].disp;
      wp->items.f_indx[i] = i;
    }
    wp->items.f_list[i] = NULL;
    wp->items.f_num = i;
    nsel = wp->sel;
  }
  return nsel;
}

/*
 * Duplicate mrls from xine-engine's returned. We shouldn't play
 * directly with these ones.
 */
static void _mrlbrowser_duplicate_mrls (_mrlb_items_t *items, xine_mrl_t **mtmp, int num_mrls) {
  char *mem, *g_orig;
  size_t slen, g_olen;
  int i, g_onum;

  items->items = NULL;
  items->i_num = 0;
  items->f_list = NULL;
  items->f_indx = NULL;
  items->f_num = 0;

  if (!mtmp)
    return;

  items->f_list = malloc ((num_mrls + 1) * sizeof (*items->f_list));
  items->f_indx = malloc (num_mrls * sizeof (*items->f_indx));

  /* origin usually is the same for all items. */
  g_orig = (num_mrls > 0) ? mtmp[0]->origin : NULL;
  g_olen = g_orig ? xitk_find_byte (g_orig, 0) + 1 : 0;
  g_onum = 0;
  slen = 0;
  for (i = 0; i < num_mrls; i++) {
    xine_mrl_t *m = mtmp[i];
    const char *orig = m->origin ? m->origin : "";
    size_t olen = xitk_find_byte (orig, 0) + 1;
    size_t mlen = (m->mrl ? xitk_find_byte (m->mrl, 0) : 0) + 1;
    size_t llen = (m->link ? xitk_find_byte (m->link, 0) : 0) + 1;
    size_t dlen = (mlen > olen ? mlen - olen + 1 : mlen) + ((m->type & XINE_MRL_TYPE_file_symlink) ? llen + 5 : 1);
    slen += olen + mlen + llen + dlen;
    if ((olen == g_olen) && !memcmp (orig, g_orig, g_olen))
      g_onum += 1;
  }
  if (g_onum != num_mrls)
    g_orig = NULL;
  if (g_orig)
    slen -= (g_onum - 1) * g_olen;

  mem = malloc (num_mrls * sizeof (_mrlb_item_t) + slen);
  if (!mem || !items->f_list || !items->f_indx) {
    free (mem);
    free (items->f_list);
    items->f_list = NULL;
    free (items->f_indx);
    items->f_indx = NULL;
    return;
  }
  items->i_num = num_mrls;
  items->items = (_mrlb_item_t *)mem;
  mem += num_mrls * sizeof (_mrlb_item_t);

  if (g_orig) {
    memcpy (mem, g_orig, g_olen);
    g_orig = mem;
    mem += g_olen;
  }

  for (i = 0; i < num_mrls; i++) {
    xine_mrl_t *m = mtmp[i];
    _mrlb_item_t *item = items->items + i;
    size_t olen;
    size_t mlen = (m->mrl ? xitk_find_byte (m->mrl, 0) : 0) + 1;
    size_t llen = (m->link ? xitk_find_byte (m->link, 0) : 0) + 1;
    char *p;

    if (g_orig) {
      olen = g_olen;
      item->xmrl.origin = g_orig;
    } else {
      olen = (m->origin ? xitk_find_byte (m->origin, 0) : 0) + 1;
      item->xmrl.origin = mem;
      memcpy (mem, m->origin ? m->origin : "", olen);
      mem += olen;
    }
    item->xmrl.mrl = mem;
    memcpy (mem, m->mrl ? m->mrl : "", mlen);
    mem += mlen;
    mem[-1] = '/';
    for (item->last = p = item->xmrl.mrl; p + 1 < mem; p += xitk_find_byte (p, '/') + 1)
      item->last = p;
    mem[-1] = 0;
    item->xmrl.link = mem;
    memcpy (mem, m->link ? m->link : "", llen);
    mem += llen;
    item->xmrl.type = m->type;
    item->xmrl.size = m->size;

    item->disp = mem;
    if (mlen > olen) {
      if (m->mrl[olen - 1] != '/')
        olen -= 1;
      memcpy (mem, m->mrl + olen, mlen - olen - 1);
      mem += mlen - olen - 1;
    } else if (m->mrl) {
      memcpy (mem, m->mrl, mlen - 1);
      mem += mlen - 1;
    }
    if (m->type & XINE_MRL_TYPE_file_symlink) {
      memcpy (mem, "@ -> ", 5);
      mem += 5;
      memcpy (mem, m->link ? m->link : "", llen);
      mem += llen;
    } else if (m->type & XINE_MRL_TYPE_file_directory) {
      memcpy (mem, "/", 2);
      mem += 2;
    } else if (m->type & XINE_MRL_TYPE_file_fifo) {
      memcpy (mem, "|", 2);
      mem += 2;
    } else if (m->type & XINE_MRL_TYPE_file_sock) {
      memcpy (mem, "=", 2);
      mem += 2;
    } else if (m->type & XINE_MRL_TYPE_file_exec) {
      memcpy (mem, "*", 2);
      mem += 2;
    } else {
      *mem++ = 0;
    }
  }
}

static void _mrlbrowser_set_source (_mrlbrowser_private_t *wp, const char *mrl_source, const char *dir) {
  char *old_source = NULL;
  _mrlb_items_t items = wp->items;
  int num_mrls;
  xine_mrl_t **mtmp;

  if (mrl_source) {
    old_source = wp->last_mrl_source;
    wp->last_mrl_source = strdup (mrl_source);
  }
  mtmp = xine_get_browse_mrls (wp->xine, wp->last_mrl_source, dir, &num_mrls);
  _mrlbrowser_duplicate_mrls (&wp->items, mtmp, num_mrls);
  free (old_source);

  strlcpy (wp->current_origin,
    (wp->items.i_num && wp->items.items[0].xmrl.origin) ? wp->items.items[0].xmrl.origin : "",
    sizeof (wp->current_origin));
  xitk_label_change_label (wp->widget_origin, wp->current_origin);

  _mrlbrowser_filter_mrls (wp);
  xitk_browser_update_list (wp->mrlb_list, (const char* const *)wp->items.f_list, NULL, wp->items.f_num, 0);
  _mrlb_items_free (&items);
  wp->sel = xitk_browser_set_select (wp->mrlb_list, -1);
  xitk_widgets_state (wp->play_add, 2, XITK_WIDGET_STATE_ENABLE, (wp->sel >= 0) ? ~0u : 0);
}

/*
 * Grab mrls from xine-engine.
 */
static void mrlbrowser_grab_mrls (xitk_widget_t *w, void *data, int state) {
  _mrlbrowser_private_t *wp = (_mrlbrowser_private_t *)data;
  const char *lbl = xitk_labelbutton_get_label (w);

  if (!state) {
    if (wp->last_clicked == w)
      xitk_widgets_state (&wp->last_clicked, 1, XITK_WIDGET_STATE_ON, ~0u);
    return;
  }
  wp->last_clicked = w;

  if (lbl)
    _mrlbrowser_set_source (wp, lbl, NULL);
}

#ifdef DEBUG_MRLB
/*
 * Dump informations, on terminal, about selected mrl.
 */
static void mrlbrowser_dumpmrl(xitk_widget_t *w, void *data) {
  _mrlbrowser_private_t *wp= (_mrlbrowser_private_t *)data;
  int j = -1;

  if ((j = wp->sel) >= 0) {
    xine_mrl_t *ms = wp->mc->mrls[j];

    printf("mrl '%s'\n\t+", ms->mrl);

    if(ms->type & XINE_MRL_TYPE_file_symlink)
      printf(" symlink to '%s' \n\t+ ", ms->link);

    printf("[");

    if(ms->type & XINE_MRL_TYPE_unknown)
      printf(" unknown ");

      if(ms->type & XINE_MRL_TYPE_dvd)
	printf(" dvd ");

      if(ms->type & XINE_MRL_TYPE_vcd)
	    printf(" vcd ");

      if(ms->type & XINE_MRL_TYPE_net)
	printf(" net ");

      if(ms->type & XINE_MRL_TYPE_rtp)
	printf(" rtp ");

      if(ms->type & XINE_MRL_TYPE_stdin)
	printf(" stdin ");

      if(ms->type & XINE_MRL_TYPE_file)
	printf(" file ");

      if(ms->type & XINE_MRL_TYPE_file_fifo)
	printf(" fifo ");

      if(ms->type & XINE_MRL_TYPE_file_chardev)
	printf(" chardev ");

      if(ms->type & XINE_MRL_TYPE_file_directory)
	printf(" directory ");

      if(ms->type & XINE_MRL_TYPE_file_blockdev)
	printf(" blockdev ");

      if(ms->type & XINE_MRL_TYPE_file_normal)
	printf(" normal ");

      if(ms->type & XINE_MRL_TYPE_file_sock)
	printf(" sock ");

      if(ms->type & XINE_MRL_TYPE_file_exec)
	printf(" exec ");

      if(ms->type & XINE_MRL_TYPE_file_backup)
	printf(" backup ");

      if(ms->type & XINE_MRL_TYPE_file_hidden)
	printf(" hidden ");

      printf("] (%Ld byte%c)\n", ms->size, (ms->size > 0) ?'s':'\0');
  }
}
#endif
/*
 *                                 END OF PRIVATES
 *****************************************************************************/

/*
 * Return window of widget.
 */
xitk_window_t *xitk_mrlbrowser_get_window(xitk_widget_t *w) {
  _mrlbrowser_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp)
    return NULL;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_MRLBROWSER)
    return NULL;

  return wp->xwin;
}


/*
 * Fill window information struct of given widget.
 */
int xitk_mrlbrowser_get_window_info(xitk_widget_t *w, window_info_t *inf) {
  _mrlbrowser_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_MRLBROWSER)
    return 0;

  return xitk_get_window_info (wp->xitk, wp->widget_key, inf);
}

/*
 * Boolean about visible state.
 */
int xitk_mrlbrowser_is_visible(xitk_widget_t *w) {
  _mrlbrowser_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_MRLBROWSER)
    return 0;

  return wp->visible;
}

/*
 * Hide mrlbrowser.
 */
void xitk_mrlbrowser_hide(xitk_widget_t *w) {
  _mrlbrowser_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp)
    return;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_MRLBROWSER)
    return;

  if (wp->visible) {
    wp->visible = 0;
    xitk_window_flags (wp->xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, 0);
  }
}

/*
 * Show mrlbrowser.
 */
void xitk_mrlbrowser_show(xitk_widget_t *w) {
  _mrlbrowser_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp)
    return;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_MRLBROWSER)
    return;

  wp->visible = 1;
  xitk_window_flags (wp->xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
  xitk_window_raise_window (wp->xwin);
}

/*
 * Leaving mrlbrowser.
 */
static void xitk_mrlbrowser_exit (xitk_widget_t *w, void *data, int state) {
  _mrlbrowser_private_t *wp = (_mrlbrowser_private_t *)data;
  xitk_widget_t *ww;

  (void)w;
  (void)state;
  if (wp->kill_callback)
    wp->kill_callback (&wp->w, wp->w.userdata);

  ww = &wp->w;
  xitk_widgets_delete (&ww, 1);
}


/*
 *
 */

void xitk_mrlbrowser_change_skins (xitk_widget_t *w, xitk_skin_config_t *skonfig) {
  _mrlbrowser_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp)
    return;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_MRLBROWSER)
    return;

  if (xitk_window_change_skin (wp->xwin, skonfig, wp->w.skin_element_name) == 2)
    XITK_DIE ("%s(): couldn't find image for background\n", __FUNCTION__);
}

/*
 *
 */
static void _mrlbrowser_select_mrl (_mrlbrowser_private_t *wp, unsigned int mode) {
  _mrlb_item_t *item;

  if (!_ZERO_TO_MAX_MINUS_1 (wp->sel, wp->items.f_num))
    return;
  item = wp->items.items + wp->items.f_indx[wp->sel];
  mode &= wp->have_callbacks;

  if ((item->xmrl.type & (XINE_MRL_TYPE_file | XINE_MRL_TYPE_file_directory))
    == (XINE_MRL_TYPE_file | XINE_MRL_TYPE_file_directory)) {
    char *new_dir = item->xmrl.mrl, temp[2048];
    /* dir */
    if (item->disp[0] == '.') {
      if (item->disp[1] == '/') { /* reload */
        new_dir = item->xmrl.origin;
      } else if ((item->disp[1] == '.') && (item->disp[2] == '/')) { /* parent */
        size_t l = xitk_find_byte (item->xmrl.origin, 0);
        new_dir = item->xmrl.origin;
        if (l > sizeof (temp) - 2)
          l = sizeof (temp) - 2;
        temp[0] = '/';
        memcpy (temp + 1, item->xmrl.origin, l + 1);
        new_dir = temp + 1 + l;
        if ((new_dir > temp + 1) && (new_dir[-1] == '/'))
          new_dir -= 1;
        do new_dir -= 1; while (new_dir[0] != '/');
        if (new_dir <= temp)
          new_dir = temp + 1;
        if ((new_dir == temp + 1) && (new_dir[0] == '/'))
          new_dir += 1;
        new_dir[0] = 0;
        new_dir = temp + 1;
      }
    } else if (mode == 1) {
      /* we can now add a full dir scan. :-) */
      wp->add_callback (&wp->w, wp->w.userdata, &item->xmrl);
      return;
    }
    _mrlbrowser_set_source (wp, NULL, new_dir);
  } else {
    /* file */
    if (mode & 1)
      wp->add_callback (&wp->w, wp->w.userdata, &item->xmrl);
    if (mode & 2)
      wp->play_callback (&wp->w, wp->w.userdata, &item->xmrl);
  }
}

/*
 * Handle selection in mrlbrowser.
 */
static void mrlbrowser_select (xitk_widget_t *w, void *data, int state) {
  _mrlbrowser_private_t *wp= (_mrlbrowser_private_t *)data;

  (void)w;
  (void)state;
  _mrlbrowser_select_mrl (wp, 1);
}

/*
 * Handle selection in mrlbrowser, then
 */
static void mrlbrowser_play(xitk_widget_t *w, void *data) {
  _mrlbrowser_private_t *wp= (_mrlbrowser_private_t *)data;

  (void)w;
  _mrlbrowser_select_mrl (wp, 2);
}

/*
 * Handle click in labelbutton list.
 */
static void handle_click (xitk_widget_t *w, void *data, int selected, int modifier) {
  _mrlbrowser_private_t *wp= (_mrlbrowser_private_t *)data;

  (void)w;
  (void)modifier;
  wp->sel = selected;
  xitk_widgets_state (wp->play_add, 2, XITK_WIDGET_STATE_ENABLE, (selected >= 0) ? ~0u : 0);
}

/*
 * Handle double click in labelbutton list.
 */
static void handle_dbl_click(xitk_widget_t *w, void *data, int selected, int modifier) {
  _mrlbrowser_private_t *wp= (_mrlbrowser_private_t *)data;
  unsigned int mode = ((modifier & MODIFIER_CTRL) ? 1 : 0 /* ctrl+shift, ctrl */)
                    + (((modifier ^ MODIFIER_CTRL) & (MODIFIER_CTRL | MODIFIER_SHIFT)) ? 2 : 0 /* ctrl+shift, shift, none */);

  (void)w;
  wp->sel = selected;
  _mrlbrowser_select_mrl (wp, mode);
  xitk_widgets_state (wp->play_add, 2, XITK_WIDGET_STATE_ENABLE, 0);
}

/*
 * Refresh filtered list
 */
static void combo_filter_select(xitk_widget_t *w, void *data, int select) {
  _mrlbrowser_private_t *wp = (_mrlbrowser_private_t *)data;
  int nsel;

  (void)w;
  wp->filter.selected = select;
  /* Keep item selection across filter switch, if possible. */
  nsel = _mrlbrowser_filter_mrls (wp);
  xitk_browser_update_list (wp->mrlb_list, (const char * const *)wp->items.f_list, NULL, wp->items.f_num, 0);
  wp->sel = xitk_browser_set_select (wp->mrlb_list, nsel);
  xitk_widgets_state (wp->play_add, 2, XITK_WIDGET_STATE_ENABLE, (wp->sel >= 0) ? ~0u : 0);
}

/*
 * Handle here mrlbrowser events.
 */
static int mrlbrowser_event (void *data, const xitk_be_event_t *e) {
  _mrlbrowser_private_t *wp = (_mrlbrowser_private_t *)data;

  switch (e->type) {
    case XITK_EV_DEL_WIN:
      xitk_mrlbrowser_exit (NULL, wp, 0);
      return 1;
    case XITK_EV_SHOW:
      xitk_label_change_label (wp->widget_origin, wp->current_origin);
      xitk_label_change_label (wp->list_title_w, wp->list_title_s);
      return 1;
    case XITK_EV_HIDE:
      /* stop possible time consuming animations. */
      xitk_label_change_label (wp->widget_origin, "");
      xitk_label_change_label (wp->list_title_w, "");
      return 1;
    case XITK_EV_KEY_DOWN:
      switch (e->utf8[0]) {
#ifdef DEBUG_MRLB
        case 'd' & 0x1f:
          /* This is for debugging purpose */
          mrlbrowser_dumpmrl (NULL, (void *)wp);
          return 1;
#endif
        case 's' & 0x1f:
          _mrlbrowser_select_mrl (wp, 1);
          return 1;
        case XITK_CTRL_KEY_PREFIX:
          switch (e->utf8[1]) {
            case XITK_KEY_RETURN:
              _mrlbrowser_select_mrl (wp, 2);
              return 1;
            case XITK_KEY_ESCAPE:
              xitk_mrlbrowser_exit (NULL, (void *)wp, 0);
              return 1;
            default: ;
          }
        default: ;
      }
      break;
    default: ;
  }
  if (wp->input_cb)
    return wp->input_cb (wp->input_cb_data, e);
  return 0;
}

/*
 * Create mrlbrowser window.
 */
xitk_widget_t *xitk_mrlbrowser_create (xitk_t *xitk, const xitk_mrlbrowser_widget_t *mb, xitk_skin_config_t *skonfig) {
  const char            *title;
  _mrlbrowser_private_t *wp;
  xitk_widget_t         *default_source = NULL;
  xitk_image_t          *bg_image;
  XITK_HV_INIT;

  if (!xitk)
    return NULL;

  title = mb->window_title;

  if(mb->ip_availables == NULL) {
    XITK_DIE("Something's going wrong, there is no input plugin "
	     "available having INPUT_CAP_GET_DIR capability !!\nExiting.\n");
  }

  if(mb->xine == NULL) {
    XITK_DIE("Xine engine should be initialized first !!\nExiting.\n");
  }

  wp = (_mrlbrowser_private_t *) xitk_widget_new (NULL, sizeof (*wp));
  if (!wp)
    return NULL;

  wp->xitk = xitk;
  wp->visible = 1;
  wp->xine = mb->xine;
  wp->input_cb = mb->input_cb;
  wp->input_cb_data = mb->input_cb_data;

  if (mb->nw.skin_element_name)
    strlcpy (wp->w.skin_element_name, mb->nw.skin_element_name, sizeof (wp->w.skin_element_name));
  {
    const xitk_skin_element_info_t *info = xitk_skin_get_info (skonfig, wp->w.skin_element_name);
    bg_image = info ? info->pixmap_img.image : NULL;
  }
  if (!bg_image) {
    XITK_WARNING("%s(%d): couldn't find image for background\n", __FILE__, __LINE__);
    XITK_FREE (wp);
    return NULL;
  }

  if (xitk_init_NULL ()) {
    wp->items.items = NULL;
    wp->items.f_list = NULL;
    wp->items.f_indx = NULL;
    wp->last_mrl_source = NULL;
    wp->last_clicked = NULL;
  }
#if 0
  wp->items.i_num = 0;
  wp->items.f_num = 0;
#endif

  wp->sel = -1;

  /* needed because we said xitk_widget_new (NULL, ...) above. */
  wp->w.userdata = mb->nw.userdata;

  wp->filter.names = (const char **)mb->filter.names;
  wp->filter.match = mb->filter.match;

  wp->xwin = xitk_window_create_window_ext (xitk, mb->x, mb->y, bg_image->width, bg_image->height,
    title, "xitk mrl browser", "xitk", 0, 0, mb->icon, bg_image);

  wp->widget_list = xitk_window_widget_list (wp->xwin);

  {
    xitk_widget_t *w;
    xitk_image_widget_t im = {
      .nw = {
        .wl = wp->widget_list,
        .skin_element_name = mb->winfocus.skin_element_name,
        .tips = "",
        .userdata = wp,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER
      }
    };

    w = xitk_image_create (&im, skonfig);
    xitk_widget_set_window_focusable (w);
  }

  {
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = wp->widget_list,
        .skin_element_name = mb->select.skin_element_name,
        .add_state = XITK_WIDGET_STATE_VISIBLE,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER,
        .tips = _("Select current entry"),
        .userdata = wp
      },
      .button_type = CLICK_BUTTON,
      .align       = ALIGN_DEFAULT,
      .label       = mb->select.caption,
      .callback    = mrlbrowser_select
    };
    wp->play_add[1] = xitk_labelbutton_create (&lb, skonfig);

    lb.nw.add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
    lb.nw.skin_element_name = mb->dismiss.skin_element_name;
    lb.nw.tips = _("Close MRL browser window");
    lb.label    = mb->dismiss.caption;
    lb.callback = xitk_mrlbrowser_exit;
    xitk_labelbutton_create (&lb, skonfig);
  }

  {
    xitk_button_widget_t pb = {
      .nw = {
        .wl = wp->widget_list,
        .add_state = XITK_WIDGET_STATE_VISIBLE,
        .skin_element_name = mb->play.skin_element_name,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER,
        .tips = _("Play selected entry"),
        .userdata = wp
      },
      .callback = mrlbrowser_play
    };

    wp->play_add[0] = xitk_button_create (&pb, skonfig);
  }

  wp->add_callback  = mb->select.callback;
  wp->play_callback = mb->play.callback;
  wp->kill_callback = mb->kill.callback;
  wp->have_callbacks = (wp->add_callback ? 1 : 0)
                     + (wp->play_callback ? 2 : 0)
                     + (wp->kill_callback ? 4 : 0);

  {
    xitk_browser_widget_t br = mb->browser;

    br.nw.wl                = wp->widget_list;
    br.nw.group             = NULL;
    br.nw.add_state         = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE;
    br.nw.mode_mask         = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER;
    br.nw.mode_value        = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER;
    br.nw.userdata          = wp;
    br.browser.num_entries  = wp->items.f_num;
    br.browser.entries      = (const char * const *)wp->items.f_list;
    br.browser.shortcuts    = NULL;
    br.callback             = handle_click;
    br.dbl_click_callback   = handle_dbl_click;
    wp->mrlb_list = xitk_browser_create (&br, skonfig);
  }

  {
    xitk_label_widget_t lbl = {
      .nw = {
        .wl = wp->widget_list,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER,
        .skin_element_name = mb->origin.skin_element_name
      },
      .label = ""
    };
    wp->widget_origin = xitk_label_create (&lbl, skonfig);

    if (mb->ip_name.label.label_str) {
      lbl.nw.skin_element_name = mb->ip_name.label.skin_element_name;
      strlcpy (wp->list_title_s, mb->ip_name.label.label_str, sizeof (wp->list_title_s));
      lbl.label = mb->ip_name.label.label_str;
      wp->list_title_w = xitk_label_create (&lbl, skonfig);
    }
  }

  strlcpy (wp->current_origin, mb->origin.cur_origin ? mb->origin.cur_origin : "", sizeof (wp->current_origin));

  /*
   * Create buttons with input plugins names.
   */
  do {
    const char *tips[64];
    const char * const *autodir_plugins = mb->ip_availables;
    unsigned int i;
    xitk_button_list_widget_t nbl = {
      .nw = {
        .wl = wp->widget_list,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER,
        .skin_element_name = mb->ip_name.button.skin_element_name,
        .userdata = wp,
        .tips = _("More sources...")
      },
      .callback = mrlbrowser_grab_mrls,
      .names = mb->ip_availables,
      .tips = tips,
      .widget_type_flags = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER,
      .select_limit = 1
    };

    if (!autodir_plugins)
      break;

    for (i = 0; autodir_plugins[i]; i++) {
      if (i >= sizeof (tips) / sizeof (tips[0]))
        break;
      tips[i] = xine_get_input_plugin_description (wp->xine, autodir_plugins[i]);
    }

    wp->autodir_buttons = xitk_button_list_new (&nbl, skonfig);
  } while (0);

  {
    xitk_combo_widget_t cmb = {
      .nw = {
        .wl                = wp->widget_list,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_MRLBROWSER,
        .skin_element_name = mb->combo.skin_element_name,
        .userdata          = wp,
        .tips              = _("Media types to show")
      },
      .layer_above = mb->layer_above,
      .entries     = wp->filter.names,
      .parent_wkey = &wp->widget_key,
      .callback    = combo_filter_select
    };
    wp->combo_filter = xitk_combo_create (&cmb, skonfig);
  }

  XITK_HV_H (wp->w.pos) = mb->x;
  XITK_HV_V (wp->w.pos) = mb->y;
  XITK_HV_H (wp->w.size) = bg_image->width;
  XITK_HV_V (wp->w.size) = bg_image->height;
  wp->w.type         = WIDGET_GROUP | WIDGET_TYPE_MRLBROWSER;
  wp->w.event        = notify_event;

  wp->widget_key = xitk_be_register_event_handler ("mrl browser", wp->xwin, mrlbrowser_event, wp, NULL, NULL);

  if (mb->reparent_window) {
    mb->reparent_window (mb->rw_data, wp->xwin);
    xitk_window_flags (wp->xwin, XITK_WINF_DND, wp->input_cb ? XITK_WINF_DND : 0);
  } else {
    xitk_window_flags (wp->xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED | XITK_WINF_DND,
      XITK_WINF_VISIBLE | (wp->input_cb ? XITK_WINF_DND : 0));
    xitk_window_raise_window (wp->xwin);
  }

  xitk_window_set_input_focus (wp->xwin);

  /* xitk_mrlbrowser_change_skins (&wp->w, skonfig); */

  default_source = xitk_button_list_find (wp->autodir_buttons, "file");
  if (default_source && !wp->last_mrl_source)
    mrlbrowser_grab_mrls (default_source, wp, 1);

  _xitk_new_widget_apply (&mb->nw, &wp->w);
  xitk_widget_set_parent (&wp->w, NULL);

  return &wp->w;
}

